/*
    protocol_subnet.c -- handle the meta-protocol, subnets
    Copyright (C) 1999-2003 Ivo Timmermans <ivo@o2w.nl>,
                  2000-2003 Guus Sliepen <guus@sliepen.eu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: protocol_subnet.c,v 1.1.4.15 2003/07/24 12:08:16 guus Exp $
*/

#include "system.h"

#include "conf.h"
#include "connection.h"
#include "logger.h"
#include "net.h"
#include "netutl.h"
#include "node.h"
#include "protocol.h"
#include "subnet.h"
#include "utils.h"
#include "xalloc.h"

bool send_add_subnet(connection_t *c, const subnet_t *subnet)
{
	bool x;
	char *netstr;

	cp();

	x = send_request(c, "%d %lx %s %s", ADD_SUBNET, random(),
					 subnet->owner->name, netstr = net2str(subnet));

	free(netstr);

	return x;
}

bool add_subnet_h(connection_t *c)
{
	char subnetstr[MAX_STRING_SIZE];
	char name[MAX_STRING_SIZE];
	node_t *owner;
	subnet_t *s;

	cp();

	if(sscanf(c->buffer, "%*d %*x " MAX_STRING " " MAX_STRING, name, subnetstr) != 2) {
		logger(LOG_ERR, _("Got bad %s from %s (%s)"), "ADD_SUBNET", c->name,
			   c->hostname);
		return false;
	}

	/* Check if owner name is a valid */

	if(!check_id(name)) {
		logger(LOG_ERR, _("Got bad %s from %s (%s): %s"), "ADD_SUBNET", c->name,
			   c->hostname, _("invalid name"));
		return false;
	}

	/* Check if subnet string is valid */

	s = str2net(subnetstr);

	if(!s) {
		logger(LOG_ERR, _("Got bad %s from %s (%s): %s"), "ADD_SUBNET", c->name,
			   c->hostname, _("invalid subnet string"));
		return false;
	}

	if(seen_request(c->buffer))
		return true;

	/* Check if the owner of the new subnet is in the connection list */

	owner = lookup_node(name);

	if(!owner) {
		owner = new_node();
		owner->name = xstrdup(name);
		node_add(owner);
	}

	/* Check if we already know this subnet */

	if(lookup_subnet(owner, s)) {
		free_subnet(s);
		return true;
	}

	/* If we don't know this subnet, but we are the owner, retaliate with a DEL_SUBNET */

	if(owner == myself) {
		ifdebug(PROTOCOL) logger(LOG_WARNING, _("Got %s from %s (%s) for ourself"),
				   "ADD_SUBNET", c->name, c->hostname);
		s->owner = myself;
		send_del_subnet(c, s);
		return true;
	}

	/* If everything is correct, add the subnet to the list of the owner */

	subnet_add(owner, s);

	/* Tell the rest */

	forward_request(c);

	return true;
}

bool send_del_subnet(connection_t *c, const subnet_t *s)
{
	bool x;
	char *netstr;

	cp();

	netstr = net2str(s);

	x = send_request(c, "%d %lx %s %s", DEL_SUBNET, random(), s->owner->name, netstr);

	free(netstr);

	return x;
}

bool del_subnet_h(connection_t *c)
{
	char subnetstr[MAX_STRING_SIZE];
	char name[MAX_STRING_SIZE];
	node_t *owner;
	subnet_t *s, *find;

	cp();

	if(sscanf(c->buffer, "%*d %*x " MAX_STRING " " MAX_STRING, name, subnetstr) != 2) {
		logger(LOG_ERR, _("Got bad %s from %s (%s)"), "DEL_SUBNET", c->name,
			   c->hostname);
		return false;
	}

	/* Check if owner name is a valid */

	if(!check_id(name)) {
		logger(LOG_ERR, _("Got bad %s from %s (%s): %s"), "DEL_SUBNET", c->name,
			   c->hostname, _("invalid name"));
		return false;
	}

	/* Check if the owner of the new subnet is in the connection list */

	owner = lookup_node(name);

	if(!owner) {
		ifdebug(PROTOCOL) logger(LOG_WARNING, _("Got %s from %s (%s) for %s which is not in our node tree"),
				   "DEL_SUBNET", c->name, c->hostname, name);
		return true;
	}

	/* Check if subnet string is valid */

	s = str2net(subnetstr);

	if(!s) {
		logger(LOG_ERR, _("Got bad %s from %s (%s): %s"), "DEL_SUBNET", c->name,
			   c->hostname, _("invalid subnet string"));
		return false;
	}

	if(seen_request(c->buffer))
		return true;

	/* If everything is correct, delete the subnet from the list of the owner */

	s->owner = owner;

	find = lookup_subnet(owner, s);

	free_subnet(s);

	if(!find) {
		ifdebug(PROTOCOL) logger(LOG_WARNING, _("Got %s from %s (%s) for %s which does not appear in his subnet tree"),
				   "DEL_SUBNET", c->name, c->hostname, name);
		return true;
	}

	/* If we are the owner of this subnet, retaliate with an ADD_SUBNET */

	if(owner == myself) {
		ifdebug(PROTOCOL) logger(LOG_WARNING, _("Got %s from %s (%s) for ourself"),
				   "DEL_SUBNET", c->name, c->hostname);
		send_add_subnet(c, find);
		return true;
	}

	/* Tell the rest */

	forward_request(c);

	/* Finally, delete it. */

	subnet_del(owner, find);

	return true;
}
