package amf;

#
# NAME
#    amf::new -- Create a new instance of the AMF (ASCII Message Format) class
#
# SYNOPSIS
#    amf::new();
#
# ARGUMENTS
#    (none)
#
# DESCRIPTION
#    amf::new() is the constructor for a new instance of the "amf" class.
#
# RETURN VALUE
#    amf::new() always returns a reference that is associated with the "amf"
#    class, and will be used as the handle for amf data.
#
# EXAMPLES
#    $container = new amf;
#
# ERROR
#    (none)
#
# SEE ALSO
#    amf::add(), amf::find()
#
sub new {
	my $this = [];

	bless $this;
	return $this;
}

#
# NAME
#    amf::add -- Add a message to an instance of the AMF (ASCII Message
#                Format) class.
#
# SYNOPSIS
#    amf::add(message);
#
# ARGUMENTS
#    message       Message to be added to this instance of the "amf" class.
#
# DESCRIPTION
#    amf::add() adds a message to the buffer of an instance of the "amf" class.
#    The format of the message is as follows:
#        _segment_|_fieldname1(3chars)__fieldvalue1_|_fieldname2(3chars)__fieldvalue2_|...||
#
# RETURN VALUE
#    On success, 1 is returned, on failure 0 is returned.
#
# EXAMPLES
#    $container = new amf;
#    $container->add("test|TSTThis is a test||");
#
# ERROR
#    (none)
#
# SEE ALSO
#    amf::new(), amf::find()
#
sub add {
	my $this = shift;
	my $message = shift;

	if ($message =~ /\|\|$/) {
		$this[$current]=$message;
		$current++;
		return 1;
	}

	return 0;
}

#
# NAME
#    amf::find -- Find a particular segment from an instance of the AMF
#                 (ASCII Message Format) class.
#
# SYNOPSIS
#    amf::find(segment);
#
# ARGUMENTS
#    segment       Segment to locate within the instance of the "amf" class.
#
# DESCRIPTION
#    amf::find() locates a particular segment in an "amf" class based on the
#    segments name, and returns an opaque handle which can be used with
#    amf::get_field() to find the value of a specific field in this segment.
#
# RETURN VALUE
#    On success, a scalar is returned.  On error 'undef' is returned.
#
# EXAMPLES
#    $container->add("test|TSTThis is a test||");
#    $ret=$container->find("test");
#
# ERROR
#    (none)
#
# SEE ALSO
#    amf::add(), amf::get_field(), amf::get_segment()
#
sub find {
	my $this = shift;
	my $seg = shift;

	for ($i=0; $i<$current; $i++) {
		if ($this[$i] =~ /^$seg\|/) {
			return $i;
		}
	}

	return undef;
}

#
# NAME
#    amf::get_field -- Get a field's value from a specific segment.
#
# SYNOPSIS
#    amf::get_field(entry, fieldname)
#
# ARGUMENTS
#    entry         Return value from previous amf::find().
#    fieldname     Name of field to search this segment for.
#
# DESCRIPTION
#    amf::get_field() locates a field within a specific segment (found with
#    amf::find()), and returns its value.
#
# RETURN VALUE
#    Upon successfully finding a field, the value is returned, other wise
#    'undef' is returned.  'undef' will be returned if non-sensical
#    arguments are passed, since a field could not be located, even
#    if a search were performed.
#
# EXAMPLES
#    $container = new amf;
#    $container->add("test|TSTThis is a test||");
#    $ret=$container->find("test");
#    if (defined($ret)) {
#       $val=$container->get_field($ret, "TST");
#    }
#
# ERROR
#    (none)
#
# SEE ALSO
#    amf::find(), amf::get_segment()
#
sub get_field {
	my $this = shift;
	my $entry = shift;
	my $fieldname = shift;

	if (!defined($entry)) {
		return undef;
	}

	if (length($fieldname)!=3) {
		return undef;
	}

	@fields=split('\|', $this[$entry]);
	shift @fields;
	foreach $field (@fields) {
		if ($field =~ /^$fieldname/) {
			return substr($field, 3);
		}
	}

	return undef;
}

#
# NAME
#    amf::get_segment -- Retrieve the name of a segment from a amf::find()
#                        result.
#
# SYNOPSIS
#    amf::get_segment(entry);
#
# ARGUMENTS
#    entry         Return value from previous amf::find().
#
# DESCRIPTION
#    amf::get_segment() returns the name of a segment from an amf::find()
#    result.
#
# RETURN VALUE
#    If $entry is valid, the name of the segment previously searched for
#    with amf::find() is returned.  Otherwise, 'undef' is returned.
#
# EXAMPLES
#    $container = new amf;
#    $container->add("test|TSTThis is a test||");
#    $ret=$container->find("test");
#    if (defined($ret)) {
#       $val=$container->get_segment($ret, "TST");
#    }
#
# ERROR
#    (none)
#
# SEE ALSO
#    amf::find(), amf::get_field()
#
sub get_segment {
	my $this = shift;
	my $entry = shift;

	if (!defined($entry)) {
		return undef;
	}

	@fields=split('\|', $this[$entry]);
	return $fields[0];
}

$current=0;

1;
