/*
 * 18 November 2003:
 *    task001
 *    This implements the nessicary data structures, abstraction layers, and
 *    functions to handle ASCII Message Format messages.
 *       -- Roy Keene <rkeene@rkeene.org>
 */

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include "common.h"


/*
 * TYPE: struct AMF_list
 * USE:  struct AMF_list is used for each node in a linked-listed to maintain
 *       an ordered record of all messages addeed to the AMF (ASCII Message 
 *       Format) buffer.
 */
struct AMF_list;
struct AMF_list {
	char *message;
	char *segname;
	struct AMF_list *_next;
};

/*
 * TYPE: amf_msg_t
 * USE:  amf_msg_t is an abstraction of the "internal" (encapsulated) message
 *       used inside the AMF_*() functions.
 */
typedef struct AMF_list * amf_msg_t;

/*
 * TYPE: struct AMF_container
 * USE:  struct AMF_container is used to keep track of the head and tail
 *       nodes of a particular instance of the AMF (ASCII Message Format).
 */
struct AMF_container {
	amf_msg_t head;
	amf_msg_t tail;
};

/*
 * TYPE: amf_container_t
 * USE:  amf_container_t is an abstraction of the "internal" (encapsulated)
 *       message container type, it is used as a handle to refer to a
 *       specific logical container.
 */
typedef struct AMF_container * amf_container_t;

/*
 * NAME
 *    AMF_add()  -- Add a message to an AMF (ASCII Message Format) container.
 *
 * SYNOPSIS
 *    signed int AMF_add(amf_container_t container, const char *msg);
 *
 * ARGUMENTS
 *    amf_container_t container  Container ("handle"), should just be passed
 *                               from AMF_init().
 *    const char *msg            Message to add to buffer.
 *
 * DESCRIPTION
 *    AMF_add() adds a message (argument 2, msg) to an instance of a AMF
 *    container buffer (argument 1, container).  The container should be
 *    initialized with AMF_init() prior to calling this function.  The format
 *    of the message is:
 *       _segment_|_fieldname1(3chars)__fieldvalue1_|_fieldname2(3chars)__fieldvalue2_|...||
 *
 * RETURN VALUE
 *    On success, 0 is returned.  On error, non-zero is returned.
 *
 * EXAMPLES
 *    AMF_init(0, container);
 *    AMF_add(container, "test|TSTThis is a test||");
 *
 * SEE ALSO
 *    AMF_init(), AMF_find()
 *
 */
signed int AMF_add(amf_container_t container, const char *msg) {
	char *msgcopy=NULL, *segment=NULL, *message=NULL;
	amf_msg_t node;

	/* Sanity checks: Valid container? */
	if (container==NULL) return(-1);

	/* Sanity checks: Valid message? */
	msgcopy=(char *) msg+strlen(msg)-2;
	if (strcmp(msgcopy,"||")!=0) return(-1);
	msgcopy=NULL;

	node=malloc(sizeof(*node));
	if (node==NULL) return(-1);

	msgcopy=strdup(msg);
	if (msgcopy==NULL) return(-1);

	segment=msgcopy;
	message=strchr(msgcopy, '|');
	if (message==NULL) {
		free(node);
		free(msgcopy);
		return(-1);
	}

	*message='\0';
	message++;

	node->segname=segment;
	node->message=message;
	node->_next=NULL;

	if (container->head==NULL) container->head=node;
	if (container->tail!=NULL) {
		container->tail->_next=node;
	} else {
		container->tail=node;
	}

	return(0);
}

/*
 * NAME
 *    AMF_init() -- Initialize a new AMF (ASCII Message Format) container.
 *
 * SYNOPSIS
 *    signed int AMF_init(unsigned int size, amf_container_t *container);
 *
 * ARGUMENTS
 *    unsigned int size          Estimated average size of the buffer.
 *    amf_container_t *container Pointer to an "amf_container_t" to
 *                               initialize.
 *
 * DESCRIPTION
 *    AMF_init() initializes a new AMF container.  A new container must be
 *    initialized before it can used for anything else (i.e., AMF_add() and
 *    AMF_find()).  The container may be tuned for an average size (argument
 *    1, size) by specifying a non-zero value (NOTE: the current
 *    implementation does not use this).   The newly created and initialized
 *    container is assigned to the pointed memory (argument 2, container).
 *
 * RETURN VALUE
 *    On success, 0 is returned.  On error, non-zero is returned.
 *
 * EXAMPLES
 *    amf_container_t *container;
 *    AMF_init(0, container);
 *
 * SEE ALSO
 *    AMF_add(), AMF_find()
 *
 */
signed int AMF_init(unsigned int size, amf_container_t *container) {
	if (container==NULL) return(-1);

	*container=malloc(sizeof(**container));
	if (*container==NULL) {
		return(-1);
	}

	(*container)->head=NULL;
	(*container)->tail=NULL;
	return(0);
}

/*
 * NAME
 *    AMF_find() -- Find a specific segment name in a AMF (ASCII Message
 *                  Format) container.
 *
 * SYNOPSIS
 *    signed int AMF_find(amf_container_t container, const char *segment,
 *                        amf_msg_t *msg);
 *
 * ARGUMENTS
 *    amf_container_t container  Container ("handle"), should just be passed
 *                               from AMF_init().
 *    const char *segment        Segment to search this AMF instance for.
 *    amf_msg_t *msg             Results are stored in the memory pointed to
 *                               by this (msg) variable for further
 *                               processing.
 *
 * DESCRIPTION
 *    AMF_find() searches the AMF container specified (argument 1, container)
 *    for segments which exactly match the specified segment (argument 2,
 *    segment).  Results are stored in the memory area pointed to by msg
 *    argument 3, msg).
 *
 * RETURN VALUE
 *    Upon succesfully finding a segment that matches the requested segment
 *    exactly, the number of matches is returned and the nessicary information
 *    is stored in *msg (which can be processed with AMF_get_field(), and
 *    AMF_get_segment()).  If no matches were found 0 is returned.  On error
 *    a negative number is returned.
 *
 * EXAMPLES
 *    amf_msg_t tmp;
 *    AMF_find(container, "test", &amf_tmp);
 *
 * SEE ALSO
 *    AMF_init(), AMF_add(), AMF_get_field(), AMF_get_segment()
 *
 */
signed int AMF_find(amf_container_t container, const char *segment, amf_msg_t *msg) {
	amf_msg_t node;

	/* Sanity checks */
	if (container==NULL || segment==NULL || msg==NULL) return(-1);

	for (node=container->head; node; node=node->_next) {
		if (strcmp(node->segname, segment)==0) {
			*msg=node;
			return(1);
		}
	}

	return(0);
}

/*
 * NAME
 *    AMF_get_field() -- Retrieve the value of a specific field from a
 *                       previous search result.
 *
 * SYNOPSIS
 *    char *AMF_get_field(amf_msg_t message, const char *fieldname);
 *
 * ARGUMENTS
 *    amf_msg_t message          The result of a previous search (from
 *                               AMF_find()).
 *    const char *fieldname      Field name to search for.
 *
 * DESCRIPTION
 *    AMF_get_field() searchs a previously located segment (argument 1,
 *    message) for a specific field (argument 2, fieldname) and returns the
 *    value of that field.
 *
 * RETURN VALUE
 *    On success, a pointer to a C-string containing the value of the
 *    requested field.  On error NULL is returned.  The lack of a matching
 *    field is considered an error, and thus returns NULL.
 *
 * EXAMPLES
 *    amf_msg_t amf_tmp;
 *    char *TST;
 *    AMF_find(container, "test", &amf_tmp);
 *    TST=AMF_get_field(amf_tmp, "TST");
 *    if (TST!=NULL) printf("TST=%s\n", TST);
 *
 * SEE ALSO
 *    AMF_find(), AMF_get_segment()
 *
 */
char *AMF_get_field(amf_msg_t message, const char *fieldname) {
	char *field=NULL, *fieldvalue=NULL;
	char *strtokkeep=NULL;

	/* Sanity check */
	if (message==NULL || fieldname==NULL) return(NULL);
	if (message->message==NULL) return(NULL);
	if (strlen(fieldname)!=3) return(NULL);

	field=strtok_r(message->message, "|", &strtokkeep);
	while (field!=NULL) {
		if (strncmp(field, fieldname, strlen(fieldname))==0) {
			fieldvalue=field+strlen(fieldname);
			return(field+strlen(fieldname));
		}
		field=strtok_r(NULL, "|", &strtokkeep);
	}

	return(NULL);
}

/*
 * NAME
 *    AMF_get_segment() -- Get the name of a previously searched for segment.
 *
 * SYNOPSIS
 *    char *AMF_get_segment(amf_msg_t message);
 *
 * ARGUMENTS
 *    amf_msg_t message          The result of a previous search (from
 *                               AMF_find()).
 *
 * DESCRIPTION
 *    AMF_get_segment() determines the name of the segment that a previous
 *    search results (argument 1, message) is related to.
 *
 * RETURN VALUE
 *    On success, a C-string is returned containing the name of the segment
 *    that this results contains.  On error, NULL is returned.
 *
 * EXAMPLES
 *    amf_msg_t amf_tmp;
 *    char *TST;
 *    AMF_find(container, "test", &amf_tmp);
 *    TST=AMF_get_segment(amf_tmp);
 *    if (TST!=NULL) printf("TST=%s\n", TST);
 *
 * SEE ALSO
 *    AMF_find(), AMF_get_field()
 *
 */
char *AMF_get_segment(amf_msg_t message) {
	if (message==NULL) return(NULL);
	return(message->segname);
}

int main(int argc, char **argv) {
	amf_container_t container;
	amf_msg_t msg;
	char *seg[2]={"fmlog","test"}, *zip=NULL, *segcheck=NULL;
	int ret=0;
	int i=0;

	ret=AMF_init(0, &container);
	if (ret<0) {
		PRINTERR("Could not initialize the container class, exiting.");
		return(1);
	}

	ret=AMF_add(container, "fmlog|ZIP39571|OIDHICAD1||");
	if (ret<0) {
		PRINTERR("Failed to add a message!");
	}

	for (i=0; i<(sizeof(seg)/sizeof(char *)); i++) {
		ret=AMF_find(container, seg[i], &msg);
		if (ret<=0) {
			PRINTERR("Segment not found (\"%s\")", seg[i]);
		} else {
			zip=AMF_get_field(msg, "ZIP");
			segcheck=AMF_get_segment(msg);
			if (zip==NULL) {
				PRINTERR("Couldn't locate ZIP within message")
			} else {
				SPOTVAR_S(zip);
			}
			SPOTVAR_S(segcheck);
		}
	}

	return(0);
}
