#include "compat.h"

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_ASM_PAGE_H
#include <asm/page.h>
#endif

#ifdef HAVE_SYS_SWAP_H
#include <sys/swap.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#endif

#ifdef HAVE_CTYPE_H
#include <ctype.h>
#endif

#ifndef SWAPD_DEF_CONFIG
#define SWAPD_DEF_CONFIG "swapd.conf"
#endif

extern char *optarg;
int64_t minfree = (10 * 1024 *1024), maxfree = 0;
size_t swapsize = (32 * 1024 * 1024);
char *swapdir = "/var/tmp";

typedef struct {
	char *pathname;
	size_t swapsize;
	int active;
} swap_t;

int daemonize(void) {
#ifndef DEBUG
	pid_t pid;

	chdir("/");

	close(STDIN_FILENO);
	close(STDOUT_FILENO);
	close(STDERR_FILENO);

	setsid();

	pid = fork();

	if (pid > 0) {
		exit(EXIT_SUCCESS);
	}

	if (pid < 0) {
//		syslog(LOG_ERR, "Fork failed while daemonizing: %s", strerror(errno));
		exit(EXIT_FAILURE);
	}
#endif
	return(0);
}

int swapd_init_stats(void) {
#ifdef HAVE_LIBSTATGRAB
	/* Initialize the libstatgrab interface. */
	if (sg_init() != 0) {
		/* Failure. */
		return(0);
	}
#endif
	/* No initialization is needed for Linux proc interface. */

	/* Success! */
	return(1);
}

int64_t swapd_get_free_mem(void) {
	int64_t retval = -1;
#ifdef HAVE_LIBSTATGRAB
	sg_mem_stats *meminfo = NULL;
	sg_swap_stats *swapinfo = NULL;

	meminfo = sg_get_mem_stats();
	if (meminfo == NULL) {
		return(-1);
	}

	swapinfo = sg_get_swap_stats();
	if (swapinfo == NULL) {
		return(-1);
	}

	retval = meminfo->free + meminfo->cache + swapinfo->free;

	free(meminfo);
	free(swapinfo);

	return(retval);
#else
	FILE *meminfofd = NULL;
	char buf[128] = {0};
	char *buf_part = NULL, *buf_whole = NULL;
	uint64_t meminfo_part = 0, meminfo = 0;

	meminfofd = fopen("/proc/meminfo", "r");
	while (1) {
		fgets(buf, sizeof(buf) - 1, meminfofd);
		if (feof(meminfofd)) {
			break;
		}

		buf_whole = buf;
		buf_part = strsep(&buf_whole, ":");
		if (buf_part == NULL || buf_whole == NULL) {
			continue;
		}
		meminfo_part = strtoull(buf_whole, NULL, 10) * 1024;
		if (strcmp(buf_part, "MemFree") == 0 ||
		    strcmp(buf_part, "Buffers") == 0 ||
		    strcmp(buf_part, "Cached") == 0 ||
		    strcmp(buf_part, "SwapFree") == 0) {
			meminfo += meminfo_part;
		}
	}

	fclose(meminfofd);

	if (meminfo != 0) {
		retval = meminfo;
	}

	return(retval);
#endif
}

int swapd_call_mkswap(const char *swapfile) {
	char cmdline[8192] = {0};
	int fd;

	snprintf(cmdline, sizeof(cmdline) - 1, "mkswap \"%s\"", swapfile);
	SPOTVAR_S(cmdline);
	system(cmdline);

	fd = open(swapfile, O_RDONLY);
	fsync(fd);
	close(fd);

	return(0);
}

int swapd_swapon(const char *swapfile) {
	int swaponret = -1;

#ifdef SWAPON_TAKES_2_ARGS
	swaponret = swapon(swapfile, 0);
#else
	swaponret = swapon(swapfile);
#endif

	return(swaponret);
}

swap_t *swapd_mkswap(const char *swapdir, size_t swapsize, const char *pathname) {
	swap_t *retval = NULL;
#ifdef PATH_MAX
	char swapfilepath[PATH_MAX + 1] = {0};
#else
	char swapfilepath[1024] = {0};
#endif
#ifdef PAGE_SIZE
	char databuf[(PAGE_SIZE * 2)] = {0};
#else
	char databuf[4096] = {0};
#endif
	int64_t numbufs = 0, bufcnt = 0;
	size_t realswapsize = 0;
	int swapfd = -1;
	int swaponret = -1;

	if (pathname == NULL) {
		/* Create a temporary file. */
		snprintf(swapfilepath, sizeof(swapfilepath) - 1, "%s/.swapXXXXXX", swapdir);

		swapfd = mkstemp(swapfilepath);
		if (swapfd < 0) {
			return(NULL);
		}

		/* Fill the file with empty data. */
		numbufs = (swapsize / sizeof(databuf)) + 1;	
		realswapsize = numbufs * sizeof(databuf);

		for (bufcnt = 0; bufcnt < numbufs; bufcnt++) {
			write(swapfd, databuf, sizeof(databuf));
		}
		close(swapfd);

		/* Initialize the swap signature. */
		swapd_call_mkswap(swapfilepath);
	} else {
		strncpy(swapfilepath, pathname, sizeof(swapfilepath));
	}

	/* Start swaping. */
	swaponret = swapd_swapon(swapfilepath);
	if (swaponret < 0) {
		unlink(swapfilepath);
		return(NULL);
	}

	retval = malloc(sizeof(*retval));
	if (retval == NULL) {
		return(NULL);
	}

	retval->pathname = strdup(swapfilepath);
	retval->swapsize = realswapsize;
	retval->active = 1;

	if (retval->pathname == NULL) {
		free(retval);
		return(NULL);
	}

	return(retval);
}

int swapd_swapoff(swap_t *swapfile) {
#ifdef HAVE_SWAPOFF
	int swapoffret = -1;

	if (swapfile == NULL) {
		return(-1);
	}

	swapoffret = swapoff(swapfile->pathname);

	if (swapoffret < 0) {
		swapfile->active = 0;
		return(0);
	}
	return(-1);
#else
	return(0);
#endif
}

int swapd_cleanup(const char *swapdir) {
	DIR *dh = NULL;
	struct dirent *dent = NULL;

	chdir(swapdir);
	dh = opendir(".");

	if (dh == NULL) {
		return(-1);
	}

	while (1) {
		dent = readdir(dh);
		if (dent == NULL) {
			break;
		}
		if (strlen(dent->d_name) == 11 && strncmp(dent->d_name, ".swap", 5) == 0) {
			SPOTVAR_S(dent->d_name);
		}
	}

	closedir(dh);

	return(0);
}

int swapd_process_config_line(const char *cmd) {
	char *tmpbuf = NULL, *tmpbuf_s = NULL, *tmppart = NULL, *argbuf[10] = {NULL}, *endbuf = NULL;
	int64_t argbuf_int[10] = {0};
	int argc = 0, i = 0;

	if (cmd == NULL) {
		return(-1);
	}

	tmpbuf_s = tmpbuf = strdup(cmd);

	if (tmpbuf_s == NULL) {
		return(-1);
	}

	for (argc = 0; argc < sizeof(argbuf) / sizeof(argbuf[0]); argc++) {
		tmppart = strsep(&tmpbuf, ": \t");
		if (tmppart == NULL) {
			break;
		}

		if (strcmp(tmppart, "") == 0) {
			argc--;
			continue;
		}

		argbuf[argc] = tmppart;
	}

	if (argc == 0 || argbuf[0] == NULL) {
		return(0);
	}

	if (argbuf[0][0] == '#') {
		return(0);
	}

	for (i = 0; i < argc; i++) {
		argbuf_int[i] = strtoull(argbuf[i], &endbuf, 10);
		if (endbuf != NULL) {
			switch (tolower(endbuf[0])) {
				case 'g':
					argbuf_int[i] *= (1024 * 1024 * 1024);
					break;
				case 'm':
					argbuf_int[i] *= (1024 * 1024);
					break;
				case 'k':
					argbuf_int[i] *= 1024;
					break;
			}
		}
	}

	if (strcasecmp(argbuf[0], "maxfree") == 0 && argc == 2) {
		maxfree = argbuf_int[1];
	} else if (strcasecmp(argbuf[0], "minfree") == 0 && argc == 2) {
		minfree = argbuf_int[1];
	} else if (strcasecmp(argbuf[0], "swapsize") == 0 && argc == 2) {
		swapsize = argbuf_int[1];
	} else if (strcasecmp(argbuf[0], "swapdir") == 0 && argc == 2) {
		swapdir = strdup(argbuf[1]);
	} else {
		PRINTERR("Unknown command: %s/%i", argbuf[0], argc);
	}

	free(tmpbuf_s);

	return(0);
}

int swapd_process_config_file(const char *pathname) {
	FILE *configfp = NULL;
	char buf[1024] = {0};

	if (pathname == NULL) {
		return(-1);
	}

	configfp = fopen(pathname, "r");
	if (configfp == NULL) {
		return(-1);
	}

	while (1) {
		fgets(buf, sizeof(buf) - 1, configfp);
		if (feof(configfp)) {
			break;
		}

		while (strlen(buf) > 0) {
			if (buf[strlen(buf) - 1] < ' ') {
				buf[strlen(buf) - 1] = '\0';
			} else {
				break;
			}
		}

		swapd_process_config_line(buf);
	}

	fclose(configfp);

	return(0);
}

int main(int argc, char **argv) {
	swap_t *swaps[128], *swapinfo = NULL;
	int64_t freemem = -1;
	char *swapfile = NULL;
	int ch = -1, i = 0;
	int active_swaps = 0;
	int swapoffret = -1;
	int retval = EXIT_SUCCESS;

	if (!swapd_init_stats()) {
		return(EXIT_FAILURE);
	}

	for (i = 0; i < (sizeof(swaps) / sizeof(swaps[0])); i++) {
		swaps[i] = NULL;
	}

	/* Process config file.. */
	swapd_process_config_file(SWAPD_DEF_CONFIG);

	/* Process arguments.. */
	while ((ch = getopt(argc, argv, "Vhm:s:d:M:o:c:")) != -1) {
		switch (ch) {
			case 'V':
				printf("%s version %s\n", PACKAGE_NAME, PACKAGE_VERSION);
				return(EXIT_SUCCESS);
			case 'M':
				maxfree = strtoull(optarg, NULL, 10);
				break;
			case 'm':
				minfree = strtoull(optarg, NULL, 10);
				break;
			case 's':
				swapsize = strtoull(optarg, NULL, 10);
				break;
			case 'd':
				swapdir = strdup(optarg);
				break;
			case 'o':
				swapd_process_config_line(optarg);
				break;
			case 'c':
				swapd_process_config_file(optarg);
				break;
			case ':':
			case '?':
				retval = EXIT_FAILURE;
			case 'h':
				printf("Usage: swapd [-Vh] [-c config] [-o cmd] [-m min] [-M max] [-s swapsize] [-d dir]\n");
				printf("  -V     Print version and exit.\n");
				printf("  -h     This help.\n");
				printf("  -c     Process an additional configuration file.\n");
				printf("  -o     Process a command as if it were in a configuration file.\n");
				printf("  -m     Specify the lower boundary and where to start making new swap files.\n");
				printf("  -M     Specify the upper boundary and where to start disabling swap files.\n");
				printf("  -s     Specify the size of each swap file.\n");
				printf("  -d     Specify the directory where swapfiles will be created.\n");
				return(retval);
				break;
		}
	}

	/* Fix illegal values. */
	if (maxfree < (minfree + swapsize + 1) && maxfree != 0) {
		PRINTERR("Illegal max value.");
		maxfree = 0;
	}

	daemonize();

	swapd_cleanup(swapdir);

	SPOTVAR_LL(maxfree);
	SPOTVAR_LL(minfree);

	while (1) {
		sleep(5);

		freemem = swapd_get_free_mem();
		if (freemem == -1) {
			continue;
		}

		SPOTVAR_LL(freemem);

		if (freemem < minfree) {
			swapfile = NULL;
			for (i = 0; i < (sizeof(swaps) / sizeof(swaps[0])); i++) {
				if (swaps[i] != NULL) {
					if (swaps[i]->active == 0 && swaps[i]->pathname != NULL) {
						swapfile = strdup(swaps[i]->pathname);
						free(swaps[i]->pathname);
						free(swaps[i]);
						swaps[i] = NULL;
						break;
					}
				}
			}

			swapinfo = swapd_mkswap(swapdir, swapsize, swapfile);

			if (swapfile != NULL) {
				free(swapfile);
			}

			if (swapinfo != NULL) {
				for (i = 0; i < (sizeof(swaps) / sizeof(swaps[0])); i++) {
					if (swaps[i] == NULL) {
						swaps[i] = swapinfo;
						break;
					}
				}
				active_swaps++;
			}

			continue;
		}

		if (freemem > maxfree && maxfree > 0) {
			SPOTVAR_I(active_swaps);
			if (active_swaps > 0) {
				for (i = 0; i < (sizeof(swaps) / sizeof(swaps[0])); i++) {
					if (swaps[i] != NULL) {
						if (swaps[i]->active == 0) continue;
						swapoffret = swapd_swapoff(swaps[i]);
						if (swapoffret >= 0) {
							active_swaps--;
							break;
						}
					}
				}
			}

			continue;
		}

	}

	return(EXIT_FAILURE);
}
