/*
    tcpcgi.c -- Client portion of rutil_tcpcgi.
    Copyright (C) 2003  Roy Keene

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	email: tcpcgi@rkeene.org

*/

/*
 * tcpcgi.c -- Local listener part of reverse-utils::TCP-over-HTTP/CGI:
 *    This needs to do quite a bit of work in the background.
 *
 *    It will create a listening socket, upon connection to that socket
 *    it will send the proper HTTP request to create the socket on the
 *    other end, and wait for an ackowledgement.
 *
 *    Every second it will query the remote machine for all the sessions
 *    that exist and pass the data to the client socket.
 *
 *    When data arrives, it needs to hex-encode it and put it on the socket.
 *               -- Roy Keene [300820031505] <tcpcgi@rkeene.org>    
 */

#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/poll.h>
#include <sys/wait.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <fcntl.h>
#include <time.h>

#include "tcpcgi.h"
#include "tcpnet.h"

extern char *optarg;
extern int optind, opterr, optopt;


struct sockinfo {
	char *session;
	char *http_host;
	char *http_path;
	int http_port;
	signed int fd;
};

int alarm_freq=0;
int use_post=1;
int local_only=0;
int poll_time_min=TCPCGI_CLIENT_POLLTIME_MIN;
int poll_time_max=TCPCGI_CLIENT_POLLTIME_MAX;
struct sockinfo socks_info[128]; /* Needs to be num of elements as socks_poll in main() */


char *hexcode(void *data, int datalen) {
	char hexabet[]="0123456789abcdef";
	unsigned char *data_num=data;
	char *ret;
	int i, destpos=0;

	if (data==NULL || datalen==0) return(strdup(""));

	ret=malloc(datalen*3+1);

	for (i=0; i<datalen; i++) {
		if (data_num[i]==' ') {
			ret[destpos++]='+';
			continue;
		}
#if 1
		/* if it's a nonprintable charectar, or [%?@&=+: ], convert it */
		if (!isgraph(data_num[i]) || data_num[i]==' ' || \
		data_num[i]=='&' || data_num[i]=='=' || \
		data_num[i]==':' || data_num[i]=='@' || \
		data_num[i]=='?' || data_num[i]=='%' || \
		data_num[i]=='+') {
#else
		/* If the charectar is not alphanumeric or and not [+./-], convert it to hex. */
		/* This is no longer used in favor of a more leanient algorithm. */
		if (!isalnum(data_num[i]) && data_num[i]!='/' && data_num[i]!='.' && data_num[i]!='~' && data_num[i]!='+') {
#endif
			ret[destpos++]='%';
			ret[destpos++]=hexabet[(data_num[i]>>4)&0xf];
			ret[destpos++]=hexabet[data_num[i]&0xf];
			continue;
		}
		/* Otherwise, we just copy it over. */
		ret[destpos++]=data_num[i];
	}
	ret[destpos]='\0';

#ifdef PARANOID
	ret=realloc(ret, strlen(ret)+1);
#endif
	return(ret);
}

void *tcpcgi_getdata(char *http_host, int http_port, char *http_path, char *cmd, char *session, char *dest, void *data, int *datalen, signed int *status) {
	FILE *fp;
	char *hex_cmd=NULL, *hex_session=NULL, *hex_data=NULL, *hex_path=NULL, *hex_dest=NULL;
	char buf[8192], *retbuf=NULL;
	int fd;
	int sink=0;
	unsigned int datalen_loc;

	if (http_host==NULL || http_path==NULL || cmd==NULL || session==NULL) {
		PRINTERR("Invalid parameters passed.");
		return(NULL);
	}

	/* Need to free the hex_* variables past this point. */
	hex_cmd=hexcode(cmd, strlen(cmd)); /* Guarenteed not NULL */
	hex_session=hexcode(session, strlen(session)); /* Guarenteed not NULL */
	hex_path=hexcode(http_path, strlen(http_path)); /* Guarenteed not NULL */
	hex_data=hexcode(data, (datalen)?(*datalen):0);
	hex_dest=hexcode(dest, strlen(dest?dest:""));

	fd=createconnection_tcp(http_host, http_port);
	fp=fdopen(fd, "r+");
	snprintf(buf, sizeof(buf), "cmd=%s&session=%s&dest=%s&data=%s&ts=%i", hex_cmd, hex_session, hex_dest, hex_data, (unsigned int) time(NULL));
	buf[sizeof(buf)-1]='\0';
	if (use_post) {
		fprintf(fp, "POST %s HTTP/1.0\r\nHost: %s\r\nContent-length: %i\r\n\r\n%s", hex_path, http_host, strlen(buf), buf);
		PRINTERR("POST http://%s:%i%s?[%i]%s HTTP/1.0\n", http_host, http_port, hex_path, strlen(buf), buf);
	} else {
		fprintf(fp, "GET http://%s:%i%s?%s HTTP/1.0\r\n\r\n", http_host, http_port, hex_path, buf);
		PRINTERR("GET http://%s:%i%s?[%i]%s HTTP/1.0\n", http_host, http_port, hex_path, strlen(buf), buf);
	}
	fflush(fp);

	if (status) *status=0;
	if (datalen) *datalen=0;
	if (fgets(buf, sizeof(buf), fp)==NULL) return(NULL);
	if (strstr(buf," 40")) {
		PRINTERR("File not available error, %s", buf);
		if (hex_cmd) free(hex_cmd);
		if (hex_session) free(hex_session);
		if (hex_data) free(hex_data);
		if (hex_dest) free(hex_dest);
		if (hex_path) free(hex_path);
		fclose(fp);
		return(NULL);
	}
	if (!strstr(buf, " 200 ")) {
		PRINTERR("Invalid response, %s", buf);
		if (hex_cmd) free(hex_cmd);
		if (hex_session) free(hex_session);
		if (hex_data) free(hex_data);
		if (hex_dest) free(hex_dest);
		if (hex_path) free(hex_path);
		sleep(1); /* Primitive backoff */
		fclose(fp);
		return(tcpcgi_getdata(http_host, http_port, http_path, cmd, session, dest, data, datalen, status));
	}
	while (1) {
		if (fgets(buf, sizeof(buf), fp)==NULL) break;
		while (buf[strlen(buf)-1]<' ' && buf[0]!='\0') buf[strlen(buf)-1]='\0';
		/* On the first blank line, assume the HTTP headers have ended. */
		if (buf[0]=='\0' && !sink) {
			sink=1; /* Only come here once. */
			if (fgets(buf, sizeof(buf), fp)==NULL) break;
			PRINTERR("statusmsg=%s", buf);
			if (strstr(buf, "stat: ERROR:")!=NULL) {
				PRINTERR("Setting status to -ERR");
				if (status) *status=-1;
			}
			datalen_loc=(fgetc(fp))<<8;
			datalen_loc|=fgetc(fp);
			if (datalen_loc>16384) continue;
			if (datalen_loc) {
				PRINTERR("Now expecting %i bytes of data.", datalen_loc);
				retbuf=malloc(datalen_loc);
				datalen_loc=fread(retbuf, 1, datalen_loc, fp);
				PRINTERR("[%i]%s", datalen_loc, retbuf);
			}
			if (datalen) *datalen=datalen_loc;
			PRINTERR("We got all that we want, let's get outta here.");
			break;
		}
		PRINTERR("buf=%s", buf);
	}

	fclose(fp);
	if (hex_cmd) free(hex_cmd);
	if (hex_session) free(hex_session);
	if (hex_data) free(hex_data);
	if (hex_dest) free(hex_dest);
	if (hex_path) free(hex_path);
	return(retbuf);
}

char *tcpcgi_gensess(void) {
	static char ret[12]={0,0,0,0,0,0,0,0,0,0,0,0};
	signed int i=0;
	char alphabet[]="0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
	int j;

	srand(time(NULL)+ret[0]+ret[1]);
	for (i=0; i<sizeof(ret); i++) {
		j=rand()%(sizeof(alphabet)-1);
		if (alphabet[j]=='\0' && i==0) { i--; continue; } /* Don't generate an empty session. */
		ret[i]=alphabet[j];
	}
	return(ret);
}

void tcpcgi_closesession(int fd, char *http_host, int http_port, char *http_path, char *session) {
	if (!session || fd<0) return;
	PRINTERR("Terminating session \"%s\".", session);

	tcpcgi_getdata(http_host, http_port, http_path, "close", session, NULL, NULL, NULL, NULL);
	return;
}

/*
 * Return values here:
 *    -1  (error)
 *    0   (no error, but no data was recvd)
 *    1   (no error, data recv)
 */
int tcpcgi_handledata(char *http_host, int http_port, char *http_path, char *session, int fd, void *buf, int datalen) {
	unsigned char *ret;
	unsigned char *lbuf=buf;
	char *cmd;
	int status=0;
	int dataretlen;
	int rv;

	while (datalen>1024) {
		if ((rv=tcpcgi_handledata(http_host, http_port, http_path, session, fd, lbuf, 1024))<0) {
			return(rv);
		}
		datalen-=1024;
		lbuf+=1024;
	}

	if (lbuf && datalen!=0) {
		cmd="put";
		dataretlen=datalen;
	} else {
		cmd="get";
		dataretlen=0;
	}

	ret=tcpcgi_getdata(http_host, http_port, http_path, cmd, session, NULL, lbuf, &dataretlen, &status);
	PRINTERR("[%i]%s", dataretlen, ret);
	if (dataretlen>0 && ret && status>=0) {
		write(fd, ret, dataretlen);
		free(ret);
		status=1;
	}

	PRINTERR("Returning %i", status);
	return(status);
}



int handle_background(void) {
	int i;
	char *http_host, *http_path, *session;
	int fd, http_port;
	int ret=0;
	int rv;

	for (i=0; i<(sizeof(socks_info)/sizeof(struct sockinfo)); i++) {
		if (socks_info[i].fd<0) continue;
		http_host=socks_info[i].http_host;
		http_port=socks_info[i].http_port;
		http_path=socks_info[i].http_path;
		session=socks_info[i].session;
		fd=socks_info[i].fd;
		if ((rv=tcpcgi_handledata(http_host, http_port, http_path, session, fd, NULL, 0))<0) {
			PRINTERR("[fd=%i] We got an error in the background and we don't know how to handle it!", fd);
			close(fd);
			continue;
		}
		if (rv>0) ret++;
	}
	return(ret);
}

void sighandler(int sig) {
	if (sig==SIGALRM) {
		if (alarm_freq) alarm(alarm_freq);
		handle_background();
	}
}

int print_help(char *msg) {
	fprintf(stderr, "Usage: tcpcgi -H host -f file -d dest [-P port] [-p port] [-m time]\n");
	fprintf(stderr, "              [-x time] [-t time] [-b amt] [-riglvh]\n");
	fprintf(stderr, "    Options:\n");
	fprintf(stderr, "         -H host      HTTP Server which has tcpcgi.cgi.\n");
	fprintf(stderr, "         -f file      Full URL-relative on HTTP server to tcpcgi.cgi,\n");
	fprintf(stderr, "                      including tcpcgi.cgi.\n");
	fprintf(stderr, "         -d dest      Destination to for HTTP server to connect to and\n");
	fprintf(stderr, "                      proxy to (must be in the form of host:port).\n");
	fprintf(stderr, "         -P port      Port to connect to HTTP server on.\n");
	fprintf(stderr, "         -p port      Port to listen for connections on locally (%i is\n", TCPCGI_CLIENT_PORT);
	fprintf(stderr, "                      the default).\n");
	fprintf(stderr, "         -m time      Lowest amount of time (in milliseconds) to check for\n");
	fprintf(stderr, "                      data from HTTP server (%i is the default).\n", TCPCGI_CLIENT_POLLTIME_MIN);
	fprintf(stderr, "         -x time      Greatest amount of time (in milliseconds) to check\n");
	fprintf(stderr, "                      for data from HTTP server (%i is the default).\n", TCPCGI_CLIENT_POLLTIME_MAX);
	fprintf(stderr, "         -t time      Specify that the amount of time for polling be fixed\n");
	fprintf(stderr, "                      at this value (in ms).\n");
	fprintf(stderr, "         -b amt       Backoff amount, if less than 5, specifies backoff\n");
	fprintf(stderr, "                      should multiply the backoff time by this number,\n");
	fprintf(stderr, "                      otherwise it's incremented by this amount (2 is the\n"); 
	fprintf(stderr, "                      default).\n");
	fprintf(stderr, "         -r           Specify that the backoff algorithm reset when remote\n");
	fprintf(stderr, "                      activity occurs, instead of decrementing.\n");
	fprintf(stderr, "         -i           Specify that the backoff algorithm should respond to\n");
	fprintf(stderr, "                      local traffic as well as remote.\n");
	fprintf(stderr, "         -g           Toggle between HTTP get and HTTP post (%s is\n", use_post?"POST":"GET");
	fprintf(stderr, "                      the default).\n");
	fprintf(stderr, "         -l           Bind to %s only for listening (%s is the\n", local_only?"ALL":"LOCALHOST", local_only?"LOCAL":"ALL");
	fprintf(stderr, "                      default).\n");
	fprintf(stderr, "         -v           Print version (%s) and exit.\n", TCPCGI_VERSION);
	fprintf(stderr, "         -h           This help information.\n");
	if (msg) {
		fprintf(stderr, "\ntcpcgi: %s\n", msg);
	}
	return(1);
}

#ifdef CLOSE_POLL
#undef CLOSE_POLL
#endif
#define CLOSE_POLL(idx) { \
	tcpcgi_closesession(socks_poll[idx].fd, http_host, http_port, http_path, socks_info[idx].session); \
	close(socks_poll[idx].fd); \
	socks_poll[idx].fd=-1; \
	socks_info[idx].fd=-1; \
	pollcnt--; \
	pollpos=idx; \
}
int main(int argc, char **argv) {
	struct pollfd socks_poll[sizeof(socks_info)/sizeof(struct sockinfo)];
	struct sockaddr addr;
	signed int pollpos=0;
	signed int status=0;
	signed int ch;
	char buf[8192];
	char *http_host=NULL, *http_path=NULL;
	char *hostport=NULL;
	socklen_t addrlen;
	int masterfd, fd, clientfd;
	int pollcnt=0, pollmax;
	int num_events;
	int errcnt=0;
	int i,x, rv;
	int listen_port=TCPCGI_CLIENT_PORT;
	int poll_time=200; /* Start out with a reasonably fast poll_time. */
	int poll_backoff=2;
	int poll_backoff_reset=0, poll_backoff_ireset=0;
	int http_port=80;
#ifndef DEBUG
	pid_t chpid;
#endif

	while ((ch=getopt(argc, argv, "H:f:P:p:d:m:x:t:b:vgrilh"))!=-1) {
		switch (ch) {
			case 'H':
				http_host=strdup(optarg);
				break;
			case 'f':
				if (optarg[0]!='/') return(print_help("File should begin with a `/\'"));
				http_path=strdup(optarg);
				break;
			case 'P':
				http_port=atoi(optarg);
				if (http_port<=0) return(print_help("Port cannot be <= 0."));
				break;
			case 'p':
				listen_port=atoi(optarg);
				if (listen_port<=0) return(print_help("Port cannot be <= 0."));
				break;
			case 'd':
				hostport=strdup(optarg);
				if (strchr(hostport,':')==NULL) return(print_help("Destination must be in the form of host:port"));
				break;
			case 'm':
				poll_time_min=atoi(optarg);
				if (poll_time_min<=0) return(print_help("Polltime min cannot be <= 0."));
				if (poll_time<poll_time_min) poll_time=poll_time_min;
				break;
			case 'x':
				poll_time_max=atoi(optarg);
				if (poll_time_max<=0) return(print_help("Polltime max cannot be <= 0."));
				if (poll_time>poll_time_max) poll_time=poll_time_max;
				break;
			case 't':
				poll_time=poll_time_max=poll_time_min=atoi(optarg);
				if (poll_time_max<=0) return(print_help("Polltime cannot be <= 0."));
				break;
			case 'b':
				poll_backoff=atoi(optarg);
				if (poll_backoff<=0) return(print_help("Poll backoff cannot be <= 0."));
				break;
			case 'r':
				poll_backoff_reset=!poll_backoff_reset;
				break;
			case 'i':
				poll_backoff_ireset=!poll_backoff_ireset;
				break;
			case 'g':
				use_post=!use_post;
				break;
			case 'v':
				printf("Reverse utilities::TCP-over-HTTP/CGI v%s\n", TCPCGI_VERSION);
				return(0);
				break;
			case 'l':
				local_only=!local_only;
				break;
			case ':':
			case '?':
			case 'h':
				return(print_help(NULL));
				break;
		}
	}

	/* Basic sanity checks for supplied args */
	if (http_host==NULL || http_path==NULL) {
		return(print_help("You must specify the HOST (-H) and FILE (-f) options."));
	}
	if (http_path[0]!='/') {
		return(print_help("File should begin with a `/\'"));
	}
	if (hostport==NULL) {
		return(print_help("You must specify the DEST (-d) option."));
	}
	if (poll_time_max<poll_time_min) {
		return(print_help("Polltime max cannot be less than polltime min."));
	}


	signal(SIGALRM, sighandler);
	signal(SIGPIPE, SIG_IGN);
	masterfd=createlisten(listen_port, local_only);
	if (masterfd<0) {
		PERROR("createlisten()");
		return(1);
	}

#ifndef DEBUG
	/* We're all out of complaints at this point, and should be ready to go in the background. */
	if ((chpid=fork())<0) { PERROR("fork"); return(1); }
	if (chpid!=0) {
		/* Parent */
		wait(NULL);
		return(0);
	}
	/* Child */
	if ((chpid=fork())<0) { return(1); }
	if (chpid!=0) return(0);
	/* Grand-child */
	setsid();
#endif

	for (i=0; i<(sizeof(socks_poll)/sizeof(struct pollfd)); i++) socks_poll[i].fd=-1;
	for (i=0; i<(sizeof(socks_info)/sizeof(struct sockinfo)); i++) {
		socks_info[i].session=NULL;
		socks_info[i].fd=-1;
	}
	socks_poll[pollpos].fd=masterfd;
	socks_poll[pollpos].events=POLLIN;
	socks_poll[pollpos].revents=0;
	pollpos++;
	pollcnt++;

	if (alarm_freq) alarm(1); /* Start scheduling background events. */

	while (1) {
		for (i=0; i<(sizeof(socks_poll)/sizeof(struct pollfd)); i++) {
			if (socks_poll[i].fd!=-1) pollmax=(i+1);
		}

		num_events=poll(socks_poll, pollmax, (pollcnt>1)?(poll_time):(-1));
		if (num_events<0) {
			if (errcnt==TCPCGI_DAEMON_MAXERR) {
				fprintf(stderr, "Maximum error count (%i) reached, aborting...\n", TCPCGI_DAEMON_MAXERR);
				break;
			}
			errcnt++;
			PERROR("poll()");
			continue;
		}
		errcnt=0;
		if (num_events==0) {
			/* If we just have the listening socket in our poll group, don't adjust polltime */
			if (pollcnt==1) continue;

			/* Check background activity, and adjust polltime if needed. */
			if (handle_background()==0) {
				/* If nothing is happening in the background, slow down. */
				if (poll_backoff<5) { poll_time*=poll_backoff; } else { poll_time+=poll_backoff; }
				PRINTERR("Lengthening polltime [poll_time=%i], nothing is happening.", poll_time);
			} else {
				/* If we're active in the background, speed up! */
				if (poll_backoff<5) { poll_time/=poll_backoff; } else { poll_time-=poll_backoff; }
				if (poll_backoff_reset) poll_time=poll_time_min;
				PRINTERR("Shortening polltime [poll_time=%i], stuff is happening.", poll_time);
			}
			if (poll_time>poll_time_max) poll_time=poll_time_max;
			if (poll_time<poll_time_min) poll_time=poll_time_min;
			continue;
		}
		for (i=0; i<pollmax; i++) {
			if (socks_poll[i].revents&(POLLERR|POLLHUP|POLLNVAL)) {
				/* Socket is broke, close it. */
				CLOSE_POLL(i);
				PRINTERR("Closing socket [idx=%i, fd=%i], due to errors.", i, fd);
				num_events--;
				if (num_events==0) break;
				continue;
			}
			if ((socks_poll[i].revents&POLLIN)!=POLLIN) continue; 
			num_events--;
			fd=socks_poll[i].fd;
			socks_poll[i].events=POLLIN;
			socks_poll[i].revents=0;
			if (fd==masterfd) {
				/* Handle incoming connection. */
				PRINTERR("New connection...");
				addrlen=sizeof(addr);
				clientfd=accept(fd, &addr, &addrlen);
				if (clientfd<0) {
					PERROR("accept");
					continue;
				}
				if (pollpos==-1) {
					for (x=0; x<pollmax; x++) {
						if (socks_poll[x].fd==-1) {
							pollpos=x;
							break;
						}
					}
				}
				if (pollpos==-1) pollpos=pollmax;
				if (pollpos>=(sizeof(socks_poll)/sizeof(struct pollfd))) {
					PRINTERR("Ran out of available socks_poll[] entries.");
					close(fd);
					continue;
				}
				socks_poll[pollpos].fd=clientfd;
				socks_poll[pollpos].events=POLLIN|POLLERR|POLLHUP|POLLNVAL;
				socks_poll[pollpos].revents=0;
				pollcnt++;

				/* Create session */
				socks_info[pollpos].session=strdup(tcpcgi_gensess());
				socks_info[pollpos].http_port=http_port;
				socks_info[pollpos].http_host=http_host;
				socks_info[pollpos].http_path=http_path;
				tcpcgi_getdata(http_host, http_port, http_path, "open", socks_info[pollpos].session, hostport, NULL, NULL, &status);
				if (status<0) {
					PRINTERR("Error creating session on remote end.");
					CLOSE_POLL(pollpos);
					continue;
				}

				/* This needs to be the last step, it actually registers the session for updates. */
				socks_info[pollpos].fd=clientfd;


				PRINTERR("... from [idx=%i, fd=%i], pollcnt=%i, pollmax=%i: session=%s", pollpos, clientfd, pollcnt, pollmax, socks_info[pollpos].session);
				pollpos=-1;
			} else {
				/* Handle data from an existing connection. */
				PRINTERR("Data waiting on [idx=%i, fd=%i]...", i, fd);
				x=read(fd, buf, sizeof(buf));
				if (x<0) PERROR("read");
				if (x==0) {
					PRINTERR("EOF from [idx=%i, fd=%i].", i, fd);
					CLOSE_POLL(i);
					continue;
				}
				if ((rv=tcpcgi_handledata(http_host, http_port, http_path, socks_info[i].session, fd, buf, x))<0) {
					PRINTERR("Unable to put data.");
					CLOSE_POLL(i);
					continue;
				}
				if (poll_backoff_ireset) {
					/* Check background activity, and adjust polltime if needed. */
					if (handle_background()==0) {
						/* If nothing is happening in the background, slow down. */
						if (poll_backoff<5) { poll_time*=poll_backoff; } else { poll_time+=poll_backoff; }
						PRINTERR("Lengthening polltime [poll_time=%i], nothing is happening.", poll_time);
					} else {
						/* If we're active in the background, speed up! */
						if (poll_backoff<5) { poll_time/=poll_backoff; } else { poll_time-=poll_backoff; }
						if (poll_backoff_reset) poll_time=poll_time_min;
						PRINTERR("Shortening polltime [poll_time=%i], stuff is happening.", poll_time);
					}
					if (poll_time>poll_time_max) poll_time=poll_time_max;
					if (poll_time<poll_time_min) poll_time=poll_time_min;
				}
			}
			if (num_events==0) break;
		}
	}
	return(errcnt);
}
