#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#define RELOC_INSTALL_FAILURE 1
#define RELOC_INSTALL_SUCCESS 0

typedef enum {
	OPT_INPLACE,
	OPT_BASH_SCRIPT,
	_OPT_LASTOPT_
} options_t;

void print_usage(FILE *fp, const char *err_str) {
	if (err_str) {
		fprintf(fp, "Error: %s\n", err_str);
	}

	fprintf(fp, "Usage: rewrite-path [-hi] -o <oldpath> -n <newpath> -f <filename>\n");
	fprintf(fp, "  -h             This help\n");
	fprintf(fp, "  -i             Modify file in-place\n");
	fprintf(fp, "  -o <oldpath>   Specify path to search for\n");
	fprintf(fp, "  -n <newpath>   Specify path to replace with\n");
	fprintf(fp, "  -f <filename>  Specify the file to modify\n");

	return;
}

int rewrite_path(FILE *fp, char *filename, char *oldpath, char *newpath, char *str_delim, unsigned int str_delim_len, long options[_OPT_LASTOPT_]) {
	size_t oldpath_len, newpath_len, buf_len, str_buf_len;
	size_t fread_ret, fwrite_ret;
	long str_start, str_end;
	long item_rem_len;
	char *buf, *str_buf, *item_end_p;
	int char_buf, delim;
	int fseek_ret;
	int idx, offset;

	if (str_delim == NULL) {
		str_delim = "\0\n'\"})]";
		str_delim_len = 7;
	}

	oldpath_len = strlen(oldpath);
	if (oldpath_len == 0) {
		return(-1);
	}

	if (newpath) {
		newpath_len = strlen(newpath);
		if (newpath_len == 0) {
			return(-1);
		}
	} else {
		newpath_len = 0;
	}

	buf_len = oldpath_len;
	buf = malloc(buf_len);
	if (buf == NULL) {
		return(-2);
	}

	while (1) {
		fread_ret = fread(buf, buf_len, 1, fp);
		if (fread_ret != 1) {
			break;
		}

		if (memcmp(buf, oldpath, oldpath_len) != 0) {
			fseek_ret = fseek(fp, (buf_len * -1) + 1, SEEK_CUR);
			if (fseek_ret != 0) {
				break;
			}

			continue;
		}

		/* We have a match! */
		/* 1. Mark start of string */
		str_start = ftell(fp) - buf_len;

		/* 2. Find end of string */
		str_end = 0;
		while (1) {
			/* 2.a. Look for deliminter \0\n'"})] */
			char_buf = fgetc(fp);
			if (char_buf == EOF) {
				break;
			}

			for (idx = 0; idx < str_delim_len; idx++) {
				if (char_buf == str_delim[idx]) {
					str_end = ftell(fp);
					delim = char_buf;

					break;
				}
			}

			if (str_end != 0) {
				break;
			}
		}

		str_buf_len = str_end - str_start;
		str_buf = malloc(str_buf_len + 1);
		if (str_buf == NULL) {
			break;
		}
		str_buf[str_buf_len] = '\0';

		fseek_ret = fseek(fp, str_start, SEEK_SET);
		if (fseek_ret != 0) {
			free(str_buf);

			break;
		}

		fread_ret = fread(str_buf, str_buf_len, 1, fp);
		if (fread_ret != 1) {
			free(str_buf);

			break;
		}

		/* 3. Find end of item */
		/* 3.a. Compute from size of old path */
		item_end_p = str_buf + oldpath_len;
		item_rem_len = str_buf_len - (item_end_p - str_buf);

		/* 4. Print found parameters */
		if (!options[OPT_INPLACE]) {
			if (options[OPT_BASH_SCRIPT]) {
				offset = 0;
				printf("( echo -n \"${NEW_PATH}\"'%.*s'; ",
					(int) (item_rem_len - 1),
					item_end_p
				);

				if (delim != '\n') {
					printf("echo -ne '\\x%02lx'; ",
						(unsigned long) delim
					);

					offset += 1;
				}

				if (delim == '\n' || delim == '\0') {
					offset += 1;
				}

				printf("eval 'for repeat in {1..'\"$[%lu - ${NEW_PATH_LEN}]\"'}; do echo -ne '\"'\"'\\x%lx'\"'\"'; done'; ",
					(unsigned long) str_end - str_start - item_rem_len - offset + 1,
					(unsigned long) ' '
				);

				if (delim == '\n' || delim == '\0') {
					printf("echo -ne '\\x%02lx'; ",
						(unsigned long) delim
					);
				}

				printf(" ) | dd of=%s seek=%lu conv=notrunc bs=1\n",
					filename,
					(unsigned long) str_start
				);
			} else {
				printf("DELIM=0x%04lx START=0x%016lx END=0x%016lx TRAIL=%.*s\n", (unsigned long) delim, (unsigned long) str_start, (unsigned long) str_end, (int) (item_rem_len - 1), item_end_p);
			}
		}

		if (options[OPT_INPLACE]) {
			/* 5. Rewrite string */
			/* 5.a. Replace old with new */
			memcpy(str_buf, newpath, newpath_len);
			memmove(str_buf + newpath_len, item_end_p, item_rem_len);

			/* 5.b. Pad string: pad with spaces, Insert delim. again (if "\0" or "\n") */
			memset(str_buf + newpath_len + item_rem_len, ' ', str_buf_len - newpath_len - item_rem_len);
			if (delim == '\n') {
				str_buf[newpath_len + item_rem_len - 1] = ' ';
			}
			if (delim == '\0' || delim == '\n') {
				str_buf[str_buf_len - 1] = delim;
			}

			fseek_ret = fseek(fp, str_start, SEEK_SET);
			if (fseek_ret != 0) {
				free(str_buf);

				break;
			}

			fwrite_ret = fwrite(str_buf, str_buf_len, 1, fp);
			if (fwrite_ret != 1) {
				free(str_buf);

				break;
			}
		}
	}

	free(buf);

	return(0);
}

int main(int argc, char **argv) {
	char *newpath = NULL, *oldpath = NULL, *filename = NULL;
	FILE *fp;
	long options[_OPT_LASTOPT_] = {0};
	int bad_args = 0;
	int ret;
	int ch;

	while ((ch = getopt(argc, argv, "hbio:n:f:")) != -1) {
		switch (ch) {
			case 'o':
				oldpath = optarg;

				break;
			case 'n':
				newpath = optarg;

				break;
			case 'f':
				filename = optarg;

				break;
			case 'i':
				options[OPT_INPLACE] = 1;

				break;
			case 'b':
				options[OPT_INPLACE] = 0;
				options[OPT_BASH_SCRIPT] = 1;

				break;
			case 'h':
				print_usage(stdout, NULL);

				return(RELOC_INSTALL_SUCCESS);
			case ':':
				print_usage(stderr, NULL);

				return(RELOC_INSTALL_FAILURE);
			case '?':
				print_usage(stderr, NULL);

				return(RELOC_INSTALL_FAILURE);
			default:
				print_usage(stderr, "Internal error");

				return(RELOC_INSTALL_FAILURE);
		}
	}

	if (oldpath == NULL) {
		fprintf(stderr, "Error: '-o' option must be specified\n");

		bad_args = 1;
	}
	if (options[OPT_INPLACE]) {
		if (newpath == NULL) {
			fprintf(stderr, "Error: '-n' option must be specified\n");

			bad_args = 1;
		}
	}
	if (filename == NULL) {
		fprintf(stderr, "Error: '-f' option must be specified\n");

		bad_args = 1;
	}
	if (bad_args) {
		print_usage(stderr, "Mandatory arguments missing");

		return(RELOC_INSTALL_FAILURE);
	}

	fp = fopen(filename, "r+");
	if (fp == NULL) {
		fprintf(stderr, "Error: Unable to open \"%s\"\n", filename);
		perror("Error: fopen");

		return(RELOC_INSTALL_FAILURE);
	}

	ret = rewrite_path(fp, filename, oldpath, newpath, NULL, 0, options);

	fclose(fp);

	if (ret == 0) {
		return(RELOC_INSTALL_SUCCESS);
	}

	return(RELOC_INSTALL_FAILURE);
}
