/* Stereograph 0.16, 12/04/2000;
 * Copyright (c) 2000 by Fabian Januszewski <fabian.linux@januszewski.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdio.h>
#include <stdlib.h>

#include "renderer.h"
#include "gfxio.h"
#include "stereograph.h"
#include "globals.h"


char *base_file_name;
char *texture_file_name;
char *stereo_file_name;

/* interface to the renderer */
struct PARAMS *pParam;
struct GFX_DATA *pBase, *pTexture, *pStereo;


void print_info(void) {
	printf("SYNOPSIS\n  stereograph [OPTIONS] -b [FILENAME] -t [FILENAME] [-o [FILENAME]]\n\n");
	printf("OPTIONS\n  -a anti-aliasing     -z zoom               (quality)\n  -d distance          -e eye shift          (perspective)\n  -x texture insert x  -y texture insert y   (layout)\n  -r this flag enables anti-artefact feature (experimental)\n  -v you should know this flag :)\n\n");
}

int main (int argc, char **argv) {

	int last_error = 0;

	Get_GFX_Pointers(&pParam, &pBase, &pTexture, &pStereo);

	pParam->Zoom = 1;
	pParam->AA = 4;
	pParam->AAr = 0;
	pParam->Startx = 0;
	pParam->Starty = 0;
	pParam->Front = 1.0;
	pParam->Distance = 0.6;
	pParam->Eyeshift = 0.0;

	pBase->Data = NULL;
	pTexture->Data = NULL;
	pStereo->Data = NULL;
	base_file_name = NULL;
	texture_file_name = NULL;
	stereo_file_name = NULL;

	verbose = 0;

	parse_args(argc, argv);

	if(verbose) printf("Stereograph for linux Version %s\n", "0.16");




	if(base_file_name && texture_file_name) {
		if(verbose) printf("loading base '%s'...", base_file_name);
		if(!Read_Gfx_File(base_file_name, pBase)) {
			if(verbose) printf("loading texture '%s'...", texture_file_name);
			if(!Read_Gfx_File(texture_file_name, pTexture)) {

				/* setting up renderer; */
				if(verbose) printf("initializing renderer...");
				if(!(last_error = Initialize_Renderer())) {
					int y, s;
					if(verbose) printf("success;\n");
					if(verbose) printf("using parametres as follows (zadexy): %i %i %f %f %i %i;\n", pParam->Zoom, pParam->AA, pParam->Distance, pParam->Eyeshift, pParam->Startx, pParam->Starty);
					if(verbose && pParam->AAr)
						printf("experimental anti-artefacts feature enabled;\n");
					if(verbose) printf("processing...");

					for(y = 0, s = 0; (y < pBase->Height); y++) {
						ProcessLine(y);
						/* print simple status points (without refresh) */
						for( ; verbose && (s < (58 * y / pBase->Height)); s++)
							printf(".");
					}

					if(verbose) printf("completed;\n");
	
					if(verbose) printf("writing stereogram to disk...");
					Write_Gfx_File(stereo_file_name, pStereo);
				} else {
					if(verbose)
						printf("FAILED;\n");
					else
						fprintf(stderr, "initializing renderer...FAILED;\n");
					switch (last_error) {
						case -1 :
							fprintf(stderr, "illegal parametre;\n\n");
							print_info();
							break;
						case -2 :
							fprintf(stderr, "width of texture file greater than the base;\nthis state is not processable;\n");
							break;
						case -3 :
							fprintf(stderr, "could not allocate memory.\n");
					}
					return -1;
				}
				Clear_Renderer();
			}
			free(pTexture->Data);
		}
		free(pBase->Data);
	} else {
		fprintf(stderr, "NOTIFY: must specify at least a base and a texture file!\n\n");
		print_info();
		return -1;
	}
	return 0;
}

int parse_args (int argc, char **argv) {
	int z;
	for(z = 1; z < argc; z++)
		if(argv[z][0] == '-')
			switch(argv[z][1]) {
				case 'b' :
					base_file_name = argv[++z];
					break;
				case 't' :
					texture_file_name = argv[++z];
					break;
				case 'o' :
					stereo_file_name = argv[++z];
					break;
				case 'x' :
					pParam->Startx = atoi(argv[++z]);
					break;
				case 'y' :
					pParam->Starty = atoi(argv[++z]);
					break;
				case 'd' :
					pParam->Distance = (float) atof(argv[++z]);
					break;
				case 'e' :
					pParam->Eyeshift = (float) atof(argv[++z]);
					break;
				case 'a' :
					pParam->AA = atoi(argv[++z]);
					break;
				case 'z' :
					pParam->Zoom = atoi(argv[++z]);
					break;
				case 'r' :
					pParam->AAr = 1;
					break;
				case 'v' :
					verbose = 1;
					break;
				fprintf(stderr, "invalid argument '%c';\n", argv[z][1]);
				return -1;
			}
	return 0;
}
