/*
 * Copyright (C) 2000  Roy Keene
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *      email: rkeene@netfueldesign.com
 */


/*
	Dynamic Compression Routines 
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include "dact.h"
#include "algorithms.h"

int print_help(int argc, char **argv) {
	printf("DACT version %i.%i.%i by Net Fuel Design (info@netfueldesign.com).\n",DACT_VER_MAJOR,DACT_VER_MINOR,DACT_VER_REVISION);
	printf("Usage:\n\t%s [-d] [-f] [-v] [-c] [--] <filename> [outfile]\n",argv[0]);
	printf("  -d\tDecompress instead of compressing.\n");
	printf("  -f\tForce overwriting existing files.\n");
	printf("  -v\tIncrease verbosity.\n");
	printf("  -c\tComplain when compression verification fails.\n");
	return(0);
}



int main(int argc, char **argv) {
	char in_filename[128], out_filename[128];
	char in_buffer[DACT_BLK_SIZE], prev_buffer[DACT_BLK_SIZE];
	char out_buffer[DACT_BLK_SIZE*2];
	char store_buffer[DACT_BLK_SIZE*2];
	char verify_buffer[DACT_BLK_SIZE*2];
	char cmp_options=0;
	char file_version[3]={DACT_VER_MAJOR, DACT_VER_MINOR, DACT_VER_REVISION};
	char option_verbose=0;
	char option_complain=0;
	char completed;
	int in_fd, out_fd;
	int i,x=0;
	int out_size=-1, out_lowest, out_algo=-1;
	int file_size, out_file_size;
	int mode=DACT_MODE_COMPR, offset_arg=0;
	int mode_force=0;
	int number_algorithm=0;
	int verify_size;
	int verified_data;
	struct stat statinfo;
	struct dact_header header;


#ifdef DEBUG
	option_verbose=1;
#endif

/* TODO: This needs to be redone (argument parsing) */

	while (1) {
		if (argv[1+offset_arg]!=NULL) {
			completed=0;
			if (!strncmp(argv[1+offset_arg],"-d",2)) {
				mode=DACT_MODE_DECMP;
				completed=1;	
			}
			if (!strncmp(argv[1+offset_arg],"-f",2)) {
				mode_force=1;
				completed=1;
			}
			if (!strncmp(argv[1+offset_arg],"-v",2)) {
				option_verbose++;
				completed=1;
			}
			if (!strncmp(argv[1+offset_arg],"-c",2)) {
				option_complain=1;
				completed=1;
			}
			if (!strncmp(argv[1+offset_arg],"--",2)) {
				offset_arg++;
				completed=0;
			}
			if (!strncmp(argv[1+offset_arg],"-h",2)) {
				print_help(argc,argv);
				return(0);
			}
			if (completed==0) break;
			offset_arg++;
		} else {
			break;
		}
	}


	for (i=0;i<sizeof(prev_buffer);i++) prev_buffer[i]=0;
	if (argv[1+offset_arg]==NULL) {
		in_filename[0]=0;
	} else {
		strncpy(in_filename,argv[1+offset_arg],sizeof(in_filename)-1);
	}

	if (!strcmp(in_filename,"-") || in_filename[0]==0) {
		in_fd=STDIN_FILENO;
	} else { 
		if ((in_fd=open(in_filename,O_RDONLY))==-1) {
			printf("%s: Unable to open %s for reading.\n",argv[0],in_filename);
			return(-1);
		}
	}

	if (argc==(3+offset_arg)) {
		strncpy(out_filename,argv[2+offset_arg],sizeof(out_filename)-1);
		if (!access(out_filename, F_OK)) {
			if (mode_force) {
				unlink(out_filename);
			} else {
				printf("%s: %s exists, exiting.\n",argv[0],out_filename);
				close(in_fd);
				return(-1);
			}
		}

		if ((out_fd=open(out_filename,O_WRONLY|O_CREAT,0644))==-1) {
			printf("%s: Unable to open %s for writing.\n",argv[0],out_filename);
			return(-1);
		}
	} else {
		if (isatty(STDERR_FILENO)) {
			printf("%s: Refusing to write compressed output to a terminal.\n",argv[0]);
			return(-1);
		} else {
			out_fd=STDERR_FILENO;
		}
	}


	for (i=0;i<255;i++) { if (algorithms[i]==NULL) break; }
	number_algorithm=i;

/* This should be phased out as much as possible (though still required
   to determine how much of BLOCK is required when decompressing.
*/
	fstat(in_fd, &statinfo);
	file_size=statinfo.st_size;

	switch (mode) { 
		case DACT_MODE_COMPR:
			write(out_fd,&file_version,sizeof(file_version));
			write(out_fd,&file_size,sizeof(file_size));
			write(out_fd,&cmp_options,sizeof(cmp_options));
			while (1) {
				x++;
				out_lowest=DACT_BLK_SIZE*2;
				if (read(in_fd,&in_buffer,sizeof(in_buffer))==0) break;
				for (i=0;i<255;i++) {
					if (algorithms[i]==NULL) break;
					out_size=algorithms[i](DACT_MODE_COMPR, &prev_buffer, &in_buffer, &out_buffer, sizeof(in_buffer));
					if (option_verbose>2)
						printf("  --       algo #%i: size=%i\n",i,out_size);
					if (out_size>0) {
						verify_size=algorithms[i](DACT_MODE_DECMP, &prev_buffer, &out_buffer, &verify_buffer, out_size);
					} else {
						verify_size=-1;
					}

					verified_data=0;
					if (verify_size==sizeof(in_buffer)) {
						if (!memcmp(&verify_buffer, &in_buffer, sizeof(in_buffer))) {
							verified_data=1;
						}
					}

					if (!verified_data && verify_size>0 && option_complain) {
						printf("Block #%05i: algo=%03i: Verification failed! %34s\n",x,i,algorithm_names[i]);
					}

					if (out_size<out_lowest && (out_size > 0) && verified_data) {
						memcpy(store_buffer,out_buffer,out_size);
						out_algo=i;
						out_lowest=out_size;
					}
					memcpy(prev_buffer,in_buffer,sizeof(in_buffer));
				}
				if (out_lowest==(DACT_BLK_SIZE*2)) {
					printf("Compression resulted in DACT_BLK_SIZE*2 !  Aborting.\n");
					return(-1);
				}
				if (option_verbose>1) {
					printf("Block #%05i: algo=%03i (size=%05i, %05i)  %35s\n", x,out_algo,out_lowest,(int) (out_lowest+sizeof(header.algo)+sizeof(header.size)),algorithm_names[out_algo]);
				}
				if (option_verbose==1) {
					printf("Block %5i/%5i\r",x,(int) (((float) (((float) file_size)/DACT_BLK_SIZE))+0.99));
					fflush(stdout);
				}
				header.size=(TWO_BYTES) (out_lowest);
				header.algo=out_algo;
				write(out_fd,&header.size,sizeof(header.size));
				write(out_fd,&header.algo,sizeof(header.algo));
				write(out_fd,&store_buffer,out_lowest);
			}
			break;
		case DACT_MODE_DECMP:
			i=0;
			read(in_fd,&file_version,sizeof(file_version));
			read(in_fd,&out_file_size,sizeof(out_file_size));
			read(in_fd,&cmp_options,sizeof(cmp_options));
			x=sizeof(out_file_size)+sizeof(cmp_options);

			if (file_version[0]!=DACT_VER_MAJOR) {
				printf("Major version numbers do not match! (%i!=%i)\n",file_version[0],DACT_VER_MAJOR);
				printf("File is not usable, exiting...\n");
				close(in_fd);
				close(out_fd);
				return(-1);
			}
			if (file_version[1]!=DACT_VER_MINOR || file_version[2]!=DACT_VER_REVISION) {
				printf("Minor or Revision numbers do not match (file=%i.%i.%i != current=%i.%i.%i)\n",file_version[0],file_version[1],file_version[2],DACT_VER_MAJOR,DACT_VER_MINOR,DACT_VER_REVISION);
				printf("File may not be usuable.\n");
			}


			while(1) {
				i++;
				read(in_fd,&header.size,sizeof(header.size));
				read(in_fd,&header.algo,sizeof(header.algo));
				read(in_fd,&in_buffer,header.size);
				x+=sizeof(header.size)+sizeof(header.algo)+header.size;
				if (header.algo>=(number_algorithm-1)) {
					printf("Request for an unknown algorithm: %i\n", header.algo);
					close(in_fd);
					close(out_fd);
					return(-1);
				}
				out_size=algorithms[header.algo](DACT_MODE_DECMP, &prev_buffer, &in_buffer, &out_buffer, header.size);
				if (out_size!=DACT_BLK_SIZE) {
					printf("Block resulted in data that is less than %i bytes.\n",DACT_BLK_SIZE);
					close(in_fd);
					close(out_fd);
					return(-1);
				}
				if (option_verbose>1) {
					printf("Block #%05i: algo=%03i (size=%05i, %05i) %35s\n",i,header.algo,header.size,header.size+3,algorithm_names[header.algo]);
				} 
				if (option_verbose==1) {
					printf("Block %5i/%5i\r",(i-1),(int) (((float) (((float) out_file_size)/DACT_BLK_SIZE))+0.99));
					fflush(stdout);
				}
				if (((i*DACT_BLK_SIZE))>=out_file_size) {
					out_size-=((i*DACT_BLK_SIZE)-out_file_size);
				}
				write(out_fd,&out_buffer,out_size);
				if (x>=file_size) break;
				memcpy(prev_buffer,out_buffer,out_size);
			}
			break;
	}

	close(in_fd);
	close(out_fd);

	if (option_verbose==1) printf("\n");
	return(0);
}
