#define LANBDD
#include "config.h"
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/poll.h>
#include <pthread.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include "lanbd_whohas.h"
#include "lanbd_write.h"
#include "lanbd_read.h"
#include "license.h"
#include "dendian.h"
#include "lanbdd.h"
#include "pqueue.h"
#include "lanbd.h"
#include "net.h"

/*
 * SYNOPSIS
 *    signed int proc_handledata(void);
 *
 * ARGUMENTS
 *    (none)
 *
 * DESCRIPTION
 *    proc_handledata should execute in the context of the master thread.
 *    It pulls data from the "packet_queue" interface, breaks the header
 *    apart and dispatches the appropriate function handler (above).
 *
 * RETURN VALUE
 *    This function returns the number of events processed.
 *
 */
signed int proc_handledata(void) {
	struct sockaddr_in *dest=NULL;
	struct lanbd_packet *p_data=NULL;
	packet_queue_t *pqi=NULL;
	signed int retval=0;
	char *buf=NULL;
	int buflen=0;
	int p_header_size=0;
	int (*cmdfunc)(struct lanbd_packet *)=NULL;
	int cmdret=-1;

	DBG_ENTER("%s", "");

	/* Allocate our packet data structure. */
	p_data=malloc(sizeof(*p_data));
	if (p_data==NULL) {
		PRINTERR("Couldn't allocate data structure, dropping packet.");
		return(0);
	}

	/* The header size is the size of all the members, cmd is really a uint8_t. */
	p_header_size=sizeof(p_data->rport)+sizeof(p_data->device)+sizeof(p_data->block)+sizeof(uint8_t);

	/* Get as many as 8192 packets at a time.  8192 is arbitrary, but it seems to work well. */
	for (retval=0; retval<8192; retval++) {
		/* Pull a packet from the queue */
		pqi=packet_queue_get(0);

		/* If we couldn't pull a packet from the queue, stop trying. */
		if (pqi==NULL) break;

		/* Cast our `dest' back to sockaddr_in */
		dest=(struct sockaddr_in *) pqi->dest;

		/* Do basic verifications, if failed, abort processing. */
		if ((pqi->datalen)<p_header_size) {
			PRINTERR("Not enough data in message to comprise basic header, dropping!");
			packet_queue_delete(pqi);
			continue;
		}

		/* Handle the packet, make a pointer we can move as we read data. */
		buf=pqi->data;
		buflen=pqi->datalen;

		/* Get common header information from the packet. */
		p_data->cmd=de_readbuf(buf, sizeof(uint8_t)); buf+=sizeof(uint8_t); buflen-=sizeof(uint8_t);
		p_data->rport=de_readbuf(buf, sizeof(p_data->rport)); buf+=sizeof(p_data->rport); buflen-=sizeof(p_data->rport);
		p_data->device=de_readbuf(buf, sizeof(p_data->device)); buf+=sizeof(p_data->device); buflen-=sizeof(p_data->device);
		p_data->block=de_readbuf(buf, sizeof(p_data->block)); buf+=sizeof(p_data->block); buflen-=sizeof(p_data->block);
		p_data->data=buf;
		p_data->datalen=buflen;
		p_data->addr=pqi->dest;
		p_data->addrlen=pqi->destlen;

		/* If packet was sent from localhost, assume it was not broadcast. */
		if (dest->sin_addr.s_addr==htonl(INADDR_LOOPBACK)) {
			p_data->flags=PFL_NONE;
		} else {
			p_data->flags=PFL_BROADCAST;
		}

		/* Security check, to disallow just anyone from reading/writing. */
		if (dest->sin_port>LANBD_RPORT_MAX || p_data->rport>LANBD_RPORT_MAX) {
			PRINTERR("[security] Invalid port number (sport=%i,rport=%i), packet dropped.", dest->sin_port, p_data->rport);
			packet_queue_delete(pqi);
			continue;
		}

		/* Determine the correct function to call based on the `cmd' information, and call it. */
		cmdfunc=cmd_val2func(p_data->cmd);

		/* If no function could be found, abort processing. */
		if (cmdfunc==NULL) {
			PRINTERR("No such command (cmd=%i), packet dropped.", p_data->cmd);
			packet_queue_delete(pqi);
			continue;
		}

		/*
		 * Change the port number in the packet data to the rport, this helps the
		 * command function send replies.
		 */
		dest->sin_port=htons(p_data->rport);

		/* This is the `call it' part */
		cmdret=cmdfunc(p_data);
		if (cmdret<0) {
			PRINTERR("Command failed!");
		}

		/* Delete the packet from the queue. */
		packet_queue_delete(pqi);
	}

	if (p_data) free(p_data);
	return(retval);
}

#ifdef LANBDD_SIGNALLING
/*
 * SYNOPSIS
 *    void sighandler(
 *           int sig
 *         );
 *
 * ARGUMENTS
 *    int sig                Signal recieved
 *
 * DESCRIPTION
 *    `sighandler' is the signal handler, currently it calls
 *    proc_handledata() if sig==SIGIO to handle incoming data.
 *
 * RETURN VALUE
 *    (none)
 *
 * NOTES
 *    This function is obsolete because it's not safe to enable
 *    it LANBDD_SIGNALLING should be defined.
 */
void sighandler(int sig) {
	static int insig=0;


	if (insig) {
		PRINTERR("ERROR ERROR ERROR ERROR Signal handler called while in signal handler!!! PANIC PANIC PANIC PANIC PANIC (pid=%i)", getpid())
		sleep(100);
		return;
	}

	insig=1;

	DBG_ENTER("%i", sig);

	switch (sig) {
		case SIGIO:
			proc_handledata();
			break;
		default:
			break;
	}

	insig=0;
	return;
}
#endif


/*
 * SYNOPSIS
 *    void *thread_cleanup(
 *            void *args
 *          );
 *
 * ARGUMENTS
 *    void *args             Pointer to argument information, ignored.
 *
 * DESCRIPTION
 *    This function executes cleanup processes when idle, it may or may not
 *    be executed in the context of a new thread.  There may or may not be
 *    things to clean up.
 *
 * RETURN VALUE
 *    This function only returns NULL.
 *
 */
void *thread_cleanup(void *args) {
	int numcleaned=0;

	DBG_ENTER("%p", args);

	/* Cleanup packets from the packet queue. */
	numcleaned=packet_queue_cleanup(NULL);
	PRINTERR("Cleaned %i packets from the packet queue.", numcleaned);

	return(NULL);
}

/*
 * SYNOPSIS
 *    void *thread_getdata(
 *            void *args
 *          );
 *
 * ARGUMENTS
 *    void *args             This is a pointer to a `struct thread_data'
 *                           which contains information about how this
 *                           thread should operate, such as:
 *                               tail        Pointer to the end of the
 *                                           packet_queue.
 *                               cleantime   Timeout (in milliseconds) for
 *                                           cleanup processes to begin if
 *                                           no data is recieved.
 *                               notifypid   PID to send a SIGIO to when new
 *                                           data arrives.
 *                               port        UDP port to listen on for data.
 *                               packetsize  Maximum packet size to assume
 *                                           we're going to receive.
 *
 * DESCRIPTION
 *    This function executes as a new thread.  It creates a listening UDP
 *    socket and handles data from that socket by adding that data to the
 *    packet_queue. It spawns a new thread to handle cleanup when needed.
 *    It notifies the "notifypid" through the signal SIGIO after it has
 *    added the item to the packet_queue.
 *
 * RETURN VALUE
 *    This function returns NULL.
 *
 */
void *thread_getdata(void *arg) {
	pthread_t pthread_data; /* Uninitialized! */
	int pthread_ret=-1;
	packet_queue_t *pq_add_ret=NULL, *tail=NULL;
	uint16_t lanbd_portno=0;
	struct thread_data *td=arg;
	struct pollfd poll_fds[1];
	struct sockaddr_in src; /* Uninitialized */
	socklen_t srclen=sizeof(src);
	ssize_t msglen=0;
	char *buf;
	int buflen=0;
	int poll_ret=0;
	int sockfd=0;
	int cleanuptime=16384;
	unsigned int packetcount=0;
	sigset_t blocksigs;

	DBG_ENTER("%p", arg);

	sigaddset(&blocksigs, SIGIO);
	sigprocmask(SIG_BLOCK, &blocksigs, NULL);

	if (arg==NULL) return(NULL);
	if (td->notifypid==getpid()) {
		PRINTERR("NOTE!  Thread has same pid as parent, this will cause problems.");
	}

	buflen=td->packetsize+sizeof(struct lanbd_packet);
	buf=malloc(buflen);
	if (buf==NULL) {
		PRINTERR("Couldn't create a buffer!  Forcefully exiting.");
		exit(EXIT_FAILURE);
	}

	tail=td->tail;
	lanbd_portno=td->port;
	cleanuptime=td->cleantime;

	sockfd=net_listen_udp(lanbd_portno);
	if (sockfd<0) {
		PERROR("net_listen_udp");
		PRINTERR("Could not create listening socket, forcefully exiting.");
		exit(EXIT_FAILURE);
	}
	poll_fds[0].fd=sockfd;
	poll_fds[0].events=POLLIN|POLLPRI|POLLERR|POLLHUP;
	poll_fds[0].revents=0;

	while (1) {
		poll_ret=poll(poll_fds, 1, cleanuptime);
		if (poll_ret<0) continue;

		/* If there are no events, we must have timed out, let's cleanup. */
		if (poll_ret==0) {
			/* Disable further cleanups (will be enabled when data is recieved) */
			cleanuptime=-1;

			/* Reset the packet counter since we've gone back to low traffic mode. */
			packetcount=0;

			/* Ensure we didn't accidently leave something going on in the background. */
#ifdef LANBDD_SIGNALLING
			kill(td->notifypid, SIGIO);
#endif

			/* Cleaning up might remove our tail, we'll fix that later. */
			tail=NULL;

			/* Try to create a thread to handle the cleanup, so we can get back to listening. */
			pthread_ret=pthread_create(&pthread_data, NULL, thread_cleanup, NULL);
			if (pthread_ret<0) {
				PRINTERR("Could not create thread!  This is not good, running cleanup in foreground.");
				thread_cleanup(NULL);
			}
			continue;
		}

		/* Reset the cleanuptime, since it may have been disabled. */
		cleanuptime=td->cleantime;

		/* We don't need to figure out which socket, because we only have one. */
		msglen=recvfrom(sockfd, buf, buflen, 0, (struct sockaddr *) &src, &srclen);

		/* IF we got no data, don't add it to the queue. */
		if (msglen==0) continue;

		/* Same for an error, but print out something. */
		if (msglen<0) {
			PERROR("recvfrom");
			continue;
		}

		/* If we get a packet with a bad srcaddr, ignore it. */
		if (src.sin_addr.s_addr==0) {
			PRINTERR("Weird packet with source address of 0x0!  Ignoring it.");
			continue;
		}

#ifdef LANBDD_CHECKFUNCBEFOREADDING
		/*
		 * This is experimental code to check the function before
		 * adding to the queue.  It should not be needed in a
		 * production environment, but it will help keep bad traffic
		 * down in a hostile one.
		 */
		if (cmd_val2func(buf[0])==NULL) {
			continue;
		}
#endif

		/* Add the packet to the list of unprocessed packets, and set it's readable flag. */
		pq_add_ret=packet_queue_add(PQ_COPY, (struct sockaddr *) &src, srclen, msglen, buf, tail);
		if (pq_add_ret==NULL) {
			PRINTERR("Packet dropped!");
			continue;
		}
		tail=pq_add_ret;

#ifdef LANBDD_SIGNALLING
		/* Notify our parent thread about the new data. */
		if (packetcount>1024) {
			if (packetcount%851) {
				kill(td->notifypid, SIGIO);
			}
		} else {
			kill(td->notifypid, SIGIO);
		}
#endif

		packetcount++;

#ifdef DEBUG
		if (packetcount==1024) {
			PRINTERR(" *** NOTE *** : Going into high-traffic mode.");
		}
#endif
	}

	return(NULL);
}

/*
 * SYNOPSIS
 *    int main(
 *          int argc,
 *          char **argv
 *        );
 *
 * ARGUMENTS
 *    int argc               This is set to the number of elements in the
 *                           array `argv' (see below).
 *    char **argv            This is an array of pointers passed to us by
 *                           the operating system, each element is a argument
 *                           to the binary called.
 *
 * DESCRIPTION
 *    This function checks licensing information, processes command line
 *    arguments, processes the configuration file, configures the export
 *    table, creates a signal handler for the master thread, and spawns
 *    a thread to feed data into the packet_queue.
 *
 * RETURN VALUE
 *    0 is returned on success, non-zero otherwise.
 *
 */
int main(int argc, char **argv) {
	packet_queue_t *pqtail=NULL;
	struct thread_data *td=NULL;
	pthread_t pthread_data; /* Uninitialized! */
#ifdef LANBDD_SIGNALLING
	struct sigaction sigact, sigactign; /* Uninitialized! */
#endif
	int pthread_ret=0;
	int packetsprocessed=0; /* XXX: TEMPORARY */
#ifndef DEBUG
	char *licfile=LANDISK_LICFILE;
#endif

	DBG_ENTER("%i, %p", argc, argv);

#ifndef DEBUG
	/* Verify license information. */
	if (license_check(licfile, "lanbd", LANBD_VERSION)<0) {
		fprintf(stderr, "Could not validate license from license file: %s\n", licfile);
		return(-1);
	}
#endif

	/* Process command line arguments. XXX */

	/* Process config file. XXX */

	/* Create export table. XXX */

	/* Initialize the packet queue */
	pqtail=packet_queue_init();
	if (pqtail==NULL) {
		PRINTERR("Couldn't initialize packet_queue! Exiting!");
		return(-1);
	}

#ifdef LANBDD_SIGNALLING
	/* Implement a signal handler, to do event-driven I/O. */
	sigact.sa_handler=sighandler;
	sigactign.sa_handler=SIG_IGN;
#ifdef SA_RESTART
	sigact.sa_flags=SA_RESTART;
#endif
	sigaction(SIGIO, &sigact, NULL);
#endif

	/* Create a thread to handle incoming data and make it accessible. */
	td=malloc(sizeof(struct thread_data));
	if (td==NULL) {
		PRINTERR("Couldn't allocate space for thread_data!  Exiting!");
		return(-1);
	}
	td->port=LANBD_PORT;
	td->notifypid=getpid();
	td->tail=pqtail;
	td->cleantime=LANBD_CLEANTIME;
	td->packetsize=LANBD_PACKETSIZE;
	pthread_ret=pthread_create(&pthread_data, NULL, thread_getdata, td);
	if (pthread_ret<0) {
		PERROR("pthread_create")
		return(-1);
	}

	/* Pull data from queue (done in background, while we sleep). */
	packetsprocessed=0;
	while (1) {
		sleep(0xffff);
#ifdef LANBDD_SIGNALLING
		sigaction(SIGIO, &sigactign, NULL);
#endif
		packetsprocessed+=proc_handledata();
#ifdef LANBDD_SIGNALLING
		sigaction(SIGIO, &sigact, NULL);
#endif
		SPOTVAR_I(packet_queue_length())
		SPOTVAR_I(packetsprocessed);
	}

	return(0);
}
