#include "config.h"
#include <inttypes.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>

#include "license.h"


/*
 * SYNOPSIS
 *    uint64_t strto64(
 *               const char *n
 *             );
 *
 * ARGUMENTS
 *    const char *n          Base-10 value in a C-string.
 *
 * DESCRIPTION
 *    This function converts a string into it's 64bit value.
 *
 * RETURN VALUE
 *    A 64bit representation of the value of the string.
 *
 */
static uint64_t strto64(const char *n) {
	uint64_t retval=0;
	uint64_t multiplier=0;
	int i, m, x;

	if (n==NULL) return(0);

	m=strspn(n, "0123456789");

	for (i=0; i<m; i++) {
		multiplier=1;
		for (x=0; x<(m-i-1); x++) multiplier*=10;
		retval+=(n[i]-48)*multiplier;
	}

	return(retval);
}


/*
 * SYNOPSIS
 *    uint64_t license_checksum(
 *               const char *line
 *             );
 *
 * ARGUMENTS
 *    const char *line       Line to checksum, it should not include the
 *                           checksum field.
 * 
 * DESCRIPTION
 *    This function calculates the checksum of a line, to verify its
 *    authenticity.
 *
 * RETURN VALUE
 *    The 64bit checksum of the variable ``line.''
 *
 */
static uint64_t license_checksum(const char *line) {
        uint64_t h=0,g=0;
	unsigned char *s=(unsigned char *) line;

	if (s==NULL) return(0);

	while (*s) {
		h=(h<<9)+(*s*255)+1;
		if (h&(0xf<<(sizeof(h)-4))) {
			g+=h;
			h+=(g&0xab3faafd);
		}
		*s++;
        }

        return(h);
}

/*
 * SYNOPSIS
 *    int license_check(
 *          const char *file,
 *          const char *svc,
 *          const char *version
 *        );
 *
 * ARGUMENTS
 *    const char *file       Filename containing license data.
 *    const char *svc        Service name to check validity.
 *    const char *version    Version number of service to check.
 *
 * DESCRIPTION
 *    This function checks the specified license file for the specified
 *    service entry.  Entries are in the format of (1 per line):
 *      service-name|organizational-information|lic-no|expiration|checksum
 *      service-name   is the name of the service.
 *      organizati...  is arbitrary information about the license.
 *      lic-no         is the license number, for remote verification.
 *      expiration     is the expiration of the license in m/d/y format.
 *      checksum       is the checksum of the previous information,
 *                     calculated by the function license_checksum().
 *
 * RETURN VALUE
 *    0    License is valid for this service.
 *    -1   License is invalid for this service.
 */
signed int license_check(const char *file, const char *svc, const char *version) {
	FILE *fp=NULL;
	char buf[2048]={0}, *buf_csum=NULL, buf_svc[64]={0}, *buf_svc_p=NULL, *buf_ver=NULL;
	uint64_t csum=0, csum_calc=0;
	signed int ret=-1;

	/* Basic sanity check. */
	if (file==NULL || svc==NULL || version==NULL) return(-1);

	fp=fopen(file, "r");
	if (fp==NULL) return(-1);

	while (!feof(fp)) {
		if (fgets(buf, sizeof(buf), fp)==NULL) {
			if (!feof(fp)) PERROR("open");
			break;
		}
		if (strlen(buf)==0) continue;
		while (buf[strlen(buf)-1]<' ' && strlen(buf)>0) buf[strlen(buf)-1]='\0';

		/* Get the service name from the line. */
		memcpy(buf_svc, buf, sizeof(buf_svc));
		buf_svc[sizeof(buf_svc)-1]='\0';
		buf_svc_p=strchr(buf_svc, '|');
		if (buf_svc_p) {
			*buf_svc_p='\0';
			buf_ver=buf_svc_p+1;
			buf_svc_p=strchr(buf_svc_p+1, '|');
			if (buf_svc_p) *buf_svc_p='\0';
		}

		/* If the service doesn't match, keep looking. */
		if (strcmp(svc, buf_svc)!=0) continue;
		if (strncmp(version, buf_ver, strlen(buf_ver))!=0) continue;

		buf_csum=strrchr(buf, '|');
		if (buf_csum==NULL) continue;
		*buf_csum='\0';
		*buf_csum++;
		csum=strto64(buf_csum);
		csum_calc=license_checksum(buf);
		ret=(csum-csum_calc);
		break;
	}

	fclose(fp);
	return(-(!!ret));
}



#ifdef LICENSE_OVERRIDE
uint64_t license_csum(const char *line) {
	return(license_checksum(line));
}
#endif
