	"#! /usr/bin/env tcl\n"
	"\n"
	"package require vfs\n"
	"\n"
	"namespace eval ::vfs::cvfs {}\n"
	"\n"
	"# Convience functions\n"
	"proc ::vfs::cvfs::Mount {hashkey local} {\n"
	"	vfs::filesystem mount $local [list ::vfs::cvfs::vfshandler $hashkey]\n"
	"	catch {\n"
	"		vfs::RegisterMount $local [list ::vfs::cvfs::Unmount]\n"
	"	}\n"
	"}\n"
	"\n"
	"proc ::vfs::cvfs::Unmount {local} {\n"
	"	vfs::filesystem unmount $local\n"
	"}\n"
	"\n"
	"# Implementation\n"
	"## I/O Handlers (pass to appropriate hashkey)\n"
	"namespace eval ::vfs::cvfs::data {}\n"
	"proc ::vfs::cvfs::data::getChildren args {\n"
	"	set hashkey [lindex $args 0]\n"
	"\n"
	"	set cmd \"::vfs::cvfs::data::${hashkey}::getChildren\"\n"
	"	set cmd [linsert $args 0 $cmd]\n"
	"\n"
	"	eval $cmd\n"
	"}\n"
	"\n"
	"proc ::vfs::cvfs::data::getMetadata args {\n"
	"	set hashkey [lindex $args 0]\n"
	"\n"
	"	set cmd \"::vfs::cvfs::data::${hashkey}::getMetadata\"\n"
	"	set cmd [linsert $args 0 $cmd]\n"
	"\n"
	"	eval $cmd\n"
	"}\n"
	"\n"
	"proc ::vfs::cvfs::data::getData args {\n"
	"	set hashkey [lindex $args 0]\n"
	"\n"
	"	set cmd \"::vfs::cvfs::data::${hashkey}::getData\"\n"
	"	set cmd [linsert $args 0 $cmd]\n"
	"\n"
	"	eval $cmd\n"
	"}\n"
	"\n"
	"## VFS and Chan I/O\n"
	"### Dispatchers\n"
	"proc ::vfs::cvfs::vfshandler {hashkey subcmd args} {\n"
	"	set cmd $args\n"
	"	set cmd [linsert $cmd 0 \"::vfs::cvfs::vfsop_${subcmd}\" $hashkey]\n"
	"\n"
	"	return [eval $cmd]\n"
	"}\n"
	"\n"
	"proc ::vfs::cvfs::chanhandler {hashkey subcmd args} {\n"
	"	set cmd $args\n"
	"	set cmd [linsert $cmd 0 \"::vfs::cvfs::chanop_${subcmd}\" $hashkey]\n"
	"\n"
	"	return [eval $cmd]\n"
	"}\n"
	"\n"
	"### Actual handlers\n"
	"#### Channel operation handlers\n"
	"proc ::vfs::cvfs::chanop_initialize {hashkey chanId mode} {\n"
	"	return [list initialize finalize watch read seek]\n"
	"}\n"
	"\n"
	"proc ::vfs::cvfs::chanop_finalize {hashkey chanId} {\n"
	"	unset -nocomplain ::vfs::cvfs::chandata([list $hashkey $chanId])\n"
	"\n"
	"	return\n"
	"}\n"
	"\n"
	"proc ::vfs::cvfs::chanop_watch {hashkey chanId eventSpec} {\n"
	"	array set chaninfo $::vfs::cvfs::chandata([list $hashkey $chanId])\n"
	"\n"
	"	set chaninfo(watching) $eventSpec\n"
	"\n"
	"	set ::vfs::cvfs::chandata([list $hashkey $chanId]) [array get chaninfo]\n"
	"\n"
	"	if {[lsearch -exact $chaninfo(watching) \"read\"] != -1} {\n"
	"		after 0 [list catch \"chan postevent $chanId [list {read}]\"]\n"
	"	}\n"
	"\n"
	"	return\n"
	"}\n"
	"\n"
	"proc ::vfs::cvfs::chanop_read {hashkey chanId bytes} {\n"
	"	array set chaninfo $::vfs::cvfs::chandata([list $hashkey $chanId])\n"
	"\n"
	"	set pos $chaninfo(pos)\n"
	"	set len $chaninfo(len)\n"
	"\n"
	"	if {[lsearch -exact $chaninfo(watching) \"read\"] != -1} {\n"
	"		after 0 [list catch \"chan postevent $chanId [list {read}]\"]\n"
	"	}\n"
	"\n"
	"	if {$pos == $len} {\n"
	"		return \"\"\n"
	"	}\n"
	"\n"
	"	set end [expr {$pos + $bytes}]\n"
	"	if {$end > $len} {\n"
	"		set end $len\n"
	"	}\n"
	"\n"
	"	set data [::vfs::cvfs::data::getData $hashkey $chaninfo(file) $pos $end]\n"
	"\n"
	"	set dataLen [string length $data]\n"
	"	incr pos $dataLen\n"
	"\n"
	"	set chaninfo(pos) $pos\n"
	"\n"
	"	set ::vfs::cvfs::chandata([list $hashkey $chanId]) [array get chaninfo]\n"
	"\n"
	"	return $data\n"
	"}\n"
	"\n"
	"proc ::vfs::cvfs::chanop_seek {hashkey chanId offset origin} {\n"
	"	array set chaninfo $::vfs::cvfs::chandata([list $hashkey $chanId])\n"
	"\n"
	"	set pos $chaninfo(pos)\n"
	"	set len $chaninfo(len)\n"
	"\n"
	"	switch -- $origin {\n"
	"		\"start\" - \"0\" {\n"
	"			set pos $offset\n"
	"		}\n"
	"		\"current\" - \"1\" {\n"
	"			set pos [expr {$pos + $offset}]\n"
	"		}\n"
	"		\"end\" - \"2\" {\n"
	"			set pos [expr {$len + $offset}]\n"
	"		}\n"
	"	}\n"
	"\n"
	"	if {$pos < 0} {\n"
	"		set pos 0\n"
	"	}\n"
	"\n"
	"	if {$pos > $len} {\n"
	"		set pos $len\n"
	"	}\n"
	"\n"
	"	set chaninfo(pos) $pos\n"
	"	set ::vfs::cvfs::chandata([list $hashkey $chanId]) [array get chaninfo]\n"
	"\n"
	"	return $pos\n"
	"}\n"
	"\n"
	"#### VFS operation handlers\n"
	"proc ::vfs::cvfs::vfsop_stat {hashkey root relative actualpath} {\n"
	"	catch {\n"
	"		set ret [::vfs::cvfs::data::getMetadata $hashkey $relative]\n"
	"	}\n"
	"\n"
	"	if {![info exists ret]} {\n"
	"		vfs::filesystem posixerror $::vfs::posix(ENOENT)\n"
	"	}\n"
	"\n"
	"	return $ret\n"
	"}\n"
	"\n"
	"proc ::vfs::cvfs::vfsop_access {hashkey root relative actualpath mode} {\n"
	"	set ret [::vfs::cvfs::data::getMetadata $hashkey $relative]\n"
	"\n"
	"	if {$mode & 0x2} {\n"
	"		vfs::filesystem posixerror $::vfs::posix(EROFS)\n"
	"	}\n"
	"\n"
	"	return 1\n"
	"}\n"
	"\n"
	"proc ::vfs::cvfs::vfsop_matchindirectory {hashkey root relative actualpath pattern types} {\n"
	"	set ret [list]\n"
	"\n"
	"	catch {\n"
	"		array set metadata [::vfs::cvfs::data::getMetadata $hashkey $relative]\n"
	"	}\n"
	"\n"
	"	if {![info exists metadata]} {\n"
	"		return [list]\n"
	"	}\n"
	"\n"
	"	if {$pattern == \"\"} {\n"
	"		set children [list $relative]\n"
	"	} else {\n"
	"		set children [::vfs::cvfs::data::getChildren $hashkey $relative]\n"
	"	}\n"
	"\n"
	"	foreach child $children {\n"
	"		if {$pattern != \"\"} {\n"
	"			if {![string match $pattern $child]} {\n"
	"				continue\n"
	"			}\n"
	"		}\n"
	"\n"
	"		unset -nocomplain metadata\n"
	"		catch {\n"
	"			array set metadata [::vfs::cvfs::data::getMetadata $hashkey $child]\n"
	"		}\n"
	"\n"
	"		if {[string index $root end] == \"/\"} {\n"
	"			set child \"${root}${child}\"\n"
	"		} else {\n"
	"			set child \"${root}/${child}\"\n"
	"		}\n"
	"		if {[string index $child end] == \"/\"} {\n"
	"			set child [string range $child 0 end-1]\n"
	"		}\n"
	"\n"
	"		if {![info exists metadata(type)]} {\n"
	"			continue\n"
	"		}\n"
	"\n"
	"		set filetype 0\n"
	"		switch -- $metadata(type) {\n"
	"			\"directory\" {\n"
	"				set filetype [expr {$filetype | 0x04}]\n"
	"			}\n"
	"			\"file\" {\n"
	"				set filetype [expr {$filetype | 0x10}]\n"
	"			}\n"
	"			\"link\" {\n"
	"				set filetype [expr {$filetype | 0x20}]\n"
	"			}\n"
	"			default {\n"
	"				continue\n"
	"			}\n"
	"		}\n"
	"\n"
	"		if {($filetype & $types) != $types} {\n"
	"			continue\n"
	"		}\n"
	"\n"
	"		lappend ret $child\n"
	"	}\n"
	"\n"
	"	return $ret\n"
	"}\n"
	"\n"
	"proc ::vfs::cvfs::vfsop_fileattributes {hashkey root relative actualpath {index -1} {value \"\"}} {\n"
	"	set attrs [list -owner -group -permissions]\n"
	"\n"
	"	if {$value != \"\"} {\n"
	"		vfs::filesystem posixerror $::vfs::posix(EROFS)\n"
	"	}\n"
	"\n"
	"	if {$index == -1} {\n"
	"		return $attrs\n"
	"	}\n"
	"\n"
	"	array set metadata [::vfs::cvfs::data::getMetadata $hashkey $relative]\n"
	"\n"
	"	set attr [lindex $attrs $index]\n"
	"\n"
	"	switch -- $attr {\n"
	"		\"-owner\" {\n"
	"			return $metadata(uid)\n"
	"		}\n"
	"		\"-group\" {\n"
	"			return $metadata(gid)\n"
	"		}\n"
	"		\"-permissions\" {\n"
	"			if {$metadata(type) == \"directory\"} {\n"
	"				set metadata(mode) [expr {$metadata(mode) | 040000}]\n"
	"			}\n"
	"\n"
	"			return [format {0%o} $metadata(mode)]\n"
	"		}\n"
	"	}\n"
	"\n"
	"	return -code error \"Invalid index\"\n"
	"}\n"
	"\n"
	"proc ::vfs::cvfs::vfsop_open {hashkey root relative actualpath mode permissions} {\n"
	"	if {$mode != \"\" && $mode != \"r\"} {\n"
	"		vfs::filesystem posixerror $::vfs::posix(EROFS)\n"
	"	}\n"
	"\n"
	"	catch {\n"
	"		array set metadata [::vfs::cvfs::data::getMetadata $hashkey $relative]\n"
	"	}\n"
	"\n"
	"	if {![info exists metadata]} {\n"
	"		vfs::filesystem posixerror $::vfs::posix(ENOENT)\n"
	"	}\n"
	"\n"
	"	if {$metadata(type) == \"directory\"} {\n"
	"		vfs::filesystem posixerror $::vfs::posix(EISDIR)\n"
	"	}\n"
	"\n"
	"	if {[info command chan] != \"\"} {\n"
	"		set chan [chan create [list \"read\"] [list ::vfs::cvfs::chanhandler $hashkey]]\n"
	"\n"
	"		set ::vfs::cvfs::chandata([list $hashkey $chan]) [list file $relative pos 0 len $metadata(size) watching \"\"]\n"
	"\n"
	"		return [list $chan]\n"
	"	}\n"
	"\n"
	"	if {[info command rechan] == \"\"} {\n"
	"		catch {\n"
	"			package require rechan\n"
	"		}\n"
	"	}\n"
	"\n"
	"	if {[info command rechan] != \"\"} {\n"
	"		set chan [rechan [list ::vfs::cvfs::chanhandler $hashkey] 2]\n"
	"\n"
	"		set ::vfs::cvfs::chandata([list $hashkey $chan]) [list file $relative pos 0 len $metadata(size) watching \"\"]\n"
	"\n"
	"		return [list $chan]\n"
	"	}\n"
	"\n"
	"	return -code error \"No way to generate a channel, need either Tcl 8.5+, \\\"rechan\\\"\"\n"
	"}\n"
	"\n"
	"##### No-Ops since we are a readonly filesystem\n"
	"proc ::vfs::cvfs::vfsop_createdirectory {args} {\n"
	"	vfs::filesystem posixerror $::vfs::posix(EROFS)\n"
	"}\n"
	"proc ::vfs::cvfs::vfsop_deletefile {args} {\n"
	"	vfs::filesystem posixerror $::vfs::posix(EROFS)\n"
	"}\n"
	"proc ::vfs::cvfs::vfsop_removedirectory {args} {\n"
	"	vfs::filesystem posixerror $::vfs::posix(EROFS)\n"
	"}\n"
	"proc ::vfs::cvfs::vfsop_utime {} {\n"
	"	vfs::filesystem posixerror $::vfs::posix(EROFS)\n"
	"}\n"
	"\n"
	"package provide vfs::cvfs 1.0\n"
