/*
 *  ds-server
 *    \_ Handle I/O requests.
 *        \_ Listen on a TCP port
 *        \_ Search
 */


#include <netinet/ip.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <sys/wait.h>
#include <sys/poll.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include "directshare.h"
#include "ds-errors.h"
#include "dendian.h"
#include "net.h"
#include "crc.h"
#include "strsep.h"

#define ARRAY_TO_INT32(x,y) ((x[y]<<24)|(x[y+1]<<16)|(x[y+2]<<8)|x[y+3])

#define DS_NUM_CMDS 5
struct directshare_cmdinfo ds_commands[DS_NUM_CMDS]={
	{DS_CMD_HELO, ds_cmd_helo, 0, 0},
	{DS_CMD_SERV, ds_cmd_serv, 1, 0},
	{DS_CMD_ADDR, ds_cmd_addr, 0, 0},
	{DS_CMD_FILE, ds_cmd_file, 0, 0},
	{DS_CMD_OKAY, ds_cmd_okay, 0, 0},
	{DS_CMD_QUIT, ds_cmd_quit, 0, 0},
};



struct directshare_connstat *conninfo[4096];
char init=0;
int master_fd=-1;
int curr_fd=-1;

int main(void);


void ds_shutdown(void) {
	int i;

	for (i=255;i>=3;i--) {
		if (i!=master_fd) close(i);
	}
	close(master_fd);
}

void ds_sighandler(int sig) {
	switch (sig) {
		case SIGINT:
			ds_shutdown();
			exit(DIRECTSHARE_OK);
			break;
		case SIGFPE:
		case SIGSEGV:
			write(curr_fd, "EROR 1 :Your command has caused an internal software error.  Do not repeat it.  A bug report has been filed.\n", 109);
			main();
			exit(DIRECTSHARE_ERROR);
		case SIGPIPE:
			PRINTERR("Sigpipe recieved.");
			break;
	}
}


/*
 * Format of a message:
 *     Line terminated ('\n'), words seperated by spaces.
 */

void ds_handledata(int sock_fd, char *data, int datalen) {
	unsigned long cmdhash;
	unsigned int x, wbuflen, cmdcnt;
	int offset=0, retval=1;
	char *dbuf, *wbuf, *cbuf, *cbuf_s, *cmdlist[10];

	if (!init) {
		PRINTERR("Not initialized!");
		return;
	}
	if (conninfo[sock_fd]==NULL) {
		PRINTERR("Not initialized!");
		return;
	}

	curr_fd=sock_fd;

	if (conninfo[sock_fd]->buf==NULL) {
		conninfo[sock_fd]->bufsize=DIRECTSHARE_BUF_SIZE;
		conninfo[sock_fd]->buf=malloc(DIRECTSHARE_BUF_SIZE);
	}

	dbuf=conninfo[sock_fd]->buf;

	if ((datalen+1+conninfo[sock_fd]->buflen)<=conninfo[sock_fd]->bufsize) {
		memcpy(dbuf+conninfo[sock_fd]->buflen,data,datalen);
		datalen=conninfo[sock_fd]->buflen+=(datalen-offset);
		dbuf[datalen]='\0';
	} else {
		PRINTERR("Buffer would fill, dropping. [datalen=%i][buflen=%i]",datalen,conninfo[sock_fd]->buflen);
	}


	offset=0;
/*
 * The work begins here.  Need to set offset to the place where we read up to.
 * Should read all available units otherwise messages could overflow buffer
 * fairly quickly.
 */

	while ((wbuf=strchr(dbuf+offset, '\n')) && offset!=conninfo[sock_fd]->buflen) {
		wbuflen=wbuf-dbuf-offset;
		cbuf_s=cbuf=malloc(wbuflen+1);
		memcpy(cbuf, dbuf+offset, wbuflen);
		offset+=(wbuflen+1);
		while (cbuf[wbuflen]<32 && wbuflen>0) { cbuf[wbuflen--]='\0'; }
		
/* Now we have our working buffer (cbuf), do something */

		for (x=0;x<(sizeof(cmdlist)/sizeof(char *));x++) {
			if (cbuf==NULL) break;
			cmdlist[x]=ds_strsep(&cbuf, " \t");

			if (cmdlist[x]==NULL) break;
			if (cmdlist[x][0]=='\0') x--;
			if (cbuf!=NULL) {
				if (cbuf[0]==':') {
					x++;
					cmdlist[x]=cbuf+1;
					x++;
					break;
				}
			}
		}
		cmdcnt=x;
/* This could be optimized by not calling strlen everytime... */
		for (x=0;x<strlen(cmdlist[0]);x++) {
			cmdlist[0][x]=toupper(cmdlist[0][x]);
		}

/* We are now ready to take on the world. */

		x=0;
		cmdhash=hash_fourbyte(cmdlist[0], '\0');

		for (x=0;x<DS_NUM_CMDS;x++) {
			if (cmdhash==ds_commands[x].hash) {
				retval=ds_commands[x].func(sock_fd, conninfo[sock_fd], cmdcnt, &cmdlist);
				break;
			}
		}

		if (retval>0 && retval<128) {
			if ((retval)<=(sizeof(directshare_errors)/sizeof(char *))) {
				ds_printf(sock_fd, "EROR %i: %s\n", retval, directshare_errors[retval-1] );
			} else {
				ds_printf(sock_fd, "EROR %i: Unknown error\n", retval);
			}
		} else {
			if (retval<128) {
				ds_printf(sock_fd, "OKAY\n");
			} else {
				switch (retval-128) {
					case DS_RET_QUIT:
						/* Close the connection */
						ds_printf(sock_fd, "GBYE\n");
						close(sock_fd);
						break;
				}
			}
		}


		free(cbuf_s);
		if (offset>conninfo[sock_fd]->buflen) { offset=conninfo[sock_fd]->buflen; break; }
	}

	conninfo[sock_fd]->buflen-=offset;

	if (conninfo[sock_fd]->buflen>0) {
		dbuf=memmove(dbuf,dbuf+offset, conninfo[sock_fd]->buflen);
	} else { 
		conninfo[sock_fd]->buflen=0;
	}

	curr_fd=-1;

	return;
}

void ds_init(int sock_fd, char *host) {
	int x;

	if (!init) {
		init=1;
		for (x=0;x<4096;x++) conninfo[x]=NULL;
	}
	/* Init */
	conninfo[sock_fd]=malloc(sizeof(struct directshare_connstat));
	conninfo[sock_fd]->state=DS_CMD_HELO;
	conninfo[sock_fd]->type=DIRECTSHARE_TYPE_UNKNOWN;
	conninfo[sock_fd]->buf=NULL;
	conninfo[sock_fd]->buflen=0;
	strncpy(conninfo[sock_fd]->host,host,16);
	PRINTERR("Initialized socket (%i) [%s].", sock_fd, conninfo[sock_fd]->host);
	write(sock_fd, "HELO hostname hostid sock_fd +server :Version\n", 46);
	return;
}

void ds_handleclose(int sock_fd) {
	int x;

	if (!init) {
		init=1;
		for (x=0;x<4096;x++) conninfo[x]=NULL;
	}

	if (sock_fd<0) {
		PRINTERR("Attempted to close a negative fd.");
		return;
	}

	/* Destroy */
	if (conninfo[sock_fd]->buf!=NULL) free(conninfo[sock_fd]->buf);
	free(conninfo[sock_fd]);
	conninfo[sock_fd]=NULL;
	close(sock_fd);
	PRINTERR("Destroyed socket (%i).", sock_fd);
	return;
}


int main(void) {
	static struct pollfd clients[50];
	static unsigned int clientcnt=0;
	struct sockaddr_in peer;
	int read_len, peerlen=sizeof(peer);
	int sock_fd, read_fd;
	int sockflags;
	char buf[1024];
	int i;

#ifndef DEBUG
	setsid();
	for (i=0;i<10;i++) close(i);
#endif

	if (master_fd!=-1) {
#ifdef DEBUG
		write(STDERR_FILENO, "*** RECOVERING FROM CRASH ***\n", 30); 
#endif
	} else {
		printf("DirectShare Server v%i.%i.%i-%s\n", DIRECTSHARE_VER_MAJ, DIRECTSHARE_VER_MIN, DIRECTSHARE_VER_REV, DIRECTSHARE_VER_INFO);
		while ((master_fd=createlisten(DIRECTSHARE_TCP_PORT))<0) {
			PERROR("createlisten");
			sleep(1);
		}
	}

	fcntl(master_fd, F_SETOWN, getpid());

	signal(SIGINT, ds_sighandler);
	signal(SIGFPE, ds_sighandler);
	signal(SIGPIPE, ds_sighandler);
	signal(SIGSEGV, ds_sighandler);

	clients[0].fd=master_fd;
	clients[0].events=POLLIN;
	clientcnt++;

	while (1) {
		read_fd=poll(clients,clientcnt,-1);

		for (i=1;i<clientcnt;i++) {
			if (clients[i].revents!=0) {
				sock_fd=clients[i].fd;
				read_len=read(sock_fd, &buf, 1024);
				if (read_len<0) {
					PERROR("read");
				}
				if (read_len<=0) {
					memmove(&clients[i], &clients[i+1], sizeof(struct pollfd) * (clientcnt-i));
					clientcnt--;
					i--;
					ds_handleclose(sock_fd);
					continue;
				}
				ds_handledata(sock_fd, buf, read_len);
			}
		}


		if (clients[0].revents!=0) {
			if ((sock_fd=accept(master_fd, (struct sockaddr *) &peer, &peerlen))>=0) {
				if (clientcnt>=sizeof(clients)/sizeof(struct pollfd)) {
					close(sock_fd);
					continue;
				}
				clients[clientcnt].fd=sock_fd;
				clients[clientcnt].events=POLLIN;
				clientcnt++;
				sockflags=fcntl(sock_fd, F_GETFL);
				fcntl(sock_fd, F_SETFL, sockflags|O_NONBLOCK);
				ds_init(sock_fd, inet_ntoa(peer.sin_addr));
			} else {
				PERROR("accept");
			}
		}
	}
	return(DIRECTSHARE_OK);
}



int ds_cmd_helo(int sock_fd, struct directshare_connstat *sinfo, int argc, char **argv) {
	return(0);
}

int ds_cmd_serv(int sock_fd, struct directshare_connstat *sinfo, int argc, char **argv) {
	return(0);
}

int ds_cmd_addr(int sock_fd, struct directshare_connstat *sinfo, int argc, char **argv) {
	return(0);
}

int ds_cmd_file(int sock_fd, struct directshare_connstat *sinfo, int argc, char **argv) {
	return(0);
}

int ds_cmd_quit(int sock_fd, struct directshare_connstat *sinfo, int argc, char **argv) {
}

int ds_cmd_okay(int sock_fd, struct directshare_connstat *sinfo, int argc, char **argv) {
	return(255);
}
