/*
 * Copyright (C) 2001  Roy Keene
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *      email: dact@rkeene.org
 */

#define __DACT_C
#include <fcntl.h>
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/types.h>
#include "dact.h"
#include "parse.h"
#include "dendian.h"
#include "crc.h"
#include "math.h"
#include "algorithms.h"
#include "ciphers.h"
#include "module.h"
#include "header.h"
#include "parse.h"
#include "net.h"
#include "ui.h"
#ifdef HAVE_ZLIB_H
#include <zlib.h>
#endif
#ifdef HAVE_BZLIB_H
#include <bzlib.h>
#endif


int print_help(int argc, char **argv);
void dact_loadconfig(const char *path, char *options);
int dact_upgrade(const char *options, uint32_t *crcs);
uint32_t is_latest (const char *options);
int dact_shutdown(int retval);
char *dact_getoutfilename(const char *orig, const int mode);
uint32_t dact_process_other(int src, const int dest, const uint32_t magic, const char *options);
uint32_t dact_blk_decompress(char *ret, const char *srcbuf, const uint32_t size, const char *options, const int algo);
uint32_t dact_blk_compress(char *algo, char *ret, const char *srcbuf, const uint32_t size, const char *options);
uint32_t dact_process_file(const int src, const int dest, const int mode, const char *options, const char *filename, uint32_t *crcs, int cipher);
int main(int argc, char **argv);

extern char *optarg;
extern int optind, opterr, optopt;
uint32_t DACT_BLK_SIZE=0;

int print_help(int argc, char **argv) {
	printf("DACT %i.%i.%i-%s by Netfuel Design. <dact@rkeene.org>\n", DACT_VER_MAJOR, DACT_VER_MINOR, DACT_VER_REVISION, DACT_VER_SUB);
	printf("usage: %s [options...] [file ...]\n",argv[0]);

	printf("Options:\n");
	printf("  -d          Decompress instead of compressing.\n");
	printf("  -s          Give statistics rather than compress or decompress.\n");
	printf("  -f          Force unsafe things to happen.\n");
	printf("  -c          (De)compress to stdout.\n");
	printf("  -v          Increase verbosity.\n");
	printf("  -l          List available algorithms.\n");
	printf("  -n          Toggle use of CRCs.\n");
#ifndef DACT_UNSAFE
	printf("  -C          Complain when compression errors occur.\n");
#endif
	printf("  -H          Write only header (no data).\n");
	printf("  -O          Toggle writing original file name.\n");
	printf("  -S          Use speed-size as a metric rather than size.\n");
	printf("  -h          Give this help.\n");
	printf("  -V          Display DACT version (%i.%i.%i-%s).\n", DACT_VER_MAJOR, DACT_VER_MINOR, DACT_VER_REVISION, DACT_VER_SUB);
	printf("  -N          Upgrade DACT.\n");
	printf("  -b NN       Use a block size of NN bytes for compression.\n");
	printf("  -e NN       Exclude algorithm NN from being used.\n");
	printf("  -m CONF     Load config file CONF.\n");
	printf("  -u URL      Specify download location as URL.\n");
	printf("  -p URL      Parse URL and print results, then exit.\n");
	printf("  -D DESC     Specify a description of DESC.\n");
	printf("  -I NN       Use ONLY 2 algorithms, NN and 0.\n");
	printf("  -U FILE     Use FILE to select download location.\n");
	printf("  -E CIPHER   Use CIPHER to encrypt data (LIST  lists available ciphers.)\n");
	printf("  file...     File(s) to (de)compress.  (If none given, use standard input).\n");
	return(0);
}

int dact_blksize_calc(int fsize) {
	if (fsize==0) return(DACT_BLK_SIZE_DEF);
	if (fsize<(204800)) {
		return(fsize);
	}
	return(((int) ((((float) fsize)/102400.0)-(0.9999999)))*65535);
}

void dact_loadconfig(const char *path, char *options) {
	char *line=NULL, *line_s, *item_buf[4]={NULL, NULL, NULL, NULL};
	int i;
	FILE *cfd;

	line_s=line=malloc(512);
	if ((cfd=fopen(path,"r"))==NULL) return;
	while (!feof(cfd)) {
		line=line_s;
		fgets(line, 511, cfd);
		if (line[0]=='#') continue;
		i=(strlen(line)-1);
		if (line[i]=='\n') line[i]=0;
		for (i=0;i<4;i++) item_buf[i]=NULL;
		for (i=0;i<4;i++) {
			if ((item_buf[i]=strsep(&line, "\t "))==NULL)  break;
			if (item_buf[i][0]==0) i--;
		}
		if (item_buf[0]==NULL || item_buf[1]==NULL) continue;

		switch (ELFCRC(0, item_buf[0], strlen(item_buf[0]))) {
			case 164209419: /* binary_check */
				options[DACT_OPT_BINCHK]=!!strcmp(item_buf[1],"off");
				break;
			case 9456603: /* version_check */
				options[DACT_OPT_VERCHK]=!!strcmp(item_buf[1],"off");
				break;
			case 204349618: /* module_dir */
				strncpy(moduledirectory,item_buf[1],127);
				break;
			case 106360197: /* load_module */
				init_modules();
				load_module(item_buf[1]);
				break;
			case 164097267: /* network_access */
#ifndef NO_NETWORK
				if (!strcmp(item_buf[1],"off")) {
					/* This should do something XXX */
				}
#endif
				break;
			case 168825941: /* block_size */
				DACT_BLK_SIZE=atoi2(item_buf[1]);
				break;
			case 162975987: /* use_urls */
				options[DACT_OPT_URL]=!!strcmp(item_buf[1],"off");
				break;
			case 104235033: /* color_ui */
				dact_ui_setopt(DACT_UI_OPT_COLOR,!!strcmp(item_buf[1],"off"));
				break;
#ifdef DEBUG
			default:
				fprintf(stderr, "Unknown command %s (%i)\n",item_buf[0],ELFCRC(0,item_buf[0],strlen(item_buf[0])));
				break;
#endif
		}

	}
	free(line_s);
	fclose(cfd);
}


int dact_upgrade(const char *options, uint32_t *crcs) {
	char *urlsubst, *buf;
	char dact_binfile[256];
	int inFd, outFd;
	uint32_t i;


#ifdef DACT_DEBIAN_UPGRADE_PROC
	int status=0;

	if (getuid()==0) {
		buf=dact_ui_getuserinput("Executing `apt-get update' okay [y/N]? ", 5, 0);
		if (toupper(buf[0])!='Y') {
			PRINTERR("Failed to upgrade DACT.");
			free(buf);
			return(-1);
		}
		free(buf);

		if (fork()==0) {
			execl("/usr/bin/apt-get","apt-get","update",NULL);
			return(-1);  /* Couldn't run binary. */
		} else {
			i=wait(&status);
			if (WIFEXITED(status)) {
				if (WEXITSTATUS(status)) {
					PRINTERR("Failed to run `apt-get update'");
					return(-1);
				}
			}
		}

		if (fork()==0) {
			execl("/usr/bin/apt-get","apt-get","install","dact",NULL);
			return(-1);  /* Couldn't run binary. */
		} else {
			i=wait(&status);
			if (WIFEXITED(status)) {
				if (WEXITSTATUS(status)) {
					PRINTERR("Failed to run `apt-get install dact'");
					return(-1);
				}
			}
		}

		return(1);
	}
#endif

	if ((i=is_latest(options))) {
		PRINTERR("**+");
		PRINT_LINE; fprintf(stderr, "dact: **> CURR: DACT %i.%i.%i\n",DACT_VER_MAJOR, DACT_VER_MINOR, DACT_VER_REVISION);
		PRINT_LINE; fprintf(stderr, "dact: **> NEW:  DACT %i.%i.%i\n",i>>16,(i>>8)&0xff,i&0xff);
		PRINTERR("**>");
		PRINTERR("**-");
	}
	if ((inFd=open_net("http://www.rkeene.org/projects/rget/rget.cgi?project=dact&file=info", O_RDONLY))>=0) {
		fprintf(stderr, "------------------------\n");
		buf=malloc(1024);
		while (1) {
			i=read_f(inFd, buf, 1024);
			write(STDERR_FILENO, buf, i);
			if (i!=1024) break;
		}
		fprintf(stderr, "------------------------\n");
		close(inFd);
		free(buf);
	}

#ifdef GO32
	mkdir("c:/dact/", 0755);
	strcpy(dact_binfile,"c:/dact/dact.exe");
#else
	strncpy(dact_binfile,getenv("HOME"),sizeof(dact_binfile)-1);
	strncat(dact_binfile,"/.dact/",sizeof(dact_binfile)-strlen(dact_binfile)-1);
	mkdir(dact_binfile, 0755);
	strncat(dact_binfile,"dact.bin",sizeof(dact_binfile)-strlen(dact_binfile)-1);
#endif
	urlsubst=parse_url_subst("http://www.rkeene.org/projects/rget/rget.cgi?os=@@OSNM@@&arch=@@ARCH@@&project=dact&file=bin&meth=gz","");
	if ((outFd=open_net(dact_binfile, O_WRONLY|O_TRUNC|O_CREAT, 0755))<0) { PERROR("open_net"); return(-1); }
	if ((inFd=open_net(urlsubst, O_RDONLY))<0) { PERROR("open_net"); return(-1); }
	if (!dact_process_file(inFd, outFd, DACT_MODE_DECMP, options, "dact", crcs,-1)) {
		close(inFd);
		close(outFd);
		unlink(dact_binfile);
		PRINTERR("Failed to upgrade DACT.");
		return(-1);
	}
	close(inFd);
	close(outFd);
	if (!options[DACT_OPT_BINCHK]) {
		PRINTERR("Note: You do not have binary_check  set to `on' in your dact.conf.");
	}
	return(1);
}

uint32_t is_latest (const char *options) {
#if defined(CHECK_VERSION)
	int fd;
	char ver_maj[4]={0,0,0,0}, ver_min[4]={0,0,0,0}, ver_rev[4]={0,0,0,0};
	char bigbuf[1024];
	int vers[3];

	if (options[DACT_OPT_VERCHK]==0) return(0);
	if (getuid()==0) return(0);

	if ((fd=createconnection_tcp("www.rkeene.org", 80))<0) return(0);

	write(fd, "GET http://www.rkeene.org/devel/dact/VERSION\n", 45);
	read(fd, &bigbuf,1024);

	memcpy(ver_maj,bigbuf,3);
	memcpy(ver_min,bigbuf+3,3);
	memcpy(ver_rev,bigbuf+6,3);

	closeconnection(fd);

	vers[0]=atoi(ver_maj);
	vers[1]=atoi(ver_min);
	vers[2]=atoi(ver_rev);

	if ( ((vers[0]<<16)|(vers[1]<<8)|vers[2]) > ((DACT_VER_MAJOR<<16)|(DACT_VER_MINOR<<8)|DACT_VER_REVISION) ) {
		return((vers[0]<<16)|(vers[1]<<8)|vers[2]);
	} else {
		return(0);
	}
#else
	return(0);
#endif
}


int dact_shutdown(int retval) {
	unload_modules();
	dact_ui_deinit();
	return(retval);
}

char *dact_getoutfilename(const char *orig, const int mode) {
	char *ret=NULL;
	int x=0;

	switch (mode) {
		case DACT_MODE_COMPR:
			ret=malloc(strlen(orig)+5);
#ifdef GO32
			strncpy(ret,orig,8);
			for (x=0;x<strlen(ret);x++) {
				if (ret[x]=='.') ret[x]='_';
			}
			strcat(ret,".dct");
#else
			strcpy(ret,orig);
			strcat(ret,".dct");
#endif
			break;
		case DACT_MODE_DECMP:
			if (strcmp(&orig[strlen(orig)-4],".dct") && \
				strcmp(&orig[strlen(orig)-4], ".bz2") && \
				strcmp(&orig[strlen(orig)-3], ".gz")) {
				return(NULL);
			}
/* XXX: I wonder if this breaks easily... */
			x=(strrchr(orig,'.')-orig);
			ret=malloc(x+1);
			strncpy(ret,orig,x);
			ret[x]=0;
			break;
		case DACT_MODE_STAT:
			return(NULL);
			break;
	}
	return(ret);
}

uint32_t dact_process_other(int src, const int dest, const uint32_t magic, const char *options) {
	char *buf, tmpbuf[128]="/tmp/dactXXXXXX";
	uint32_t filesize=0, x;
	int tmpfd=0;
#if defined(HAVE_LIBBZ2) && (defined(HAVE_BZDOPEN) || defined(HAVE_NEW_BZDOPEN))
	BZFILE *bzfd;
#endif
#if defined(HAVE_LIBZ) && defined(HAVE_GZDOPEN)
	gzFile gzfd;
#endif


	filesize=0;  /* Fix a warning, that is all. */


/*
 * bad and broke stuff XXX FIXME XXX FIXME XXX FIXME
 * There has to be a better way to do this... I just want
 * to rewind my socket/pipe 4 bytes... 4 bytes is all I ask
 * Is that so much to ask for?  Well?!  Is it?!  I don't
 * think it is.
 * I give up on this.
 *
 * Someone please fix it.
 *  -- Roy Keene <dact@rkeene.org>
 */
	if (lseek_net(src, 0, SEEK_SET)<0) {
		PRINTERR("This should not be happening.");
/*
 * lseek_net() should make this obsolete.
 *  ... EXCEPT! when reading from stdin.
 *
 */
		tmpfd=mkstemp(tmpbuf);
		write_de(tmpfd, magic, 4);
		buf=malloc(1024);
		while (1) {
			x=read_f(src, buf, 1024);
			write(tmpfd, buf, x);
			if (x<1024) break;
		}
		close(src);
		src=tmpfd;
		lseek_net(src, 0, SEEK_SET); /* Now bitch. */
		free(buf);
	}
#if defined(HAVE_LIBZ) && defined(HAVE_GZDOPEN)


	if ((magic&0xffff0000)==0x1f8b0000) { /* gzip */
		dact_ui_status(DACT_UI_LVL_GEN, "Gunzipping...");
		buf=malloc(1024);

		gzfd=gzdopen(src, "r");
/*XXX: need to dact_ui_setup() */
		while (1) {
			dact_ui_incrblkcnt(1);
			x=gzread(gzfd,buf,1024);
			filesize+=write(dest, buf, x);
			if (x<1024) break;
		}
		free(buf);
		if (tmpfd!=0) unlink(tmpbuf);
		return(filesize);
	}
#endif

#if defined(HAVE_LIBBZ2) && (defined(HAVE_BZDOPEN) || defined(HAVE_NEW_BZDOPEN))
	if ((magic&0xffffff00)==0x425a6800) { /* bzip2 */ 
		dact_ui_status(DACT_UI_LVL_GEN, "Bunzipping...");
		buf=malloc(1024);

#ifdef HAVE_NEW_BZDOPEN
		bzfd=BZ2_bzdopen(src, "r");
#else
		bzfd=bzdopen(src, "r");
#endif
/*XXX: need to dact_ui_setup() */
		while(1) {
			dact_ui_incrblkcnt(1);
#ifdef HAVE_NEW_BZDOPEN
			x=BZ2_bzread(bzfd, buf, 1024);
#else
			x=bzread(bzfd, buf, 1024);
#endif
			filesize+=write(dest, buf, x);
			if (x<1024) break;
		}
		free(buf);
		if (tmpfd!=0) unlink(tmpbuf);
		return(filesize);
	}
#endif
	return(0);
}


uint32_t dact_blk_decompress(char *ret, const char *srcbuf, const uint32_t size, const char *options, const int algo) {
	uint32_t retval;

	if (algo==0xff) return(0);

	if (algorithms[algo]==NULL) {
		PRINTERR("Algorithm unavailble.");
		return(0);
	}

	retval=algorithms[algo](DACT_MODE_DECMP, NULL, srcbuf, ret, size);

	return(retval);
}


uint32_t dact_blk_compress(char *algo, char *ret, const char *srcbuf, const uint32_t size, const char *options) {
	char *tmpbuf, *smallbuf=NULL;
	int i, highest_algo=0;
	char smallest_algo;
	uint32_t smallest_size=-1, x;
#ifndef DACT_UNSAFE
	char *verif_bf=NULL;
	uint32_t m;
	if ((verif_bf=malloc(size))==NULL) { PERROR("malloc"); return(0); }
#endif

	if ((tmpbuf=malloc(size*2))==NULL) { PERROR("malloc"); return(0); }

	for (i=0;i<256;i++) {
		if (algorithms[i]!=NULL && algorithms[i]!=DACT_FAILED_ALGO) highest_algo=i;
	}

	for (i=0;i<=highest_algo;i++) {
		if (algorithms[i]!=NULL && algorithms[i]!=DACT_FAILED_ALGO) {
			x=algorithms[i](DACT_MODE_COMPR, NULL, srcbuf, tmpbuf, size);
#ifndef DACT_UNSAFE
			if ((x<smallest_size || smallest_size==-1) && x!=-1) {
				m=algorithms[i](DACT_MODE_DECMP, NULL, tmpbuf, verif_bf, x);
				if (memcmp(verif_bf, srcbuf,m) || m!=size) {
					x=-1;
					if (options[DACT_OPT_COMPLN]) {
						dact_ui_status(DACT_UI_LVL_ALL, "Compression verification failed (ignoring)");
					}
				}
			}
#endif
			if ((x<smallest_size || smallest_size==-1) && x!=-1) {
				smallest_size=x;
				smallest_algo=i;
				if (smallbuf!=NULL) free(smallbuf);
				if ((smallbuf=malloc(smallest_size))==NULL) { 
					PERROR("malloc");
					free(tmpbuf);
#ifndef DACT_UNSAFE
					free(verif_bf);
#endif
					return(0);
				}
				memcpy(smallbuf, tmpbuf, smallest_size);
			}

			if (options[DACT_OPT_VERB]>2) {
				PRINT_LINE; fprintf(stderr, "dact: \033[%im----| %03i  | %-7i | %s\033[0m\n", (smallest_algo==i)*7 , i, x, algorithm_names[i]);
			}

		}
	}

	free(tmpbuf);
#ifndef DACT_UNSAFE
	free(verif_bf);
#endif
	if (smallest_size==-1) {
		return(0);
	}
	memcpy(algo, &smallest_algo, sizeof(char));
	memcpy(ret, smallbuf, smallest_size);
/* This was MISSING !  memory leak. */
	free(smallbuf);
	return(smallest_size);
}

uint32_t dact_process_file(const int src, const int dest, const int mode, const char *options, const char *filename, uint32_t *crcs, int cipher) {
	struct stat filestats;
	FILE *extd_urlfile;
	char *file_extd_urls[256];
	unsigned char algo;
	char ch;
	char *in_buf, *out_buf, *hdr_buf, *keybuf=NULL, *tmpbuf=NULL;
	char version[3]={DACT_VER_MAJOR, DACT_VER_MINOR, DACT_VER_REVISION};
	char file_opts=0;
	uint32_t bytes_read, retsize;
	uint32_t filesize=0, blk_cnt=0, file_extd_size=0, blksize=0, fileoutsize=0;
	uint32_t magic=0, file_extd_read=0, file_extd_urlcnt=0;
	int blksize_size;
	int x=0, new_fd, canlseek=0;

	fstat(src,&filestats);

	if (mode==DACT_MODE_COMPR) {
/*
 * Calculate the default block size.
 */
		if (DACT_BLK_SIZE==0) {
			DACT_BLK_SIZE=dact_blksize_calc(filestats.st_size);
		}
		if (((in_buf=malloc(DACT_BLK_SIZE))==NULL) || \
			((out_buf=malloc(DACT_BLK_SIZE*2))==NULL)) {
				PERROR("malloc");
				return(0);
		}

		dact_ui_setup(((float) (filestats.st_size/DACT_BLK_SIZE)+0.9999));
		if (cipher!=-1) {
			dact_hdr_ext_regn(DACT_HDR_CIPHER, cipher, sizeof(cipher));
			keybuf=malloc(DACT_KEY_SIZE);

			ciphers[cipher](NULL, NULL, 0, keybuf, DACT_MODE_CINIT+DACT_MODE_CENC);
			
		}
		blksize_size=BYTESIZE(DACT_BLK_SIZE);

		if (!options[DACT_OPT_ORIG] && filename!=NULL)
			dact_hdr_ext_regs(DACT_HDR_NAME, filename, strlen(filename));
		file_extd_size=(dact_hdr_ext_size()+14); /* The +14 is for crc0 and crc1 */
		write_de(dest, DACT_MAGIC_NUMBER, 4);
		write(dest, &version[0], 1);
		write(dest, &version[1], 1);
		write(dest, &version[2], 1);
		write_de(dest, 0, 4); /* Place holder for ORIG FILE SIZE */
		write_de(dest, 0, 4); /* Place holder for NUM BLOCKS */
		write_de(dest, DACT_BLK_SIZE, 4);
		write_de(dest, file_opts, 1); /* XXX: Option byte... Or not? */
		write_de(dest, file_extd_size, 4); /* Place holder for SIZEOF EXTENDED DTA */
/* Fill the header with NOPs incase we can't come back and put the CRCs */
		ch=DACT_HDR_NOP;
		for (x=0;x<file_extd_size;x++) write(dest, &ch, 1);

		if (options[DACT_OPT_VERB]>1) {
			PRINTERR("Blk | Algo | Size    | Name");
			PRINTERR("----+------+---------+---------------------------");
		}

		memset(in_buf,0,DACT_BLK_SIZE);
		while ( (bytes_read=read_f(src, in_buf, DACT_BLK_SIZE))>0) {
			filesize+=bytes_read;
			blk_cnt++;

			retsize=dact_blk_compress(&algo, out_buf, in_buf, DACT_BLK_SIZE, options);

/* CIPHER the data if an encryption algorithm is specified. */
			if (cipher!=-1) {
				tmpbuf=malloc(retsize*2);
				x=ciphers[cipher](out_buf, tmpbuf, retsize, keybuf, DACT_MODE_CENC);
				memcpy(out_buf,tmpbuf,x);
				free(tmpbuf);
			}

			if (retsize>0) {
				if (options[DACT_OPT_VERB]>1) {
					if (options[DACT_OPT_VERB]>2) {
						PRINTERR("^^^\\ /^^^^\\ /^^^^^^^\\ /^^^^^^^^^^^^^^^^^^^^^^^^^^");
					}
					PRINT_LINE; fprintf(stderr, "dact: %03i | %03i  | %-7i | %s\n",blk_cnt,algo,retsize,algorithm_names[algo]);
					if (options[DACT_OPT_VERB]>2) {
						PRINTERR("___/ \\____/ \\_______/ \\__________________________");
					}
				}


				dact_ui_incrblkcnt(1);
				dact_ui_status(DACT_UI_LVL_GEN, "Algorithm ");
				dact_ui_status_append(DACT_UI_LVL_GEN,algorithm_names[algo]);

				crcs[0]=ELFCRC(crcs[0], out_buf, retsize);
/* Do not generate a CRC of the plaintext if encrypting */
				if (cipher==-1) {
					crcs[1]=ELFCRC(crcs[1], in_buf, DACT_BLK_SIZE);
				}

				if (!options[DACT_OPT_HDONLY]) {
					write(dest, &algo, 1);
					write_de(dest, retsize, blksize_size);

					if (write(dest, out_buf, retsize)!=retsize) {
						PERROR("write");
						free(in_buf);
						free(out_buf);
						return(0);
					}
				}
			} else {
				PRINTERR("Compression resulted in 0-byte block.");
				free(in_buf);
				free(out_buf);
				return(0);
			}
			memset(in_buf,0,DACT_BLK_SIZE);
		}

		if (bytes_read<0) {
			PERROR("read");
		}

		free(in_buf);
		free(out_buf);

		if (lseek_net(dest, 7, SEEK_SET)<0) {
/* If we can't rewind the stream, put magic+fileisze */
			write_de(dest, DACT_MAGIC_PEOF, 4);
			write_de(dest, filesize, 4);
		} else {
			write_de(dest, filesize, 4);
			write_de(dest, blk_cnt, 4);
		} 

		if (lseek_net(dest, 24, SEEK_SET)>0) {
			if (!options[DACT_OPT_NOCRC]) {
				dact_hdr_ext_regn(DACT_HDR_CRC0, crcs[0], 4);
				dact_hdr_ext_regn(DACT_HDR_CRC1, crcs[1], 4);
			}
			write(dest, dact_hdr_ext_data(), dact_hdr_ext_size());
		}

		dact_hdr_ext_clear();

		return(filesize);
	}

	if (mode==DACT_MODE_DECMP) {

		dact_ui_status(DACT_UI_LVL_GEN, "Decompressing.");

		dact_hdr_ext_clear();

		read_de(src, &magic, 4);

		if (magic!=DACT_MAGIC_NUMBER) {
			dact_ui_status(DACT_UI_LVL_GEN, "Bad DACT magic, checking others...");
			return(dact_process_other(src,dest,magic,options));
		}

		read(src, &version[0], 1);
		read(src, &version[1], 1);
		read(src, &version[2], 1);
		read_de(src, &filesize, 4);
		read_de(src, &blk_cnt, 4);
		read_de(src, &DACT_BLK_SIZE, 4);
		read(src, &file_opts, 1);
		read_de(src, &file_extd_size, 4);



		while (file_extd_read<file_extd_size) {
			x=0;
			read(src, &ch, 1);
			if (ch!=DACT_HDR_NOP) read_de(src, &x, 2); 
			switch (ch) {
				case DACT_HDR_CRC0:
					read_de(src, &crcs[2], 4);
					if (crcs[4]!=0 && crcs[2]!=crcs[4]) {
						dact_ui_status(DACT_UI_LVL_GEN, "CRC error.");
						if (!options[DACT_OPT_NOCRC])
							return(0);
					}
					break;
				case DACT_HDR_CRC1:
					read_de(src, &crcs[3], 4);
					if (crcs[5]!=0 && crcs[3]!=crcs[5]) {
						dact_ui_status(DACT_UI_LVL_GEN, "CRC error.");
						if (!options[DACT_OPT_NOCRC])
							return(0);
					}
					break;
/*

XXX: Todo, make this do something...
		 		case DACT_HDR_NAME:
					break;
*/
				case DACT_HDR_URL:
					hdr_buf=malloc(x+1);
					read_f(src, hdr_buf, x);
					hdr_buf[x]=0;
					file_extd_urls[file_extd_urlcnt++]=parse_url_subst(hdr_buf,filename);
					free(hdr_buf);
					break;
				case DACT_HDR_URLFILE:
					hdr_buf=malloc(x+1);
					read_f(src, hdr_buf, x);
					hdr_buf[x]=0;
					extd_urlfile=fopen(hdr_buf, "r");
					free(hdr_buf);   /* We shouldn't need this anymore. */
					if (extd_urlfile==NULL) break;
					hdr_buf=malloc(4096);
					while (1) {
						fgets(hdr_buf, 4095, extd_urlfile);
						if (feof(extd_urlfile)) break;
						hdr_buf=strsep(&hdr_buf,"\n");
						file_extd_urls[file_extd_urlcnt++]=parse_url_subst(hdr_buf,filename);
					}
					free(hdr_buf);
					break;
				case DACT_HDR_DESC:
					hdr_buf=malloc(x+1);
					read_f(src, hdr_buf, x);
					hdr_buf[x]=0;
					fprintf(stderr, "DESC: %s\n",hdr_buf);
					free(hdr_buf);
					break;
				case DACT_HDR_CIPHER:
					read_de(src,&cipher,x);
					break;
				case DACT_HDR_NOP:
					x=-2;
					break;
				default:
					hdr_buf=malloc(x);
					read_f(src, hdr_buf, x);
					free(hdr_buf);
					break;
			}



			file_extd_read+=(x+3);
		}

		if (options[DACT_OPT_URL]) {
			for (x=0;x<file_extd_urlcnt;x++) {
				dact_ui_status(DACT_UI_LVL_GEN,"Trying to get remote url ");
				dact_ui_status_append(DACT_UI_LVL_SPEC, file_extd_urls[x]);
				if ((new_fd=open_net(file_extd_urls[x],O_RDONLY))<0) {
					dact_ui_status(DACT_UI_LVL_GEN, "Failed.");
					continue;
				}
				close(src);
				crcs[4]=crcs[2];
				crcs[5]=crcs[3];
				return(dact_process_file(new_fd, dest, mode, options, filename, crcs, cipher));
			}
		}


/* 
   XXX: Even if we don't resolve it here, we can resolve it later...
   Should we even bother to do it here if we can?

   XXX: When CAN'T we rewind a read stream?
		When it's a pipe

*/
		if (filesize==0) {
/* See if we can rewind our stream, so when we get to the end, we can come back! */
			if (lseek_net(src, 1, SEEK_SET)==1) { /* MAJOR BUG HERE! was: lseek(src,1,SEEK_SET)==0 which will always be false. */
				canlseek=1;
				lseek_net(src, -8, SEEK_END);
				read_de(src, &magic, 4);
				if (magic!=DACT_MAGIC_PEOF) {
					dact_ui_status(DACT_UI_LVL_GEN, "File is corrupt.");
					return(0);
				}
				read_de(src, &filesize, 4);
				lseek_net(src, 24+file_extd_size, SEEK_SET);
			} else {
				canlseek=0;
			}
		}


		if (((out_buf=malloc(DACT_BLK_SIZE))==NULL) ) {
				PERROR("malloc");
				return(0);
		}


		blksize_size=BYTESIZE(DACT_BLK_SIZE);

		dact_ui_setup((int)(((float) filesize/(float) DACT_BLK_SIZE)+0.9999));

		if (cipher!=-1) {
			keybuf=malloc(DACT_KEY_SIZE);
			ciphers[cipher](NULL, NULL, 0, keybuf, DACT_MODE_CINIT+DACT_MODE_CDEC);
			
		}



		while (1) {
			if (read(src, &algo, 1)==0) break;
			if (algo==0xff) break; /* 0xff is reserved for EOF */

			read_de(src, &blksize, blksize_size);

			if ((in_buf=malloc(blksize))==NULL) {
				PERROR("malloc");
				free(out_buf);
				return(0);
			}

			read_f(src, in_buf, blksize);

			crcs[0]=ELFCRC(crcs[0],in_buf,blksize);

			if (cipher!=-1) {
				tmpbuf=malloc(blksize*2);
				x=ciphers[cipher](in_buf, tmpbuf, blksize, keybuf, DACT_MODE_CDEC);
				memcpy(in_buf,tmpbuf,x);
				free(tmpbuf);
			}

/*
 * If the filesize is not specified, try to find it in the stream...
 * this is pretty stupid, because if we can't rewind OUR read stream
 * we're SOL... do we really need the filesize that badly?  I guess
 * we do to truncate the last of it...  Adding more checks in here..
 *
 * This will never get used, canlseek will be 1 only if we can sucessfully
 * seek, in which case, we have done it above.
 */
#if 0
			if (filesize==0 && canlseek) {
				read_de(src, &magic, 4);
				read_de(src, &filesize, 4);
				if (read(src, &x, 1)==0) {
					if (magic!=DACT_MAGIC_PEOF) {
						dact_ui_status(DACT_UI_LVL_GEN, "Stream is corrupt.");
						free(in_buf);
						free(out_buf);
						return(0);
					}
				} else {
					lseek(src, -9, SEEK_CUR);
					filesize=0;
				}
			}
#endif


			if ((bytes_read=dact_blk_decompress(out_buf, in_buf, blksize, 0, algo))==0) {
				if (cipher!=-1) {
					PRINTERR("Decompression resulted in 0-byte block.  Invalid key?");
				} else {
					PRINTERR("Decompression resulted in 0-byte block.");
				}
			}
			fileoutsize+=bytes_read;

/* If ciphering, don't try to calculate this CRC. */
			if (cipher==-1) {
				crcs[1]=ELFCRC(crcs[1],out_buf,bytes_read);
			}

			dact_ui_incrblkcnt(1);

			if (fileoutsize>filesize && filesize!=0) {
				write(dest, out_buf, DACT_BLK_SIZE-(fileoutsize-filesize));
			} else {
				write(dest, out_buf, bytes_read);
			}



			free(in_buf);

			
		}

		free(out_buf);

		if ((crcs[0]!=crcs[2] && crcs[0]!=0 && crcs[2]!=0) \
		  || (crcs[1]!=crcs[3] && crcs[1]!=0 && crcs[3]!=0)) {
			dact_ui_status(DACT_UI_LVL_GEN, "CRC error.");
			if (!options[DACT_OPT_NOCRC] || options[DACT_OPT_FORCE]<1)
				return(0);
		}

		dact_hdr_ext_clear();

		return(filesize);

	}


	if (mode==DACT_MODE_STAT) {
		read_de(src, &magic, 4);
		read(src, &version[0], 1);
		read(src, &version[1], 1);
		read(src, &version[2], 1);
		read_de(src, &filesize, 4);
		read_de(src, &blk_cnt, 4);
		read_de(src, &DACT_BLK_SIZE, 4);
		read(src, &file_opts, 1);
		read_de(src, &file_extd_size, 4);

		printf("File              :   %s\n", filename);
		printf("Magic             :   0x%08x",magic);
		if (magic!=DACT_MAGIC_NUMBER) {
			printf(" (bad magic)\n");
			return(0);
		} else {
			printf("\n");
		}

		if (filesize==0) {
			lseek_net(src, -8, SEEK_END);
			read_de(src, &magic, 4);
			read_de(src, &filesize, 4);
			if (magic!=DACT_MAGIC_PEOF) {
				PRINTERR("Bad magic, corrupt stream.");
				return(0);
			}
		}
		fileoutsize=lseek_net(src, 0, SEEK_END);

		printf("Dact Version      :   %i.%i.%i\n",version[0],version[1],version[2]);
		printf("Block Size        :   %i\n", DACT_BLK_SIZE);
		printf("Block Header Size :   %i\n", BYTESIZE(DACT_BLK_SIZE)+1);
		printf("Compressed Size   :   %i\n", fileoutsize);
		printf("Uncompressed Size :   %i\n", filesize);
		printf("Ratio             :   %2.3f to 1.0\n", ((float) filesize)/((float) fileoutsize) );
		lseek_net(src, 24, SEEK_SET);
		while (file_extd_read<file_extd_size) {
			x=0;
			read(src, &ch, 1);
			if (ch!=DACT_HDR_NOP) read_de(src, &x, 2); 
			switch (ch) {
				case DACT_HDR_CRC0:
					read_de(src, &crcs[2], 4);
					printf("CRC 0             :   0x%08x\n", crcs[2]);
					break;
				case DACT_HDR_CRC1:
					read_de(src, &crcs[3], 4);
					printf("CRC 1             :   0x%08x\n", crcs[3]);
					break;
		 		case DACT_HDR_NAME:
					hdr_buf=malloc(x+1);
					read_f(src, hdr_buf, x);
					hdr_buf[x]=0;
					printf("Original Name     :   %s\n", hdr_buf);
					free(hdr_buf);
					break;
				case DACT_HDR_URL:
					hdr_buf=malloc(x+1);
					read_f(src, hdr_buf, x);
					hdr_buf[x]=0;
					file_extd_urls[file_extd_urlcnt++]=parse_url_subst(hdr_buf,filename);
					free(hdr_buf);
					break;
				case DACT_HDR_URLFILE:
					hdr_buf=malloc(x+1);
					read_f(src, hdr_buf, x);
					hdr_buf[x]=0;
					printf("Download Loc File :   %s\n", hdr_buf);
					extd_urlfile=fopen(hdr_buf, "r");
					free(hdr_buf);   /* We shouldn't need this anymore. */
					if (extd_urlfile==NULL) break;
					hdr_buf=malloc(4096);
					while (1) {
						fgets(hdr_buf, 4095, extd_urlfile);
						if (feof(extd_urlfile)) break;
						hdr_buf=strsep(&hdr_buf,"\n");
						file_extd_urls[file_extd_urlcnt++]=parse_url_subst(hdr_buf,filename);
					}
					free(hdr_buf);
					break;
				case DACT_HDR_DESC:
					hdr_buf=malloc(x+1);
					read_f(src, hdr_buf, x);
					hdr_buf[x]=0;
					printf("Description       :   %s\n", hdr_buf);
					free(hdr_buf);
					break;
				case DACT_HDR_CIPHER:
					read_de(src, &cipher, x);
					printf("Ciphered using    :   %s\n", ciphers_name[cipher]);
					break;
				case DACT_HDR_NOP:
					x=-2;
					break;
				default:
					hdr_buf=malloc(x);
					read_f(src, hdr_buf, x);
					free(hdr_buf);
					break;
			}


			file_extd_read+=(x+3);
		}

		for (x=0;x<file_extd_urlcnt;x++) {
			printf("Download Location :   %s",file_extd_urls[x]);
			if (options[DACT_OPT_VERB]) {
				if ((new_fd=open_net(file_extd_urls[x],O_RDONLY))<0) {
					printf(" [broken]\n");
					continue;
				}
				close(new_fd);
			}
			printf("\n");
		}

		printf("\n");


		return(1);
	}

	return(0);
}

int main(int argc, char **argv) {
	unsigned char options[15]={0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	signed char opt;
	char **in_files, *in_file=NULL, *out_file=NULL;
	char dact_binfile[256];
	int filecnt=0;
	int in_fd, out_fd;
	int mode=DACT_MODE_COMPR, ciphernum=-1;
	uint32_t crcs[6]={0,0,0,0,0,0};
	uint32_t i,x;


	dact_ui_init();

	dact_loadconfig(DACT_CONFIG_FILE, options);
#ifndef GO32
	strncpy(dact_binfile,getenv("HOME"),sizeof(dact_binfile)-1);
	strncat(dact_binfile,"/.dact/dact.conf",sizeof(dact_binfile)-strlen(dact_binfile)-1);
	dact_loadconfig(dact_binfile, options);
#endif
/* XXX: hack, to make upgrade work even if DACT_OPT_BINCHK is enabled, we must
 *      get the new version before executing it.
 */
	if (argv[1]!=NULL) {
		if (!strcmp(argv[1],"-N")) { return(dact_upgrade(options,crcs)); }
	}

	if (options[DACT_OPT_BINCHK]) {
#ifdef GO32
		strcpy(dact_binfile,"c:/dact/dact.exe");
#else
		strncpy(dact_binfile,getenv("HOME"),sizeof(dact_binfile)-1);
		strncat(dact_binfile,"/.dact/dact.bin",sizeof(dact_binfile)-strlen(dact_binfile)-1);
#endif

		if (strcmp(argv[0],dact_binfile)) {
			if (!access(dact_binfile,X_OK)) {
				argv[0]=dact_binfile;
				execv(dact_binfile,argv);
			}
		}
	}

	while ((opt=getopt(argc,argv,"adfsvcnNVHCE:p:I:m:e:lb:hu:U:TPOD:o:"))!=-1) {
		switch (opt) {
			case 'a':
				PRINTERR("This option does nothing.");
				break;
			case 'd':
				mode=DACT_MODE_DECMP;
				break;
			case 'f':
				options[DACT_OPT_FORCE]++;
				break;
			case 's':
				mode=DACT_MODE_STAT;
				break;
			case 'c':
				options[DACT_OPT_STDOUT]=!options[DACT_OPT_STDOUT];
				break;
			case 'b':
				i=atoi2(optarg);
				if (i<DACT_BLK_SIZE_MAX) DACT_BLK_SIZE=i;
				break;
			case 'v':
				options[DACT_OPT_VERB]++;
				dact_ui_setopt(DACT_UI_OPT_LEVEL,dact_ui_getopt(DACT_UI_OPT_LEVEL)+1);
				break;
			case 'n':
				options[DACT_OPT_NOCRC]=!options[DACT_OPT_NOCRC];
				break;
			case 'p':
				PRINT_LINE; fprintf(stderr, "dact: %s\n",parse_url_subst(optarg,"@@file@@"));
				mode=DACT_MODE_RET;
				break;
			case 'C':
				options[DACT_OPT_COMPLN]++;
				break;
			case 'm':
				dact_loadconfig(optarg, options);
				break;
			case 'e':
				i=(atoi(optarg)&0xff);
				algorithms[i]=DACT_FAILED_ALGO;
				break;
			case 'H':
				options[DACT_OPT_HDONLY]=!options[DACT_OPT_HDONLY];
				break;
			case 'o':
				out_file=malloc(strlen(optarg)+1);
				strcpy(out_file,optarg);
				break;
			case 'N':
				return(dact_upgrade(options,crcs));
				break;
			case 'E':
				strtolower(optarg);
				x=hash_fourbyte(optarg, ' ');
				if (x==hash_fourbyte("list", ' ')) {
					PRINT_LINE; fprintf(stderr, "dact: Num | Name\n");
					for (i=0;i<CIPHER_COUNT;i++) {
						if (ciphers_name[i]!=NULL) {
							PRINT_LINE; fprintf(stderr, "dact: %3i | %s\n",i,ciphers_name[i]);
						}
					}
					return(0);
				}
				for (i=0;i<CIPHER_COUNT;i++) {
					if (ciphers_name[i]!=NULL) {
						if (x==hash_fourbyte(ciphers_name[i], ' ')) {
							break;
						}
					}
				}
				if (i==CIPHER_COUNT) i=-1;
				ciphernum=i;
				if (ciphernum==-1) {
					PRINTERR("No such cipher.");
					return(-1);
				}
				break;
			case 'l':
				PRINT_LINE; fprintf(stderr, "dact: Num | Name\n");
				for (i=0;i<255;i++) {
					if (algorithms[i]==NULL || algorithms[i]==DACT_FAILED_ALGO) continue;
					PRINT_LINE; fprintf(stderr, "dact: %3i | %s\n",i,algorithm_names[i]);
			
				}
				mode=DACT_MODE_RET;
				break;
			case 'I':
				x=atoi(optarg);
				for (i=1;i<255;i++) {
					if (i!=x) algorithms[i]=NULL;
				}
				break;
			case 'V':
				printf("DACT %i.%i.%i-%s", DACT_VER_MAJOR, DACT_VER_MINOR, DACT_VER_REVISION, DACT_VER_SUB);
#if defined(__DATE__) && defined(__TIME__)
				printf("  built on %s at %s",__DATE__,__TIME__);
#endif
#ifdef DACT_CONTACT
				printf(" %s",DACT_CONTACT);
#endif
				printf("\n");
				return(0);
				break;
			case 'u':
				dact_hdr_ext_regs(DACT_HDR_URL, optarg, strlen(optarg));
				break;
			case 'U':
				dact_hdr_ext_regs(DACT_HDR_URLFILE, optarg, strlen(optarg));
				break;
			case 'D':
				dact_hdr_ext_regs(DACT_HDR_DESC, optarg, strlen(optarg));
				break;
			case 'T':
				options[DACT_OPT_TIME]=!options[DACT_OPT_TIME];
				break;
			case 'P':
				options[DACT_OPT_PERM]=!options[DACT_OPT_PERM];
				break;
			case 'O':
				options[DACT_OPT_ORIG]=!options[DACT_OPT_ORIG];
				break;
			case 'S':
				options[DACT_OPT_SZSPD]=!options[DACT_OPT_SZSPD];
				break;
			case '?':
			case ':':
			case 'h':
				return(print_help(argc,argv));
		}

	}

/* Check for a new version of DACT (is_latest() returns immediately if
 * this option is disabled.)
 */
	if ((i=is_latest(options))) {
		PRINTERR("**+");
		PRINTERR("**> There is a new version of DACT available.");
		PRINTERR("**>");
		PRINT_LINE; fprintf(stderr, "dact: **> [CURR: DACT %i.%i.%i]\n",DACT_VER_MAJOR, DACT_VER_MINOR, DACT_VER_REVISION);
		PRINT_LINE; fprintf(stderr, "dact: **> [NEW:  DACT %i.%i.%i]\n",i>>16,(i>>8)&0xff,i&0xff);
		PRINTERR("**>");
		PRINTERR("**> Run `dact -N' to get it.");
		PRINTERR("**> or get the source at: http://www.rkeene.org/devel/dact.tar.gz");
		PRINTERR("**>");
		PRINTERR("**-");
	}

	if (mode==DACT_MODE_RET) return(0);

	in_files=&argv[optind];

/* Loop through extra parameters (files ...) and setup FDs for them */
	do {
		in_fd=-1;
		out_fd=-1;

		in_file=in_files[filecnt];
		if (in_file!=NULL) {
/* Determine resulting file name */
			if (out_file==NULL) out_file=dact_getoutfilename(in_file,mode);
			if (!strcmp("-",in_file)) {
				in_fd=STDIN_FILENO;
			} else if ((in_fd=open_net(in_file,O_RDONLY))==-1) {
				fprintf(stderr, "dact: Can't open %s.\n",in_file);
				PERROR("open");
				continue;
			}
			if (out_file!=NULL) {
/*
 *
 * This is a bad thing if this program is SUID root, which it NEVER EVER 
 * should be.
 *
 *  DO NOT MAKE DACT SUID ROOT OR YOU WILL BE HACKED
 * 
 * that should be a suffcient warning.
 *
 */
				if (access(out_file,F_OK)!=-1 && options[DACT_OPT_FORCE]==0 && options[DACT_OPT_STDOUT]==0) {
					fprintf(stderr, "dact: %s exists.\n",out_file);
					close(in_fd);
					continue;
				}
				if (!options[DACT_OPT_STDOUT]) {
					if ((out_fd=open_net(out_file,O_WRONLY|O_CREAT|O_TRUNC,0644))==-1) {
						fprintf(stderr, "dact: Can't open %s for writing.\n",out_file);
						PERROR("open");
						continue;
					}
				}
			}
			if (options[DACT_OPT_STDOUT]) out_fd=STDOUT_FILENO;
		}

/* Use STDIN/STDOUT if no files specified ... unless an outfile was specified... */
		if (in_file==NULL && filecnt==0) {
/* ... But only if STDOUT isn't a terminal */
			if (isatty(STDOUT_FILENO) && options[DACT_OPT_FORCE]==0) {
				fprintf(stderr, "dact: Refusing to write compressed output to a terminal.\n");
			} else {
				out_fd=STDOUT_FILENO;
				in_fd=STDIN_FILENO;
			}
		}

/* Okay, we're all done, now pass these to something to do the real stuff */
		if (in_fd!=-1 && (out_fd!=-1 || mode==DACT_MODE_STAT)) {
			crcs[1]=crcs[0]=0;
			if (dact_process_file(in_fd, out_fd, mode, options, in_file, crcs, ciphernum)==0) {
				close(in_fd);
				close(out_fd);
				if (out_fd!=STDOUT_FILENO) {
					unlink(out_file);
				}
				return(dact_shutdown(-1));
			}
		}
/* Cleanup */
	
		if (out_fd!=-1) close(out_fd);
		if (in_fd!=-1) close(in_fd);

	} while (in_files[filecnt++]!=NULL);

	return(dact_shutdown(0));
}
