/*
 * Copyright (C) 2001  Roy Keene
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *      email: dact@rkeene.org
 */

/*
	Encrypt data.
*/
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#ifndef RANDOM_DEV
#include <time.h>
#endif
#include "cipher_sub.h"
#include "parse.h"
#include "dact.h"
#include "ui.h"



int cipher_sub(const char *inblock, char *outblock, const int blksize, char *key, const int mode) {
	switch (mode) {
		case (DACT_MODE_CINIT+DACT_MODE_CDEC):
		case (DACT_MODE_CINIT+DACT_MODE_CENC):
		case DACT_MODE_CINIT:
			return(cipher_sub_init(mode,key));
			break;
		case DACT_MODE_CDEC:
			return(cipher_sub_decrypt(inblock, outblock, blksize, key));
			break;
		case DACT_MODE_CENC:
			return(cipher_sub_encrypt(inblock, outblock, blksize, key));
			break;
	}
	return(0);
}


int cipher_sub_init(const int mode, char *key) {
	return(cipher_sub_init_getkey(mode-DACT_MODE_CINIT,key));
}

int cipher_sub_init_getkey(const int mode, char *key) {
	char *fname;
	char keybuf[1024], *ptrbuf;
	int fd,x=257;

	fname=dact_ui_getuserinput("Key file: ",128,0);
	fd=open(fname, O_RDONLY);
	if (fd>=0) {
		x=read(fd, &keybuf, sizeof(keybuf));
		if (x==257) {
			memcpy(key,keybuf,257); /* For backward compatability with DACT 0.8.1*/
		} else {
			memcpy(key,ptrbuf=demime64(keybuf),257);
			free(ptrbuf);
		}
		close(fd);
		return(257);
	}
	if (fd<0 && mode==DACT_MODE_CENC) {
		fd=open(fname, O_WRONLY, 0600);
		memcpy(key,generatekey(),257);
		memcpy(keybuf, ptrbuf=mimes64(key,&x), 400);
		write(fd, keybuf, x);
		write(fd, "\n", 1);
		close(fd);
		free(ptrbuf);
		return(257);
	}
	return(-1);
}

int cipher_sub_encrypt(const char *inblk, char *outblk, int blksize, char *key) {
	char *tmpbuf;
	tmpbuf=cryptdata(key,inblk,blksize);
	memcpy(outblk,tmpbuf,blksize);
	free(tmpbuf);
	return(blksize);
}

int cipher_sub_decrypt(const char *inblk, char *outblk, int blksize, char *key) {
	char *tmpbuf;
	tmpbuf=decryptdata(key,inblk,blksize);
	memcpy(outblk,tmpbuf,blksize);
	free(tmpbuf);
	return(blksize);
}

char *cryptdata(unsigned char key[257], const unsigned char *data, int size) {
	int i,mod;
	char *result;
	static int keyoffset=0;
	result=malloc(size);

	mod=(int) key[0];
	for (i=0;i<size;i++) {
		if (!(i%mod)) {
			keyoffset=((keyoffset+1)&0xff);
		}
		result[i]=key[((((int) data[i])+keyoffset)&0xff)+1];
	}
	return(result);
}

char *decryptdata(unsigned char key[257], const unsigned char *data, int size) {
	int i,mod,x;
	char reversekey[256];
	char *result;
	static int keyoffset=0;

	mod=(int) key[0];
	result=malloc(size);
	for (i=1;i<257;i++) reversekey[(int) key[i]]=(i-1);
	for (i=0;i<size;i++) {
		if (!(i%mod)) {
			keyoffset=((keyoffset+1)&0xff);
			for (x=0;x<256;x++) reversekey[(int) key[((x+keyoffset)&0xff)+1]]=x;
		}
		result[i]=reversekey[(int) data[i]];
	}
	return(result);
}

char *generatekey(void) {
	static char key[257];
	char used[256];
	int i,x;
#ifdef RANDOM_DEV
	unsigned char buff[1];
	int fd;
	fd=open(RANDOM_DEV, O_RDONLY);
	read(fd, buff, 1);
	if (buff[0]==0) buff[0]=3;
	key[0]=buff[0];
#else
	srand(time(NULL)+rand());
	key[0]=1+(int) (255.0*rand()/(RAND_MAX+1.0));
#endif
	for (i=0;i<256;i++) used[i]=0;
	for (i=1;i<257;i++) {
#ifdef RANDOM_DEV
		read(fd, buff, 1);
		x=buff[0];
#else
		srand(time(NULL)+rand());
		x=(int) (256.0*rand()/(RAND_MAX+1.0));
#endif
		if (used[x]==0) { key[i]=x; used[x]=1; } else { i--; }
	}
#ifdef RANDOM_DEV
	close(fd);
#endif
	return(key);
}
