/*
 * Copyright (C) 2000  Roy Keene
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *      email: rkeene@netfueldesign.com
 */


/*
	Dynamic Compression Routines 
*/

#define __DACT_C
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include "dact.h"
#include "algorithms.h"
#include "endian.h"
#ifdef CHECK_VERSION
#include "net.h"
#endif

#ifdef HAVE_GETOPT
extern char *optarg;
extern int optind, opterr, optopt;
#endif

int print_help(int argc, char **argv) {
	fprintf(stderr, "DACT version %i.%i.%i by Roy Keene at Netfuel Design (rkeene@netfueldesign.com).\n\n",DACT_VER_MAJOR,DACT_VER_MINOR,DACT_VER_REVISION);
	fprintf(stderr, "Usage:\n\t%s [-sdfv[v[v]]Clc] [-e XX] [-b XX] [--] <filename> [outfile]\n\n",argv[0]);
	fprintf(stderr, "  -s\tDisplay statistics about file (Don't compress or decompress.)\n");
	fprintf(stderr, "  -d\tDecompress instead of compressing.\n");
	fprintf(stderr, "  -f\tForce overwriting existing files.\n");
	fprintf(stderr, "  -v\tIncrease verbosity.\n");
	fprintf(stderr, "  -C\tComplain when compression verification fails.\n");
	fprintf(stderr, "  -l\tList compression algorithms availble.\n");
	fprintf(stderr, "  -c\tOutput to stdout forcefully.\n");
	fprintf(stderr, "  -e\tExclude algorithm XX.\n");
	fprintf(stderr, "  -b\tUse block size XX.\n");
	return(0);
}


FOUR_BYTES is_latest (void) {
#ifdef CHECK_VERSION
	int fd;
	char ver_maj[4]={0,0,0,0}, ver_min[4]={0,0,0,0}, ver_rev[4]={0,0,0,0};
	char bigbuf[1024], *smallbuf;
	int vers[3];
	int past=0;

	if (getuid()==0) return (0);

	if ((fd=createconnection_tcp("keene.netfueldesign.com", 80))<0) return(0);

	write(fd, "GET /devel/dact/VERSION HTTP/1.1\n", 33);
	write(fd, "Host: keene.netfueldesign.com\n\n", 31);
	read(fd, &bigbuf,1024);

	strtok(bigbuf, "\n");
	while ((smallbuf=strtok(NULL, "\n"))!=NULL) {
		if (strlen(smallbuf)==1) past=1;
		if (strlen(smallbuf)==9 && past==1) {
			memcpy(ver_maj,smallbuf,3);
			memcpy(ver_min,smallbuf+3,3);
			memcpy(ver_rev,smallbuf+6,3);
			past=2;
		}
	}
	closeconnection(fd);

	if (past!=2) return(0);

	vers[0]=atoi(ver_maj);
	vers[1]=atoi(ver_min);
	vers[2]=atoi(ver_rev);

	if (vers[0]>DACT_VER_MAJOR || vers[1]>DACT_VER_MINOR || vers[2]>DACT_VER_REVISION) {
		return((vers[0]<<16)|(vers[1]<<8)|vers[2]);
	} else {
		return(0);
	}
#else
	return(0);
#endif
}



int main(int argc, char **argv) {
	char in_filename[128], out_filename[128];
	char in_buffer[MAX_DACT_BLK_SIZE], prev_buffer[MAX_DACT_BLK_SIZE];
	char out_buffer[MAX_DACT_BLK_SIZE*2];
	char store_buffer[MAX_DACT_BLK_SIZE*2];
	char verify_buffer[MAX_DACT_BLK_SIZE*2];
	char cmp_options=0;
	char file_version[3]={DACT_VER_MAJOR, DACT_VER_MINOR, DACT_VER_REVISION};
	char option_verbose=0;
	char option_complain=0;
	char option_old=0;
	char ch;
	FOUR_BYTES file_size_real=0, file_size, out_file_size, in_magic;
	FOUR_BYTES version_latest;
	int stat_cnt[256];
	int in_fd, out_fd=-1;
	int i,x=0;
	int out_size=-1, out_lowest, out_algo=-1;
	int bread;
	int mode=DACT_MODE_COMPR;
	int mode_force=0;
	int number_algorithm=0;
	int verify_size;
	int verified_data;
	struct stat statinfo;
	struct dact_header header;
#ifndef HAVE_GETOPT
	char *optarg=NULL;
	int optind;
#endif


#ifdef DEBUG
	option_verbose=1;
#endif

	if ((version_latest=is_latest())!=0) {
		fprintf(stderr, "\033[1mNOTE:\n");
		fprintf(stderr, "  There is a newer version of DACT: %i.%i.%i\n", version_latest>>16, (version_latest>>8)&0xff, version_latest&0xff);
		fprintf(stderr, "  http://keene.netfueldesign.com/devel/dact.tar.gz is the latest version.\033[0m\n\n");
	}

#ifdef HAVE_GETOPT
	while ((ch = getopt(argc,argv,"dfvClshce:b:")) != -1) {
#else
	for (optind=1;optind<argc;optind++) {
		if (argv[optind][0]=='-' && argv[optind][1]!='-') {
			ch=argv[optind][1];
			if (ch=='e' || ch=='b') optarg=argv[++optind];
		} else {
			break;
		} 
#endif
		switch (ch) {
			case 'd':
				mode=DACT_MODE_DECMP;
				break;
			case 'f':
				mode_force++;
				break;
			case 'v':
				option_verbose++;
				break;
			case 'C':
				option_complain=1;
				break;
			case 'b':
				if (optarg!=NULL) {
					i=atoi(optarg);
					if (i>MAX_DACT_BLK_SIZE) i=MAX_DACT_BLK_SIZE;
					DACT_BLK_SIZE=i;
				}
				break;
			case 'l':
				fprintf(stderr, " Number | Algorithm Name\n");
				fprintf(stderr, "--------+----------------------------------------------------------------------\n");
				for (i=0;i<256;i++) {
					if (algorithms[i]==NULL) break;
					fprintf(stderr, "    %03i | %s\n",i,algorithm_names[i]);
				}
				return(0);
			case 'e':
				if (optarg!=NULL) {
					i=atoi(optarg);
					algorithms[i]=comp_fail_algo;
				}
				break;
			case 's':
				mode=DACT_MODE_STAT;
				break;
			case 'c':
				out_fd=STDOUT_FILENO;
				break;
			case 'h':
				print_help(argc,argv);
				return(0);
			case ':':
			case '?':
				fprintf(stderr, "%s: Try %s -h for help.\n",argv[0], argv[0]);
				break;

		}
	}


	for (i=0;i<sizeof(prev_buffer);i++) prev_buffer[i]=0;
	for (i=0;i<256;i++) stat_cnt[i]=0;

	if (argv[optind]==NULL) {
		in_filename[0]=0;
	} else {
		strncpy(in_filename,argv[optind],sizeof(in_filename)-1);
	}

	if (!strcmp(in_filename,"-") || in_filename[0]==0) {
		in_fd=STDIN_FILENO;
	} else { 
		if ((in_fd=open(in_filename,O_RDONLY))==-1) {
			fprintf(stderr, "%s: Unable to open %s for reading.\n",argv[0],in_filename);
			return(-1);
		}
	}

	if (mode!=DACT_MODE_STAT) {
	 	if (argc==(2+optind)) {
			strncpy(out_filename,argv[1+optind],sizeof(out_filename)-1);
		} else {
			if (isatty(STDOUT_FILENO)) {
				if (isatty(in_fd)) {
					return(print_help(argc,argv));
				}
				if (mode==DACT_MODE_COMPR) {
					strncpy(out_filename,in_filename,sizeof(out_filename)-4);
					strcat(out_filename,".dct");
				} else {
// TODO: Fix this, it will break in paths with dots in them, heh.
					strncpy(out_filename,strtok(in_filename,"."),sizeof(out_filename)-1);
				}
			} else {
				out_fd=STDOUT_FILENO;
			}
		}

		if (out_fd==-1) {
			if (!access(out_filename, F_OK)) {
				if (mode_force) {
					unlink(out_filename);
				} else {
					fprintf(stderr, "%s: %s exists, exiting.\n",argv[0],out_filename);
					close(in_fd);
					return(-1);
				}
			}
	
			if ((out_fd=open(out_filename,O_WRONLY|O_CREAT,0644))==-1) {
				fprintf(stderr, "%s: Unable to open %s for writing.\n",argv[0],out_filename);
				return(-1);
			}
		}
	}


	for (i=0;i<256;i++) { if (algorithms[i]==NULL) break; }
	number_algorithm=i;

/* This should be phased out as much as possible (though still required
   to determine how much of BLOCK is required when decompressing.
*/
	fstat(in_fd, &statinfo);
	file_size=statinfo.st_size;

	switch (mode) { 
		case DACT_MODE_COMPR:
			write(out_fd,&file_version,sizeof(file_version));
			i=WRITE_4BYTE_DE(out_fd,DACT_MAGIC_NUMBER);
			i=WRITE_4BYTE_DE(out_fd,file_size);
			i=WRITE_2BYTE_DE(out_fd,DACT_BLK_SIZE);
//			write(out_fd,&file_size,sizeof(file_size));
//			write(out_fd,&DACT_BLK_SIZE,sizeof(DACT_BLK_SIZE));
			write(out_fd,&cmp_options,sizeof(cmp_options));
			while (1) {
				x++;
				out_lowest=DACT_BLK_SIZE*2;
				if ((bread = read(in_fd,&in_buffer,DACT_BLK_SIZE))==0) break;
				file_size_real+=bread;
				for (i=0;i<255;i++) {
					if (algorithms[i]==NULL) break;
					out_size=algorithms[i](DACT_MODE_COMPR, &prev_buffer, &in_buffer, &out_buffer, DACT_BLK_SIZE);
					if (option_verbose>2)
						fprintf(stderr, "  --       algo #%i: size=%i\n",i,out_size);
					if (out_size>0) {
						verify_size=algorithms[i](DACT_MODE_DECMP, &prev_buffer, &out_buffer, &verify_buffer, out_size);
					} else {
						verify_size=-1;
					}

					verified_data=0;
					if (verify_size==DACT_BLK_SIZE) {
						if (!memcmp(&verify_buffer, &in_buffer, DACT_BLK_SIZE)) {
							verified_data=1;
						}
					}

					if (!verified_data && verify_size>0 && option_complain) {
						fprintf(stderr, "Block #%05i: algo=%03i: Verification failed! %34s\n",x,i,algorithm_names[i]);
					}

					if (out_size<out_lowest && (out_size > 0) && verified_data) {
						memcpy(store_buffer,out_buffer,out_size);
						out_algo=i;
						out_lowest=out_size;
					}
					memcpy(prev_buffer,in_buffer,DACT_BLK_SIZE);
				}
				if (out_lowest==(DACT_BLK_SIZE*2)) {
					fprintf(stderr, "Compression resulted in DACT_BLK_SIZE*2 !  Aborting.\n");
					return(-1);
				}
				if (option_verbose>1) {
					fprintf(stderr, "Block #%05i: algo=%03i (size=%05i, %05i)  %35s\n", x,out_algo,out_lowest,(int) (out_lowest+sizeof(header.algo)+sizeof(header.size)),algorithm_names[out_algo]);
				}
				if (option_verbose==1) {
					fprintf(stderr, "Block %5i/%5i\r",x,(int) (((float) (((float) file_size)/DACT_BLK_SIZE))+0.99));
					fflush(stderr);
				}
				header.size=(TWO_BYTES) (out_lowest);
				header.algo=out_algo;
				i=WRITE_2BYTE_DE(out_fd,header.size);
//				write(out_fd,&header.size,sizeof(header.size));
				write(out_fd,&header.algo,sizeof(header.algo));
				write(out_fd,&store_buffer,out_lowest);
			}
			lseek(out_fd, sizeof(file_version)+DACT_MAGIC_SIZE, SEEK_SET);
			i=WRITE_4BYTE_DE(out_fd, file_size_real);
//			write(out_fd, &file_size_real, sizeof(file_size_real));
			break;
		case DACT_MODE_DECMP:
			i=x=0;
			x+=read(in_fd,&file_version,sizeof(file_version));

			if ( ((file_version[0]<<16)|(file_version[1]<<8)|file_version[2])  <= 0x000402) {
				fprintf(stderr, "Warning: Old Format\n");
				
				option_old=1;
				if (file_version[2]==2) option_old=2;
				in_magic=DACT_MAGIC_NUMBER;
			} else {
				x+=READ_4BYTE_DE(in_fd,in_magic);
			}

			if (in_magic!=DACT_MAGIC_NUMBER && mode_force<2) {
				printf("%s: Bad magic.\n", argv[0]);
				return(-1);
			}

			if (option_old) {
				x+=read(in_fd,&out_file_size,sizeof(out_file_size));
				if (option_old==2) { x+=read(in_fd,&DACT_BLK_SIZE,sizeof(int)); } else { DACT_BLK_SIZE=1024; }
			} else {
				x+=READ_4BYTE_DE(in_fd,out_file_size);
				x+=READ_2BYTE_DE(in_fd,DACT_BLK_SIZE);
			}

			x+=read(in_fd,&cmp_options,sizeof(cmp_options));
//			x=sizeof(out_file_size)+sizeof(cmp_options);

			if (file_version[0]!=DACT_VER_MAJOR && mode_force<2) {
				fprintf(stderr, "Major version numbers do not match! (%i!=%i)\n",file_version[0],DACT_VER_MAJOR);
				fprintf(stderr, "File is not usable, exiting...\n");
				close(in_fd);
				close(out_fd);
				return(-1);
			}
			if (file_version[1]!=DACT_VER_MINOR || file_version[2]!=DACT_VER_REVISION) {
				fprintf(stderr, "Minor or Revision numbers do not match (file=%i.%i.%i != current=%i.%i.%i)\n",file_version[0],file_version[1],file_version[2],DACT_VER_MAJOR,DACT_VER_MINOR,DACT_VER_REVISION);
				fprintf(stderr, "File may not be usuable.\n");
			}


			while(1) {
				i++;
				if (option_old) {
					read(in_fd,&header.size,sizeof(header.size));
				} else {
					i+=2-READ_2BYTE_DE(in_fd,header.size);
				}
				read(in_fd,&header.algo,sizeof(header.algo));
				read(in_fd,&in_buffer,header.size);
				x+=sizeof(header.size)+sizeof(header.algo)+header.size;
				if (header.algo>=(number_algorithm-1)) {
					fprintf(stderr, "Request for an unknown algorithm: %i\n", header.algo);
					close(in_fd);
					close(out_fd);
					return(-1);
				}
				out_size=algorithms[header.algo](DACT_MODE_DECMP, &prev_buffer, &in_buffer, &out_buffer, header.size);
				if (out_size!=DACT_BLK_SIZE) {
					fprintf(stderr, "Block resulted in data that is less than %i bytes (=%i).\n",DACT_BLK_SIZE,out_size);
					close(in_fd);
					close(out_fd);
					return(-1);
				}
				if (option_verbose>1) {
					fprintf(stderr, "Block #%05i: algo=%03i (size=%05i, %05i) %35s\n",i,header.algo,header.size,header.size+3,algorithm_names[header.algo]);
				} 
				if (option_verbose==1) {
					fprintf(stderr, "Block %5i/%5i\r",(i-1),(int) (((float) (((float) out_file_size)/DACT_BLK_SIZE))+0.99));
					fflush(stderr);
				}
				if (((i*DACT_BLK_SIZE))>=out_file_size) {
					out_size-=((i*DACT_BLK_SIZE)-out_file_size);
				}
				write(out_fd,&out_buffer,out_size);
				if (x>=file_size) break;
				memcpy(prev_buffer,out_buffer,out_size);
			}
			break;
		case DACT_MODE_STAT:
			x=read(in_fd,&file_version,sizeof(file_version));

			if ( ((file_version[0]<<16)|(file_version[1]<<8)|file_version[2])  <= 0x000402) {
				fprintf(stderr, "Warning: Old file format.\n");
				option_old=1;
				x+=read(in_fd,&out_file_size,sizeof(out_file_size));
				if (file_version[2]==2) {
					x+=read(in_fd,&DACT_BLK_SIZE,sizeof(int));
				} else {
					DACT_BLK_SIZE=1024;
				}

				in_magic=DACT_MAGIC_NUMBER;
			} else {
				x+=READ_4BYTE_DE(in_fd,in_magic);
				x+=READ_4BYTE_DE(in_fd,out_file_size);
				x+=READ_2BYTE_DE(in_fd,DACT_BLK_SIZE);
			}
//			x+=read(in_fd,&out_file_size,sizeof(out_file_size));
//			x+=read(in_fd,&DACT_BLK_SIZE,sizeof(DACT_BLK_SIZE));
			x+=read(in_fd,&cmp_options,sizeof(cmp_options));

			printf("File              = %s\n",in_filename);
			printf("Magic             = 0x%08x",in_magic);
			if (in_magic!=DACT_MAGIC_NUMBER && mode_force<2) {
				printf(" (Bad magic.)\n");
				close(in_fd);
				close(out_fd);
				return(-1);
			} else {
				printf("\n");
			}

			printf("DACT version      = %i.%i.%i\n",file_version[0],file_version[1],file_version[2]);
			printf("Block size        = %i\n",DACT_BLK_SIZE);
			printf("Compressed size   = %u\n",file_size);
			printf("Uncompressed size = %u\n",out_file_size);
			printf("Ratio             = %2.3f%%\n",100.0-((((float) (file_size))/((float) (out_file_size)))*100));


			if (option_verbose!=0) {
				printf("NUMBER | COUNT | NAME\n");
				printf("-------+-------+----------------------------------------\n");
				while (1) {
					if (option_old) {
						x+=read(in_fd,&header.size,sizeof(header.size));
					} else {
						x+=READ_2BYTE_DE(in_fd,header.size);
					}
					x+=read(in_fd,&header.algo,sizeof(header.algo));
					x+=header.size;
					lseek(in_fd,header.size,SEEK_CUR);
					stat_cnt[header.algo]++;
					if (x>=file_size) break;
				}
				for (i=0;i<256;i++) {
					if (algorithms[i]==NULL) break;
					printf("%6i | %5i | %s\n",i,stat_cnt[i],algorithm_names[i]);
				}
			}
			break;			
	}

	close(in_fd);
	close(out_fd);

	if (option_verbose==1 && mode!=DACT_MODE_STAT) fprintf(stderr, "\n");
	return(0);
}
