#include <sys/types.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <utime.h>
#include <stdio.h>
#include <fcntl.h>

#define MOUNT_DEV_RW_CMD "mount -o rw %s %s"
#define MOUNT_DEV_RO_CMD "mount -o ro %s %s"
#define MOUNT_LOOP_RW_CMD "mount -o rw,loop %s %s"
#define MOUNT_LOOP_RO_CMD "mount -o ro,loop %s %s"


uint32_t elfhash(const unsigned char *str) {
	uint32_t h=0,g=0;
	while (*str) {
		h = (h << 4) + (*str++);
		if ((g = (h & 0xf0000000))) h ^= (g >> 24);
		h &= ~g;
	}
	return(h);
}

char *mkfs_getcmd(const char *fs, const char *target) {
	uint32_t fs_hash=0;
	char *ret;
	int retlen=0;

	ret=malloc((retlen=256));

	fs_hash=elfhash(fs);
	switch(fs_hash) {
		case 0x6cf72:   /* ext2 */
		case 0x6cf73:   /* ext3 */
		case 0x740508:  /* minix */
			snprintf(ret, retlen, "mkfs -t %s -F %s", fs, target);
			break;
		case 0x70d3:    /* jfs */
		case 0x7ed3:    /* xfs */
			snprintf(ret, retlen, "mkfs -t %s -f %s", fs, target);
			break;
		case 0xc09cf53: /* reiserfs */
			snprintf(ret, retlen, "mkreiserfs -f %s", target);
			break;
		case 0x7cc84:   /* vfat */
		case 0x75ad3:   /* ntfs */
			break;
		default:
			fprintf(stderr, "Unknown filesystem hash: 0x%x (%s)\n", fs_hash, fs);
			free(ret);
			ret=NULL;
			break;
	}

	if (ret==NULL) return("");
	return(ret);
}

int copyfile_shrink(const char *srcfile, const char *destfile, uint64_t maxlen) {
	int src_fd=-1, dest_fd=-1;

	dest_fd=open(destfile, O_WRONLY|O_CREAT|O_EXCL);
	if (dest_fd<0) return(-1);
	close(dest_fd);

	return(0);
}

int copydir_shrink(const char *srcdir, const char *destdir, const char *exclude) {
	struct utimbuf utimebuf={0};
	struct statfs statfsbuf={0};
	struct stat statbuf={0};
	struct dirent *dinfo=NULL;
	char srcfile[1024]={0}, destfile[1024]={0}, linkbuf[1024]={0};
	uint64_t fsfree=0;
	DIR *dirtype=NULL;
	int count=0, copy_ret=0, copied=0, i=0;

	if ((dirtype=opendir(srcdir))==NULL) return(-1);
	while (1) {
		if ((dinfo=readdir(dirtype))==NULL) break;
		if (strcmp(dinfo->d_name, exclude)==0 || \
		    strcmp(dinfo->d_name, ".")==0 || \
		    strcmp(dinfo->d_name, "..")==0) continue;
		snprintf(srcfile, sizeof(srcfile), "%s/%s", srcdir, dinfo->d_name);
		snprintf(destfile, sizeof(destfile), "%s/%s", destdir, dinfo->d_name);
		if (lstat(srcfile, &statbuf)<0) return(-1);
		copied=0;

		/*
                  ino_t         st_ino;      * inode *
                  nlink_t       st_nlink;    * number of hard links *
		*/

		/* If it's a directory, recurse into it. */
		if (S_ISDIR(statbuf.st_mode)) {
			/* Abort if we can't make a directory. */
			printf("mkdir %s\n", destfile);
			if (mkdir(destfile, statbuf.st_mode)<0) return(-1);

			/* Recurse into directory, abort on failure. */
			copy_ret=copydir_shrink(srcfile, destfile, exclude);
			if (copy_ret<0) return(-1);
			count+=copy_ret;
			copied=1;
		}

		if (S_ISCHR(statbuf.st_mode)) {
                	/*  dev_t         st_rdev;     * device type (if inode device) * */
			/* Create a charectar device */
			printf("mk_chr_dev %s\n", destfile);
			if (mknod(destfile, statbuf.st_mode, statbuf.st_dev)<0) return(-1);
			copied=1;
		}

		if (S_ISBLK(statbuf.st_mode)) {
                	/*  dev_t         st_rdev;     * device type (if inode device) * */
			/* Create a block device */
			printf("mk_blk_dev %s\n", destfile);
			if (mknod(destfile, statbuf.st_mode, statbuf.st_dev)<0) return(-1);
			copied=1;
		}

		if (S_ISFIFO(statbuf.st_mode)) {
			/* Create a FIFO */
			printf("mk_fifo %s\n", destfile);
			if (mknod(destfile, statbuf.st_mode, 0)<0) return(-1);
			copied=1;
		}

		if (S_ISLNK(statbuf.st_mode)) {
			for (i=0; i<sizeof(linkbuf); i++) linkbuf[i]='\0';
			/* Create a symbolic link */
			if (readlink(srcfile, linkbuf, sizeof(linkbuf))<0) {
				fprintf(stderr, "readlink() failed.\n");
				return(-1);
			}
			if (symlink(linkbuf, destfile)<0) {
				fprintf(stderr, "symlink() failed.\n");
				return(-1);
			}
			printf("symlink %s -> %s\n", destfile, linkbuf);
			copied=1;
		}

		if (S_ISSOCK(statbuf.st_mode)) {
			/* Ignore sockets? */
			continue;
		}

		/* Just copy regular files. */
		if (S_ISREG(statbuf.st_mode) && !copied) {
			/* Determine how much free space we have. */
			statfs(srcdir, &statfsbuf);
			fsfree=((statfsbuf.f_bfree*statfsbuf.f_bsize)/2)+1;

			/* Copy file. */
			printf("copy %s -> %s\n", srcfile, destfile);
			copy_ret=copyfile_shrink(srcfile, destfile, fsfree);
			if (copy_ret<0) {
				fprintf(stderr, "Copy failed.\n");
				return(-1);
			}
			count++;
			copied=1;
		}

		if (!copied) {
			/* If we're unable to copy a file, abort! */
			fprintf(stderr, "Unknown filetype: %s (%08o).\n", srcfile, statbuf.st_mode);
			return(-1);
		}

		/* Set the file attributes */
		utimebuf.actime=statbuf.st_atime;
		utimebuf.modtime=statbuf.st_mtime;
		if (utime(destfile, &utimebuf)<0) {
			fprintf(stderr, "utime() failed.\n");
			return(-1);
		}
		if (lchown(destfile, statbuf.st_uid, statbuf.st_gid)<0) {
			fprintf(stderr, "lchown() failed.\n");
			return(-1);
		}
		if (chmod(destfile, statbuf.st_mode)<0) {
			fprintf(stderr, "chmod() failed.\n");
			return(-1);
		}
		/* 
                  time_t        st_ctime;    * time of last change *
		*/

	}
	closedir(dirtype);
	return(count);
}

int main(int argc, char **argv) {
	struct statfs statfsbuf={0};
	char *device=NULL, *targetfs=NULL, tmpdir[256]="/tmp/convertfs", tmploopdir[256]="/tmp/convertfs_loop",
	     mountcmd[512]={0}, *mkfscmd=NULL, tmpfile[384]={0};
	uint64_t fslen;
	int fd;
	int sys_ret;

	if (argc<3) {
		fprintf(stderr, "Usage: convertfs <targetfs> <device>\n");
		return(1);
	}
	targetfs=argv[1];
	device=argv[2];

	/* Make our temporary directory, to `mount' at. */
	mkdir(tmpdir, 0600);
	mkdir(tmploopdir, 0600);
	snprintf(tmpfile, sizeof(tmpfile), "%s/__NEWFS__", tmpdir);

	/* Mount the filesystem read-write. */
	snprintf(mountcmd, sizeof(mountcmd), MOUNT_DEV_RW_CMD, device, tmpdir);
	if ((sys_ret=system(mountcmd))!=0) {
		fprintf(stderr, "Unable to mount %s on %s.\n>>  %-50s [%i]\n", device, tmpdir, mountcmd, sys_ret);
		return(1);
	}

	/* Determine size of file system. */
	statfs(tmpdir, &statfsbuf);
	fslen=statfsbuf.f_blocks*statfsbuf.f_bsize;

	/* Create a sparse file on the filesystem. */
	fd=open(tmpfile, O_WRONLY|O_CREAT|O_EXCL|O_TRUNC, 0600);
	if (fd<0) {
		fprintf(stderr, "Unable to create temporary file (%s), exiting.\n", tmpfile);
		return(1);
	}
	if (lseek(fd, (fslen-1), SEEK_SET)!=(fslen-1)) {
		fprintf(stderr, "Unable to create temporary file (%s:%llu), exiting.\n", tmpfile, fslen);
		return(1);
	}
	if (write(fd, "\0", 1)<0) {
		fprintf(stderr, "Unable to write to temporary file (%s:%llu), exiting.\n", tmpfile, fslen);
		return(1);
	}

	/* Make a filesystem on the sparse file. */
	mkfscmd=mkfs_getcmd(targetfs, tmpfile);
	if ((sys_ret=system(mkfscmd))!=0) {
		fprintf(stderr, "Unable to mkfs %s on %s.\n>>  %-50s [%i]\n", targetfs, tmpfile, mkfscmd, sys_ret);
		return(1);
	}

	/* Mount the new filesystem. */
	snprintf(mountcmd, sizeof(mountcmd), MOUNT_LOOP_RW_CMD, tmpfile, tmploopdir);
	if ((sys_ret=system(mountcmd))!=0) {
		fprintf(stderr, "Unable to mount %s on %s.\n>>  %-50s [%i]\n", device, tmpdir, mountcmd, sys_ret);
		return(1);
	}

	/* Move files from real filesystem to new filesystem. */
	if (copydir_shrink(tmpdir, tmploopdir, "__NEWFS__")<0) {
		fprintf(stderr, "Unable to copy files from source to destination.\n");
		return(1);
	}

	/* Unmount the filesystems. */
	/* Mount the old filesystem read-only. */
	/* Verify that no other files exist. */
	/* Create a map of the new filesystem's file on the device. */
	/* Unmount the filesystem. */
	/* Copy file from device to outer device. */
	/* Mount new filesystem read-write. */

	return(0);
}
