#include "compat.h"
#include "backuppcd.h"
#include "backuppcd-common.h"
#include "libbackuppcd.h"
#include "sha1.h"
#include "net.h"

#define BPC_CMD_GET_DATA (BPC_CMD_GET | 0x100)
#define BPC_MAX_RDIFF_BLOCKSIZE 8192

struct bpc_transfer {
	uint64_t filesize;
	int64_t idealblocknum;
	uint32_t currblocknum;
	uint32_t blocksize;
	uint32_t blockoffset;
};

struct bpc_conn {
	struct bpc_fileinfo dent;
	struct bpc_transfer tinfo;
	backuppc_cmd_t state;
	int fd;
};

typedef enum {
        BPC_CDT_END,
        BPC_CDT_UINT8,
        BPC_CDT_UINT16,
        BPC_CDT_UINT32,
        BPC_CDT_UINT64,
        BPC_CDT_BYTEARRAY,
        BPC_CDT_STRING,
        BPC_CDT_STRING_PTR,
	BPC_CDT_SINK,
} bpc_clntdt_t;

#ifndef MSG_WAITALL
#define MSG_WAITALL 0
#endif

#ifndef BPC_MAXPATH_LEN
#define BPC_MAXPATH_LEN 4096
#endif                                                                                                                                    

static int _bpc_client_read(const int sockid, ...) {
	bpc_clntdt_t typeid;
	char tmpbuf[1024];
	va_list ap;
	uint8_t *u8v;
	uint16_t *u16v;
	uint32_t *u32v;
	uint64_t *u64v;
	char *cpv, **cppv;
	void *vpv = NULL, *vpv_s;
	size_t vpv_len, bytes_to_copy;
	ssize_t read_ret;

	va_start(ap, sockid);
	while (1) {
		typeid = va_arg(ap, bpc_clntdt_t);

		if (typeid == BPC_CDT_END) {
			break;
		}

		switch (typeid) {
			case BPC_CDT_UINT8:
				u8v = va_arg(ap, uint8_t *);
				vpv_len = sizeof(*u8v);
				vpv = u8v;
				break;
			case BPC_CDT_UINT16:
				u16v = va_arg(ap, uint16_t *);
				vpv_len = sizeof(*u16v);
				vpv = u16v;
				break;
			case BPC_CDT_UINT32:
				u32v = va_arg(ap, uint32_t *);
				vpv_len = sizeof(*u32v);
				vpv = u32v;
				break;
			case BPC_CDT_UINT64:
				u64v = va_arg(ap, uint64_t *);
				vpv_len = sizeof(*u64v);
				vpv = u64v;
				break;
			case BPC_CDT_BYTEARRAY:
				vpv_len = va_arg(ap, size_t);
				vpv = va_arg(ap, void *);
				break;
			case BPC_CDT_STRING:
				vpv_len = va_arg(ap, size_t);
				vpv_len--;
				cpv = va_arg(ap, char *);
				cpv[vpv_len] = '\0';
				vpv = cpv;
				break;
			case BPC_CDT_STRING_PTR:
				vpv_len = va_arg(ap, size_t);
				cppv = va_arg(ap, char **);
				if (*cppv == NULL) {
					cpv = *cppv = malloc(vpv_len + 1);
					cpv[vpv_len] = '\0';
				} else {
					cpv = *cppv;
				}
				vpv = cpv;
				break;
			case BPC_CDT_SINK:
				vpv_len = va_arg(ap, size_t);
				vpv = NULL;
				break;
			case BPC_CDT_END:
				return(0);
				break;
			default:
				return(0);
		}

		vpv_s = vpv;
		while (vpv_len) {
			if (vpv) {
				read_ret = recv(sockid, vpv, vpv_len, MSG_WAITALL);
			} else {
				/*
				 * If we have been given no place to put the
				 * data, we just read it to a buffer and
				 * ignore it (aka, sink it).
				 */
				bytes_to_copy = sizeof(tmpbuf);
				if (vpv_len < bytes_to_copy) {
					bytes_to_copy = vpv_len;
				}
				read_ret = recv(sockid, tmpbuf, bytes_to_copy, MSG_WAITALL);
			}

			if (read_ret <= 0) {
				return(0);
			}

			vpv_len -= read_ret;

			if (vpv) {
				vpv += read_ret;
			}
		}
		vpv = vpv_s;

		switch (typeid) {
			case BPC_CDT_UINT16:
				u16v = vpv;
				*u16v = ntohs(*u16v);
				break;
			case BPC_CDT_UINT32:
				u32v = vpv;
				*u32v = ntohl(*u32v);
				break;
			case BPC_CDT_UINT64:
				u64v = vpv;
				*u64v = ntohll(*u64v);
				break;
			default:
				break;
		}
	}
	va_end(ap);

	return(1);
}

static int _bpc_client_write(const int sockid, ...) {
	bpc_clntdt_t typeid;
	va_list ap;
	uint8_t u8v;
	uint16_t u16v;
	uint32_t u32v;
	uint64_t u64v;
	void *vpv;
	char *cpv;
	size_t vpv_len;
	ssize_t write_ret;

	va_start(ap, sockid);
	while (1) {
		typeid = va_arg(ap, bpc_clntdt_t);

		if (typeid == BPC_CDT_END) {
			break;
		}

		switch (typeid) {
			case BPC_CDT_UINT8:
				u8v = va_arg(ap, int);
				vpv_len = sizeof(u8v);
				vpv = &u8v;
				break;
			case BPC_CDT_UINT16:
				u16v = va_arg(ap, int);
				u16v = htons(u16v);
				vpv_len = sizeof(u16v);
				vpv = &u16v;
				break;
			case BPC_CDT_UINT32:
				u32v = va_arg(ap, uint32_t);
				u32v = htonl(u32v);
				vpv_len = sizeof(u32v);
				vpv = &u32v;
				break;
			case BPC_CDT_UINT64:
				u64v = va_arg(ap, uint64_t);
				u64v = htonll(u64v);
				vpv_len = sizeof(u64v);
				vpv = &u64v;
				break;
			case BPC_CDT_BYTEARRAY:
				vpv_len = va_arg(ap, size_t);
				vpv = va_arg(ap, void *);
				break;
			case BPC_CDT_STRING:
				cpv = va_arg(ap, char *);
				vpv_len = strlen(cpv);
				vpv = cpv;
				break;
			case BPC_CDT_SINK:
			case BPC_CDT_STRING_PTR:
			case BPC_CDT_END:
				return(0);
			default:
				return(0);
		}

		while (vpv_len) {
			write_ret = send(sockid, vpv, vpv_len, 0);

			if (write_ret < 0) {
				return(0);
			}

			vpv_len -= write_ret;
			vpv += write_ret;
		}
	}
	va_end(ap);

	return(1);
}

int bpc_auth(BPC_CONN *handle, const char *username, const char *password) {
	uint8_t cmd_reply;
	uint8_t status;
	int sockid;

	if (!handle || !username || !password) {
		return(0);
	}

	if (handle->state != BPC_CMD_NONE) {
		return(0);
	}

	sockid = handle->fd;

	if (!_bpc_client_write(sockid,
	                      BPC_CDT_UINT8, (uint8_t) BPC_CMD_AUTH,
	                      BPC_CDT_UINT16, (uint16_t) strlen(username),
	                      BPC_CDT_UINT16, (uint16_t) strlen(password),
	                      BPC_CDT_STRING, username,
	                      BPC_CDT_STRING, password,
	                      BPC_CDT_END)) {
		return(0);
	}

	if (!_bpc_client_read(sockid,
	                     BPC_CDT_UINT8, (uint8_t *) &cmd_reply,
	                     BPC_CDT_UINT8, (uint8_t *) &status,
	                     BPC_CDT_END)) {
		return(0);
	}

	if (cmd_reply != BPC_CMD_AUTH_REPLY) {
		return(0);
	}

	if (status != BPC_STATUS_OKAY) {
		return(0);
	}

	return(1);
}

static int _bpc_listget_open(BPC_CONN *handle, backuppc_cmd_t sendcmd, const char *rootpath, const int recursive, backuppc_hashid_t hashalgo, const char *exclpat, const char *inclpat) {
	uint8_t cmd, cmd_options = 0;
	int sockid;

	sockid = handle->fd;

	if (sendcmd != BPC_CMD_LIST && sendcmd != BPC_CMD_GET) {
		return(0);
	}

	/*
	 * Handle NULL patterns as blank ones.
	 */
	if (!exclpat) {
		exclpat = "";
	}
	if (!inclpat) {
		inclpat = "";
	}

	if (recursive) {
		cmd_options |= BPC_OPT_RECURSIVE;
	}

	switch (hashalgo) {
		case BPC_HASH_NONE:
			break;
		case BPC_HASH_MD4:
			cmd_options |= BPC_OPT_MD4;
			break;
		case BPC_HASH_MD5:
			cmd_options |= BPC_OPT_MD5;
			break;
		case BPC_HASH_SHA1:
			cmd_options |= BPC_OPT_SHA1;
			break;
		case BPC_HASH_BPC:
			cmd_options |= BPC_OPT_BPCHASH;
			break;
	}

	if (!_bpc_client_write(sockid,
	                      BPC_CDT_UINT8, (uint8_t) sendcmd,
	                      BPC_CDT_UINT8, (uint8_t) cmd_options,
	                      BPC_CDT_UINT32, (uint32_t) strlen(exclpat),
	                      BPC_CDT_UINT32, (uint32_t) strlen(inclpat),
	                      BPC_CDT_UINT32, (uint32_t) strlen(rootpath),
	                      BPC_CDT_STRING, (char *) rootpath,
	                      BPC_CDT_STRING, (char *) exclpat,
	                      BPC_CDT_STRING, (char *) inclpat,
	                      BPC_CDT_END)) {
		CHECKPOINT;
		return(0);
	}

	if (!_bpc_client_read(sockid,
	                     BPC_CDT_UINT8, (uint8_t *) &cmd,
	                     BPC_CDT_END)) {
		CHECKPOINT;
		return(0);
	}

	/*
	 * If the server sends us a reply for something other than what we
	 * asked for, something went wrong, abort.
	 */
	if (cmd != (sendcmd | 0x80)) {
		CHECKPOINT;
		return(0);
	}

	handle->state = sendcmd;

	handle->dent.owner_group[0] = '\0';
	handle->dent.owner_user[0] = '\0';
	handle->dent.linkdest[0] = '\0';
	handle->dent.name[0] = '\0';

	memset(&handle->dent.hash_md4, 0, sizeof(handle->dent.hash_md4));
	memset(&handle->dent.hash_md5, 0, sizeof(handle->dent.hash_md5));
	memset(&handle->dent.hash_sha1, 0, sizeof(handle->dent.hash_sha1));
	memset(&handle->dent.hash_bpc, 0, sizeof(handle->dent.hash_bpc));

	return(1);
}

static int _bpc_listget_close(BPC_CONN *handle) {

	handle->state = BPC_CMD_NONE;

	return(1);
}

static struct bpc_fileinfo *_bpc_list(BPC_CONN *handle, backuppc_cmd_t cmd) {
	struct bpc_fileinfo *ret;
	struct bpc_transfer *tinfo;
	int sockid;

	uint64_t filesize;
	uint32_t attr_sect_len, blocksize, pathnamelen;
	uint32_t attr_uid, attr_gid, attr_ctime, attr_mtime, attr_devmajor, attr_devminor;
	uint32_t attrlen;
	uint16_t attrid;
	uint16_t attr_mode;
	uint8_t filetype;
	mode_t modeval;

	ret = &handle->dent;
	tinfo = &handle->tinfo;

	sockid = handle->fd;

	if (!_bpc_client_read(sockid,
	                      BPC_CDT_UINT8, (uint8_t *) &filetype,
	                      BPC_CDT_END)) {
		CHECKPOINT;
		return(NULL);
	}

	if (filetype == 0xff) {
		CHECKPOINT;
		_bpc_listget_close(handle);
		return(NULL);
	}

	if (!_bpc_client_read(sockid,
	                      BPC_CDT_UINT32, (uint32_t *) &attr_sect_len,
	                      BPC_CDT_UINT64, (uint64_t *) &filesize,
	                      BPC_CDT_UINT32, (uint32_t *) &blocksize,
	                      BPC_CDT_UINT32, (uint32_t *) &pathnamelen,
	                      BPC_CDT_END)) {
		CHECKPOINT;
		return(NULL);
	}

	if (pathnamelen >= sizeof(ret->name)) {
		SPOTVAR_I(pathnamelen);
		CHECKPOINT;
		return(NULL);
	}

	if (!_bpc_client_read(sockid,
	                      BPC_CDT_STRING, (size_t) (pathnamelen + 1), &ret->name,
	                      BPC_CDT_END)) {
		CHECKPOINT;
		return(NULL);
	}

	attr_uid = 0;
	attr_gid = 0;
	attr_mode = 0;
	attr_ctime = 0;
	attr_mtime = 0;
	attr_devmajor = 0;
	attr_devminor = 0;

	/*
	 * Read the sent attributes.
	 */
	if (attr_sect_len != 0) {
		while (1) {
			if (!_bpc_client_read(sockid,
			                      BPC_CDT_UINT16, (uint16_t *) &attrid,
			                      BPC_CDT_END)) {
				CHECKPOINT;
				return(NULL);
			}

			if (attrid == 0xffff) {
				break;
			}

			if (!_bpc_client_read(sockid,
			                      BPC_CDT_UINT32, (uint32_t *) &attrlen,
			                      BPC_CDT_END)) {
				CHECKPOINT;
				return(NULL);
			}

			switch ((backuppc_attrid_t) attrid) {
				case BPC_ATTRID_NOP:
					if (attrlen != 0) {
						CHECKPOINT;
						return(NULL);
					}
					break;
				case BPC_ATTRID_MTIME:
					if (!_bpc_client_read(sockid,
					                      BPC_CDT_UINT32, (uint32_t *) &attr_mtime,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
				case BPC_ATTRID_CTIME:
					if (!_bpc_client_read(sockid,
					                      BPC_CDT_UINT32, (uint32_t *) &attr_ctime,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
				case BPC_ATTRID_UID:
					if (!_bpc_client_read(sockid,
					                      BPC_CDT_UINT32, (uint32_t *) &attr_uid,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
				case BPC_ATTRID_GID:
					if (!_bpc_client_read(sockid,
					                      BPC_CDT_UINT32, (uint32_t *) &attr_gid,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
				case BPC_ATTRID_ACL:
					if (!_bpc_client_read(sockid,
					                      BPC_CDT_UINT16, (uint16_t *) &attr_mode,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}

					break;
				case BPC_ATTRID_USER:
					if (attrlen >= sizeof(ret->owner_user)) {
						CHECKPOINT;
						return(NULL);
					}

					if (!_bpc_client_read(sockid,
					                      BPC_CDT_STRING, (size_t) attrlen + 1, (char *) &ret->owner_user,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
				case BPC_ATTRID_GROUP:
					if (attrlen >= sizeof(ret->owner_group)) {
						CHECKPOINT;
						return(NULL);
					}

					if (!_bpc_client_read(sockid,
					                      BPC_CDT_STRING, (size_t) attrlen + 1, (char *) &ret->owner_group,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
				case BPC_ATTRID_SYMLINKDEST:
					if (attrlen >= sizeof(ret->linkdest)) {
						CHECKPOINT;
						return(NULL);
					}

					if (!_bpc_client_read(sockid,
					                      BPC_CDT_STRING, (size_t) attrlen + 1, (char *) &ret->linkdest,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
				case BPC_ATTRID_HRDLINKDEST:
					if (attrlen >= sizeof(ret->linkdest)) {
						CHECKPOINT;
						return(NULL);
					}

					if (!_bpc_client_read(sockid,
					                      BPC_CDT_STRING, (size_t) attrlen + 1, (char *) &ret->linkdest,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
				case BPC_ATTRID_MD4:
					if (attrlen != sizeof(ret->hash_md4)) {
						CHECKPOINT;
						return(NULL);
					}

					if (!_bpc_client_read(sockid,
					                      BPC_CDT_BYTEARRAY, (size_t) sizeof(ret->hash_md4), (void *) &ret->hash_md4,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
				case BPC_ATTRID_MD5:
					if (attrlen != sizeof(ret->hash_md5)) {
						CHECKPOINT;
						return(NULL);
					}

					if (!_bpc_client_read(sockid,
					                      BPC_CDT_BYTEARRAY, (size_t) sizeof(ret->hash_md5), (void *) &ret->hash_md5,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
				case BPC_ATTRID_SHA1:
					if (attrlen != sizeof(ret->hash_sha1)) {
						CHECKPOINT;
						return(NULL);
					}

					if (!_bpc_client_read(sockid,
					                      BPC_CDT_BYTEARRAY, (size_t) sizeof(ret->hash_sha1), (void *) &ret->hash_sha1,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
				case BPC_ATTRID_BPCHASH:
					if (attrlen != sizeof(ret->hash_bpc)) {
						CHECKPOINT;
						return(NULL);
					}

					if (!_bpc_client_read(sockid,
					                      BPC_CDT_BYTEARRAY, (size_t) sizeof(ret->hash_bpc), (void *) &ret->hash_bpc,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
#ifndef DEBUG
				default:
					if (!_bpc_client_read(sockid,
					                      BPC_CDT_SINK, (size_t) attrlen,
					                      BPC_CDT_END)) {
						CHECKPOINT;
						return(NULL);
					}
					break;
#endif
			}
		}
	}

	/*
	 * Determine the POSIX "mode" value from the ACL attrs.
	 */
	modeval = 0;
	if (attr_mode) {
		if ((attr_mode & BPC_ACL_XUSR) == BPC_ACL_XUSR) {
			modeval |= S_IXUSR;
		}
		if ((attr_mode & BPC_ACL_WUSR) == BPC_ACL_WUSR) {
			modeval |= S_IWUSR;
		}
		if ((attr_mode & BPC_ACL_RUSR) == BPC_ACL_RUSR) {
			modeval |= S_IRUSR;
		}
#ifndef _USE_WIN32_
		if ((attr_mode & BPC_ACL_XGRP) == BPC_ACL_XGRP) {
			modeval |= S_IXGRP;
		}
		if ((attr_mode & BPC_ACL_WGRP) == BPC_ACL_WGRP) {
			modeval |= S_IWGRP;
		}
		if ((attr_mode & BPC_ACL_RGRP) == BPC_ACL_RGRP) {
			modeval |= S_IRGRP;
		}
		if ((attr_mode & BPC_ACL_XOTH) == BPC_ACL_XOTH) {
			modeval |= S_IXOTH;
		}
		if ((attr_mode & BPC_ACL_WOTH) == BPC_ACL_WOTH) {
			modeval |= S_IWOTH;
		}
		if ((attr_mode & BPC_ACL_ROTH) == BPC_ACL_ROTH) {
			modeval |= S_IROTH;
		}
		if ((attr_mode & BPC_ACL_STCK) == BPC_ACL_STCK) {
			modeval |= S_ISVTX;
		}
		if ((attr_mode & BPC_ACL_SGID) == BPC_ACL_SGID) {
			modeval |= S_ISGID;
		}
		if ((attr_mode & BPC_ACL_SUID) == BPC_ACL_SUID) {
			modeval |= S_ISUID;
		}
#endif
	}

#if 0
	if (filetype == BPC_FILE_REG) {
		modeval |= S_IFREG;
	}
#ifndef _USE_WIN32_
	if (filetype == BPC_FILE_SYMLINK) {
		modeval |= S_IFLNK;
	}
	if (filetype == BPC_FILE_SOCKET) {
		modeval |= S_IFSOCK;
	}
#endif
	if (filetype == BPC_FILE_DIR) {
		modeval |= S_IFDIR;
	}
	if (filetype == BPC_FILE_BDEV) {
		modeval |= S_IFBLK;
	}
	if (filetype == BPC_FILE_CDEV) {
		modeval |= S_IFCHR;
	}
	if (filetype == BPC_FILE_FIFO) {
		modeval |= S_IFIFO;
	}
#endif

	ret->mode = modeval;
	ret->type = filetype;
	ret->size = filesize;
	ret->uid = attr_uid;
	ret->gid = attr_gid;
	ret->mtime = attr_mtime;
	ret->ctime = attr_ctime;

	if (cmd == BPC_CMD_GET) {
		tinfo->blocksize = blocksize;
		tinfo->idealblocknum = -1;
		tinfo->currblocknum = 0;
		tinfo->filesize = filesize;
		tinfo->blockoffset = 0;
	}

	return(ret);
}

int bpc_list_open(BPC_CONN *handle, const char *rootpath, const int recursive, backuppc_hashid_t hashalgo, const char *exclpat, const char *inclpat) {
	if (!handle) {
		return(0);
	}

	if (handle->state != BPC_CMD_NONE) {
		return(0);
	}

	return(_bpc_listget_open(handle, BPC_CMD_LIST, rootpath, recursive, hashalgo, exclpat, inclpat));
}

int bpc_list_close(BPC_CONN *handle) {
	if (!handle) {
		return(0);
	}

	if (handle->state != BPC_CMD_NONE) {
		return(0);
	}

	return(1);
}

struct bpc_fileinfo *bpc_list(BPC_CONN *handle) {
	if (!handle) {
		return(NULL);
	}

	if (handle->state != BPC_CMD_LIST) {
		return(NULL);
	}

	return(_bpc_list(handle, BPC_CMD_LIST));
}

int bpc_get_open(BPC_CONN *handle, const char *rootpath, const int recursive, backuppc_hashid_t hashalgo, const char *exclpat, const char *inclpat) {
	struct bpc_transfer *tinfo;
	int retval;

	if (!handle) {
		return(0);
	}

	if (handle->state != BPC_CMD_NONE) {
		return(0);
	}

	retval = _bpc_listget_open(handle, BPC_CMD_GET, rootpath, recursive, hashalgo, exclpat, inclpat);

	if (retval == 0) {
		return(0);
	}

	tinfo = &handle->tinfo;

	return(retval);
}

int bpc_get_close(BPC_CONN *handle) {
	if (!handle) {
		return(0);
	}

	if (handle->state != BPC_CMD_NONE) {
		return(0);
	}

	return(1);
}

struct bpc_fileinfo *bpc_get_head(BPC_CONN *handle) {
	uint32_t blocknum;
	struct bpc_fileinfo *ret;

	if (!handle) {
		CHECKPOINT;
		return(NULL);
	}

	if (handle->state != BPC_CMD_GET) {
		CHECKPOINT;
		return(NULL);
	}

	/*
	 * The "GET" operation is the same as the "LIST" operation, except
	 * data will follow if the type is BPC_FILE_REG.
	 */
	ret = _bpc_list(handle, BPC_CMD_GET);

	if (!ret) {
		CHECKPOINT;
		return(NULL);
	}

	if (ret->type != BPC_FILE_REG) {
		if (!_bpc_client_read(handle->fd,
		                      BPC_CDT_UINT32, &blocknum,
		                      BPC_CDT_END)) {
			CHECKPOINT;
			return(NULL);
		}

		if (blocknum != 0xffffffff) {
			CHECKPOINT;
			return(NULL);
		}

		CHECKPOINT;
		return(ret);
	}

	handle->state = BPC_CMD_GET_DATA;

	return(ret);
}

ssize_t bpc_get(BPC_CONN *handle, char *buf, size_t count) {
	struct bpc_transfer *tinfo;
	ssize_t recv_ret;
	uint32_t bytes_left;
	uint32_t sinkbytes = 0;
	int sockid;

	if (!handle) {
		return(-1);
	}

	if (handle->state != BPC_CMD_GET_DATA) {
		return(-1);
	}

	sockid = handle->fd;

	tinfo = &handle->tinfo;

	if (tinfo->blockoffset == 0) {
		if (tinfo->currblocknum == tinfo->idealblocknum || tinfo->idealblocknum == -1) {
			if (!_bpc_client_read(sockid,
			                      BPC_CDT_UINT32, &tinfo->currblocknum,
			                      BPC_CDT_END)) {
				CHECKPOINT;
				return(-1);
			}

			if (tinfo->currblocknum == 0xffffffff) {
				handle->state = BPC_CMD_GET;
				return(0);
			}
		}

		if (tinfo->currblocknum == 0) {
			tinfo->idealblocknum = 0;
		} else {
			tinfo->idealblocknum++;
		}
	}

	bytes_left = tinfo->blocksize - tinfo->blockoffset;

	if (count > bytes_left) {
		count = bytes_left;
	}

	if ((tinfo->idealblocknum * tinfo->blocksize + tinfo->blockoffset) > tinfo->filesize) {
		count = 0;
		sinkbytes = tinfo->blocksize - tinfo->blockoffset;
	} else {
		if ((tinfo->idealblocknum * tinfo->blocksize + tinfo->blockoffset + count) > tinfo->filesize) {
			count = tinfo->filesize - (tinfo->idealblocknum * tinfo->blocksize + tinfo->blockoffset);
			sinkbytes = tinfo->blocksize - count;
		}
	}

	if (tinfo->currblocknum != tinfo->idealblocknum) {
		memset(buf, '\0', count);
		recv_ret = count;
	} else {
		recv_ret = recv(sockid, buf, count, MSG_WAITALL);

		if (sinkbytes) {
			if (!_bpc_client_read(sockid,
			                      BPC_CDT_SINK, (ssize_t) sinkbytes,
			                      BPC_CDT_END)) {
				CHECKPOINT;
				return(-1);
			}
		}
	}

	if (recv_ret < 0) {
		return(-1);
	}

	tinfo->blockoffset += recv_ret + sinkbytes;

	if (tinfo->blockoffset == tinfo->blocksize) {
		tinfo->blockoffset = 0;
	}

	return(recv_ret);
}

int bpc_copy(BPC_CONN *handle, bpc_mode_t mode, int fd) {
	char buf[32768];
	ssize_t read_ret, write_ret;
	int write_fail_count = 0;
	int retval = 1;

	while (1) {
		read_ret = bpc_get(handle, buf, sizeof(buf));

		if (read_ret == 0) {
			break;
		}

		if (read_ret < 0) {
			retval = 0;
			break;
		}

		while (read_ret) {
			switch (mode) {
				case BPC_MODE_WRITE:
					write_ret = write(fd, buf, read_ret);
					break;
				case BPC_MODE_SEND:
					write_ret = send(fd, buf, read_ret, 0);
					break;
				default:
					write_ret = -1;
			}
			

			if (write_ret < 0) {
				retval = 0;
				break;
			}

			if (write_ret == 0) {
				write_fail_count++;
			} else {
				write_fail_count = 0;
			}

			if (write_fail_count >= 10) {
				retval = 0;
				break;
			}

			read_ret -= write_ret;
		}
	}

	return(retval);
}

int bpc_rdiffget(BPC_CONN *handle, const char *remote_src, const char *dest) {
	backuppc_hashid_t hashalgo = BPC_HASH_MD4;
	struct stat stbuf;
	uint64_t filesize;
	uint32_t blocksize;
	uint32_t num_hash_ents, hash_ent, hash_ent_size, hash_list_size;
	ssize_t read_ret;
	char buf[BPC_MAX_RDIFF_BLOCKSIZE];
	void *hash_handle;
	int stat_ret;
	int sockid;
	int fd;

	if (!handle) {
		return(0);
	}

	sockid = handle->fd;

	stat_ret = stat(dest, &stbuf);
	if (stat_ret < 0) {
		filesize = 0xffffffffffffffffLLU;
		blocksize = 0;
		num_hash_ents = 0;
	} else {
		filesize = stbuf.st_size;

		/*
		 * Determine the block size:
		 * 	0 bytes to 20 MBytes:  2 KBytes
		 * 	20 MBytes to 1 GByte:  4 KBytes
		 *	1 GByte to Infinity:   8 KBytes
		 */
		if (filesize < (20 * 1024 * 1024)) {
			blocksize = 2048;
		} else if (filesize < (1024 * 1024 * 1024)) {
			blocksize = 4096;
		} else {
			blocksize = BPC_MAX_RDIFF_BLOCKSIZE;
		}

		if (blocksize > BPC_MAX_RDIFF_BLOCKSIZE) {
			blocksize = BPC_MAX_RDIFF_BLOCKSIZE;
		}

		num_hash_ents = (filesize + blocksize - 1) / blocksize;
	}

	switch (hashalgo) {
		case BPC_HASH_NONE:
			break;
		case BPC_HASH_MD5:
			hash_ent_size = (128 / 8);
			hash_handle = NULL;
			break;
		case BPC_HASH_MD4:
			hash_ent_size = (128 / 8);
			hash_handle = NULL;
			break;
		case BPC_HASH_SHA1:
			hash_ent_size = (160 / 8);
			hash_handle = malloc(sizeof(SHA1_CTX));
			break;
		case BPC_HASH_BPC:
			hash_ent_size = (128 / 8);
			hash_handle = NULL;
			break;
	}

	hash_list_size = num_hash_ents * hash_ent_size;

	if (!_bpc_client_write(sockid,
	                       BPC_CDT_UINT8, (uint8_t) BPC_CMD_RDGET,
	                       BPC_CDT_UINT64, (uint64_t) filesize,
	                       BPC_CDT_UINT32, (uint32_t) blocksize, 
	                       BPC_CDT_UINT8, (uint8_t) hashalgo,
	                       BPC_CDT_UINT32, (uint32_t) hash_list_size, 
	                       BPC_CDT_UINT32, (uint32_t) strlen(remote_src),
	                       BPC_CDT_STRING, (char *) remote_src,
	                       BPC_CDT_END)) {
		return(0);
	}

	fd = open(dest, O_RDONLY);

	if (fd < 0) {
		return(0);
	}

	for (hash_ent = 0; hash_ent < num_hash_ents; hash_ent++) {
		read_ret = read(fd, buf, blocksize);
		if (read_ret < 0) {
			read_ret = 0;
		}

		if (read_ret == 0) {
			CHECKPOINT;
		}

		switch (hashalgo) {
			case BPC_HASH_NONE:
				break;
			case BPC_HASH_SHA1:
				SHA1Init(hash_handle);
				SHA1Update(hash_handle, buf, read_ret);
//				SHA1Final(digest[20], hash_handle);
				break;
			case BPC_HASH_MD5:
			case BPC_HASH_MD4:
			case BPC_HASH_BPC:
				break;
		}
	}

	close(fd);

	if (hash_handle) {
		free(hash_handle);
	}

	return(1);
}

int bpc_copyfile(BPC_CONN *handle, struct bpc_fileinfo *src, const char *dest) {
	int fd;
	int retval = 0;
	int sys_ret;

	switch (src->type) {
		case BPC_FILE_DIR:
			backuppc_mkdir(dest);
			retval = 1;
			break;
		case BPC_FILE_SYMLINK:
#ifdef HAVE_SYMLINK
			sys_ret = symlink(src->linkdest, dest);
			if (sys_ret == 0) {
				retval = 1;
			}
#endif
			break;
		case BPC_FILE_HRDLINK:
			sys_ret = link(src->linkdest, dest);
			if (sys_ret == 0) {
				retval = 1;
			}
			break;
		case BPC_FILE_CDEV:
		case BPC_FILE_BDEV:
		case BPC_FILE_SOCKET:
		case BPC_FILE_FIFO:
		case BPC_FILE_UNKNOWN:
			return(0); 
		case BPC_FILE_REG:
			/*
			 * Regular files are handled seperately
			 */
			break;
	}

	if (src->type != BPC_FILE_REG) {
		if (retval) {
			chmod(dest, src->mode);
#ifdef HAVE_LCHOWN
			lchown(dest, src->uid, src->gid);
#else
			chown(dest, src->uid, src->gid);
#endif
#ifdef HAVE_UTIME
//			utime(...)
#endif
		}
		return(retval);
	}


	/*
	 * Handle the regular files using either RDIFF or GET.
	 */
	if (access(dest, F_OK) == 0) {
		/*
		 * Do RDIFF 
		 */
	} else {
		/*
		 * Copy the entire file.
		 */
		fd = open(dest, O_WRONLY | O_CREAT | O_TRUNC | O_EXCL, 0600);

		if (bpc_copy(handle, BPC_MODE_WRITE, fd)) {
			retval = 1;
		}

		close(fd);
	}
	chmod(dest, src->mode);
	chown(dest, src->uid, src->gid);

	return(retval);
}

BPC_CONN *bpc_connect(const char *host, const int port, const char *username, const char *password) {
	BPC_CONN *ret;
	int sockid;
	int real_port;

	if (port == 0) {
		real_port = BPC_TCP_PORT;
	} else {
		real_port = port;
	}

	sockid = net_connect_tcp(host, real_port);

	if (sockid < 0) {
		return(NULL);
	}

	ret = malloc(sizeof(*ret));

	if (!ret) {
		close(sockid);
		return(NULL);
	}

	ret->fd = sockid;
	ret->state = BPC_CMD_NONE;

	if (username && password) {
		if (!bpc_auth(ret, username, password)) {
			close(ret->fd);
			free(ret);
			return(NULL);
		}
	}

	return(ret);
}

int bpc_disconnect(BPC_CONN *handle) {
	int close_ret;

	if (!handle) {
		return(-1);
	}

	if (handle->fd < 0) {
		return(0);
	}

	close_ret = net_close(handle->fd);

	if (close_ret < 0) {
		return(close_ret);
	}

	free(handle);

	return(0);
}
