#-----------------------------------------------------------------------------
#   Copyright (C) 1999-2004 Jochen C. Loewer (loewerj@web.de)
#   Copyright (C) 2004-2011 Michael Schlenker (mic42@users.sourceforge.net)
#-----------------------------------------------------------------------------
#   
#   A partial ASN decoder/encoder implementation in plain Tcl. 
#
#   See ASN.1 (X.680) and BER (X.690).
#   See 'asn_ber_intro.txt' in this directory.
#
#   This software is copyrighted by Jochen C. Loewer (loewerj@web.de). The 
#   following terms apply to all files associated with the software unless 
#   explicitly disclaimed in individual files.
#
#   The authors hereby grant permission to use, copy, modify, distribute,
#   and license this software and its documentation for any purpose, provided
#   that existing copyright notices are retained in all copies and that this
#   notice is included verbatim in any distributions. No written agreement,
#   license, or royalty fee is required for any of the authorized uses.
#   Modifications to this software may be copyrighted by their authors
#   and need not follow the licensing terms described here, provided that
#   the new terms are clearly indicated on the first page of each file where
#   they apply.
#  
#   IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY PARTY
#   FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
#   ARISING OUT OF THE USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY
#   DERIVATIVES THEREOF, EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE
#   POSSIBILITY OF SUCH DAMAGE.
#
#   THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY WARRANTIES,
#   INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY,
#   FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THIS SOFTWARE
#   IS PROVIDED ON AN "AS IS" BASIS, AND THE AUTHORS AND DISTRIBUTORS HAVE
#   NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR
#   MODIFICATIONS.
#
#   written by Jochen Loewer
#   3 June, 1999
#
#   $Id: asn.tcl,v 1.20 2011/01/05 22:33:33 mic42 Exp $
#
#-----------------------------------------------------------------------------

# needed for using wide()
package require Tcl 8.4

namespace eval asn {
    # Encoder commands
    namespace export \
        asnSequence \
	asnSequenceFromList \
        asnSet \
	asnSetFromList \
        asnApplicationConstr \
        asnApplication \
	asnContext\
	asnContextConstr\
        asnChoice \
        asnChoiceConstr \
        asnInteger \
        asnEnumeration \
        asnBoolean \
        asnOctetString \
        asnNull	   \
	asnUTCTime \
	asnNumericString \
        asnPrintableString \
        asnIA5String\
	asnBMPString\
	asnUTF8String\
        asnBitString \
        asnObjectIdentifer 
        
    # Decoder commands
    namespace export \
        asnGetResponse \
        asnGetInteger \
        asnGetEnumeration \
        asnGetOctetString \
        asnGetSequence \
        asnGetSet \
        asnGetApplication \
	asnGetNumericString \
        asnGetPrintableString \
        asnGetIA5String \
	asnGetBMPString \
	asnGetUTF8String \
        asnGetObjectIdentifier \
        asnGetBoolean \
        asnGetUTCTime \
        asnGetBitString \
        asnGetContext 
    
    # general BER utility commands    
    namespace export \
        asnPeekByte  \
        asnGetLength \
        asnRetag     \
	asnPeekTag   \
	asnTag	     
        
}

#-----------------------------------------------------------------------------
# Implementation notes:
#
# See the 'asn_ber_intro.txt' in this directory for an introduction
# into BER/DER encoding of ASN.1 information. Bibliography information
#
#   A Layman's Guide to a Subset of ASN.1, BER, and DER
#
#   An RSA Laboratories Technical Note
#   Burton S. Kaliski Jr.
#   Revised November 1, 1993
#
#   Supersedes June 3, 1991 version, which was also published as
#   NIST/OSI Implementors' Workshop document SEC-SIG-91-17.
#   PKCS documents are available by electronic mail to
#   <pkcs@rsa.com>.
#
#   Copyright (C) 1991-1993 RSA Laboratories, a division of RSA
#   Data Security, Inc. License to copy this document is granted
#   provided that it is identified as "RSA Data Security, Inc.
#   Public-Key Cryptography Standards (PKCS)" in all material
#   mentioning or referencing this document.
#   003-903015-110-000-000
#
#-----------------------------------------------------------------------------

#-----------------------------------------------------------------------------
# asnLength : Encode some length data. Helper command.
#-----------------------------------------------------------------------------

proc ::asn::asnLength {len} {
    
    if {$len < 0} {
        return -code error "Negative length octet requested"
    }
    if {$len < 128} {
        # short form: ISO X.690 8.1.3.4 
        return [binary format c $len]
    }
    # long form: ISO X.690 8.1.3.5
    # try to use a minimal encoding, 
    # even if not required by BER, but it is required by DER
    # take care for signed vs. unsigned issues
    if {$len < 256  } {
        return [binary format H2c 81 [expr {$len - 256}]]
    }
    if {$len < 32769} {
        # two octet signed value
        return [binary format H2S 82 $len]
    }
    if {$len < 65536} {
        return [binary format H2S 82 [expr {$len - 65536}]]
    }
    if {$len < 8388608} {
        # three octet signed value    
        return [binary format H2cS 83 [expr {$len >> 16}] [expr {($len & 0xFFFF) - 65536}]] 
    }    
    if {$len < 16777216} {
        # three octet signed value    
        return [binary format H2cS 83 [expr {($len >> 16) -256}] [expr {($len & 0xFFFF) -65536}]] 
    }
    if {$len < 2147483649} { 
        # four octet signed value
        return [binary format H2I 84 $len]
    }
    if {$len < 4294967296} {
        # four octet unsigned value
        return [binary format H2I 84 [expr {$len - 4294967296}]]
    }
    if {$len < 1099511627776} {
        # five octet unsigned value
        return [binary format H2 85][string range [binary format W $len] 3 end]  
    }
    if {$len < 281474976710656} {
        # six octet unsigned value
        return [binary format H2 86][string range [binary format W $len] 2 end]
    }
    if {$len < 72057594037927936} {
        # seven octet value
        return [binary format H2 87][string range [binary format W $len] 1 end]
    }
    
    # must be a 64-bit wide signed value
    return [binary format H2W 88 $len] 
}

#-----------------------------------------------------------------------------
# asnSequence : Assumes that the arguments are already ASN encoded.
#-----------------------------------------------------------------------------

proc ::asn::asnSequence {args} {
    asnSequenceFromList $args
}

proc ::asn::asnSequenceFromList {lst} {
    # The sequence tag is 0x30. The length is arbitrary and thus full
    # length coding is required. The arguments have to be BER encoded
    # already. Constructed value, definite-length encoding.

    set out ""
    foreach part $lst {
        append out $part
    }
    set len [string length $out]
    return [binary format H2a*a$len 30 [asnLength $len] $out]
}


#-----------------------------------------------------------------------------
# asnSet : Assumes that the arguments are already ASN encoded.
#-----------------------------------------------------------------------------

proc ::asn::asnSet {args} {
    asnSetFromList $args
}

proc ::asn::asnSetFromList {lst} {
    # The set tag is 0x31. The length is arbitrary and thus full
    # length coding is required. The arguments have to be BER encoded
    # already.

    set out ""
    foreach part $lst {
        append out $part
    }
    set len [string length $out]
    return [binary format H2a*a$len 31 [asnLength $len] $out]
}


#-----------------------------------------------------------------------------
# asnApplicationConstr
#-----------------------------------------------------------------------------

proc ::asn::asnApplicationConstr {appNumber args} {
    # Packs the arguments into a constructed value with application tag.

    set out ""
    foreach part $args {
        append out $part
    }
    set code [expr {0x060 + $appNumber}]
    set len  [string length $out]
    return [binary format ca*a$len $code [asnLength $len] $out]
}

#-----------------------------------------------------------------------------
# asnApplication
#-----------------------------------------------------------------------------

proc ::asn::asnApplication {appNumber data} {
    # Packs the arguments into a constructed value with application tag.

    set code [expr {0x040 + $appNumber}]
    set len  [string length $data]
    return [binary format ca*a$len $code [asnLength $len] $data]
}

#-----------------------------------------------------------------------------
# asnContextConstr
#-----------------------------------------------------------------------------

proc ::asn::asnContextConstr {contextNumber args} {
    # Packs the arguments into a constructed value with application tag.

    set out ""
    foreach part $args {
        append out $part
    }
    set code [expr {0x0A0 + $contextNumber}]
    set len  [string length $out]
    return [binary format ca*a$len $code [asnLength $len] $out]
}

#-----------------------------------------------------------------------------
# asnContext
#-----------------------------------------------------------------------------

proc ::asn::asnContext {contextNumber data} {
    # Packs the arguments into a constructed value with application tag.
    set code [expr {0x080 + $contextNumber}]
    set len  [string length $data]
    return [binary format ca*a$len $code [asnLength $len] $data]
}
#-----------------------------------------------------------------------------
# asnChoice
#-----------------------------------------------------------------------------

proc ::asn::asnChoice {appNumber args} {
    # Packs the arguments into a choice construction.

    set out ""
    foreach part $args {
        append out $part
    }
    set code [expr {0x080 + $appNumber}]
    set len  [string length $out]
    return [binary format ca*a$len $code [asnLength $len] $out]
}

#-----------------------------------------------------------------------------
# asnChoiceConstr
#-----------------------------------------------------------------------------

proc ::asn::asnChoiceConstr {appNumber args} {
    # Packs the arguments into a choice construction.

    set out ""
    foreach part $args {
        append out $part
    }
    set code [expr {0x0A0 + $appNumber}]
    set len  [string length $out]
    return [binary format ca*a$len $code [asnLength $len] $out]
}

#-----------------------------------------------------------------------------
# asnInteger : Encode integer value.
#-----------------------------------------------------------------------------

proc ::asn::asnInteger {number} {
    asnIntegerOrEnum 02 $number
}

#-----------------------------------------------------------------------------
# asnEnumeration : Encode enumeration value.
#-----------------------------------------------------------------------------

proc ::asn::asnEnumeration {number} {
    asnIntegerOrEnum 0a $number
}

#-----------------------------------------------------------------------------
# asnIntegerOrEnum : Common code for Integers and Enumerations
#                    No Bignum version, as we do not expect large Enums.
#-----------------------------------------------------------------------------

proc ::asn::asnIntegerOrEnum {tag number} {
    # The integer tag is 0x02 , the Enum Tag 0x0a otherwise identical. 
    # The length is 1, 2, 3, or 4, coded in a
    # single byte. This can be done directly, no need to go through
    # asnLength. The value itself is written in big-endian.

    # Known bug/issue: The command cannot handle very wide integers, i.e.
    # anything above 8 bytes length. Use asnBignumInteger for those.
    
    # check if we really have an int
    set num $number
    incr num
    
    if {($number >= -128) && ($number < 128)} {
        return [binary format H2H2c $tag 01 $number]
    }
    if {($number >= -32768) && ($number < 32768)} {
        return [binary format H2H2S $tag 02 $number]
    }
    if {($number >= -8388608) && ($number < 8388608)} {
        set numberb [expr {$number & 0xFFFF}]
        set numbera [expr {($number >> 16) & 0xFF}]
        return [binary format H2H2cS $tag 03 $numbera $numberb]
    }
    if {($number >= -2147483648) && ($number < 2147483648)} {
        return [binary format H2H2I $tag 04 $number]
    }
    if {($number >= -549755813888) && ($number < 549755813888)} {
        set numberb [expr {$number & 0xFFFFFFFF}]
        set numbera [expr {($number >> 32) & 0xFF}]
        return [binary format H2H2cI $tag 05 $numbera $numberb]
    }
    if {($number >= -140737488355328) && ($number < 140737488355328)} {
        set numberb [expr {$number & 0xFFFFFFFF}]
        set numbera [expr {($number >> 32) & 0xFFFF}]
        return [binary format H2H2SI $tag 06 $numbera $numberb]        
    }
    if {($number >= -36028797018963968) && ($number < 36028797018963968)} {
        set numberc [expr {$number & 0xFFFFFFFF}]
        set numberb [expr {($number >> 32) & 0xFFFF}]
        set numbera [expr {($number >> 48) & 0xFF}]
        return [binary format H2H2cSI $tag 07 $numbera $numberb $numberc]        
    }    
    if {($number >= -9223372036854775808) && ($number <= 9223372036854775807)} {
        return [binary format H2H2W $tag 08 $number]
    }
    return -code error "Integer value to large to encode, use asnBigInteger" 
}

#-----------------------------------------------------------------------------
# asnBigInteger : Encode a long integer value using math::bignum
#-----------------------------------------------------------------------------

proc ::asn::asnBigInteger {bignum} {
    # require math::bignum only if it is used
    package require math::bignum
    
    # this is a hack to check for bignum...
    if {[llength $bignum] < 2 || ([lindex $bignum 0] ne "bignum")} {
        return -code error "expected math::bignum value got \"$bignum\""
    }
    if {[math::bignum::sign $bignum]} {
        # generate two's complement form
        set bits [math::bignum::bits $bignum]
        set padding [expr {$bits % 8}]
        set len [expr {int(ceil($bits / 8.0))}]
        if {$padding == 0} {
            # we need a complete extra byte for the sign
            # unless this is a base 2 multiple
            set test [math::bignum::fromstr 0]
            math::bignum::setbit test [expr {$bits-1}]
            if {[math::bignum::ne [math::bignum::abs $bignum] $test]} {
                incr len
            }
        }
        set exp [math::bignum::pow \
		    [math::bignum::fromstr 256] \
		    [math::bignum::fromstr $len]]
        set bignum [math::bignum::add $bignum $exp]
        set hex [math::bignum::tostr $bignum 16]
    } else {
        set bits [math::bignum::bits $bignum]
        if {($bits % 8) == 0 && $bits > 0} {
            set pad "00"
        } else {
            set pad ""
        }
        set hex $pad[math::bignum::tostr $bignum 16]
    }
    if {[string length $hex]%2} {
        set hex "0$hex"
    }
    set octets [expr {(([string length $hex]+1)/2)}]
    return [binary format H2a*H* 02 [asnLength $octets] $hex]   
}


#-----------------------------------------------------------------------------
# asnBoolean : Encode a boolean value.
#-----------------------------------------------------------------------------

proc ::asn::asnBoolean {bool} {
    # The boolean tag is 0x01. The length is always 1, coded in
    # a single byte. This can be done directly, no need to go through
    # asnLength. The value itself is written in big-endian.

    return [binary format H2H2c 01 01 [expr {$bool ? 0x0FF : 0x0}]]
}

#-----------------------------------------------------------------------------
# asnOctetString : Encode a string of arbitrary bytes
#-----------------------------------------------------------------------------

proc ::asn::asnOctetString {string} {
    # The octet tag is 0x04. The length is arbitrary, so we need
    # 'asnLength' for full coding of the length.

    set len [string length $string]
    return [binary format H2a*a$len 04 [asnLength $len] $string]
}

#-----------------------------------------------------------------------------
# asnNull : Encode a null value
#-----------------------------------------------------------------------------

proc ::asn::asnNull {} {
    # Null has only one valid encoding
    return \x05\x00
}

#-----------------------------------------------------------------------------
# asnBitstring : Encode a Bit String value
#-----------------------------------------------------------------------------

proc ::asn::asnBitString {bitstring} {
    # The bit string tag is 0x03.
    # Bit strings can be either simple or constructed
    # we always use simple encoding
    
    set bitlen [string length $bitstring]
    set padding [expr {(8 - ($bitlen % 8)) % 8}]
    set len [expr {($bitlen / 8) + 1}]
    if {$padding != 0} { incr len }

    return [binary format H2a*cB* 03 [asnLength $len] $padding $bitstring]    
}

#-----------------------------------------------------------------------------
# asnUTCTime : Encode an UTC time string
#-----------------------------------------------------------------------------

proc ::asn::asnUTCTime {UTCtimestring} {
    # the utc time tag is 0x17.
    # 
    # BUG: we do not check the string for well formedness
    
    set ascii [encoding convertto ascii $UTCtimestring]
    set len [string length $ascii]
    return [binary format H2a*a* 17 [asnLength $len] $ascii]
}

#-----------------------------------------------------------------------------
# asnPrintableString : Encode a printable string
#-----------------------------------------------------------------------------
namespace eval asn {
    variable nonPrintableChars {[^ A-Za-z0-9'()+,.:/?=-]}
}	
proc ::asn::asnPrintableString {string} {
    # the printable string tag is 0x13
    variable nonPrintableChars
    # it is basically a restricted ascii string
    if {[regexp $nonPrintableChars $string ]} {
        return -code error "Illegal character in PrintableString."
    }
    
    # check characters
    set ascii [encoding convertto ascii $string]
    return [asnEncodeString 13 $ascii]
}

#-----------------------------------------------------------------------------
# asnIA5String : Encode an Ascii String
#-----------------------------------------------------------------------------
proc ::asn::asnIA5String {string} {
    # the IA5 string tag is 0x16
    # check for extended charachers
    if {[string length $string]!=[string bytelength $string]} {
	return -code error "Illegal character in IA5String"
    }
    set ascii [encoding convertto ascii $string]
    return [asnEncodeString 16 $ascii]
}

#-----------------------------------------------------------------------------
# asnNumericString : Encode a Numeric String type
#-----------------------------------------------------------------------------
namespace eval asn {
    variable nonNumericChars {[^0-9 ]}
}
proc ::asn::asnNumericString {string} {
    # the Numeric String type has tag 0x12
    variable nonNumericChars
    if {[regexp $nonNumericChars $string]} {
        return -code error "Illegal character in Numeric String."
    }
    
    return [asnEncodeString 12 $string]
}
#----------------------------------------------------------------------
# asnBMPString: Encode a Tcl string as Basic Multinligval (UCS2) string
#-----------------------------------------------------------------------
proc asn::asnBMPString  {string} {
    if {$::tcl_platform(byteOrder) eq "littleEndian"} {
	set bytes ""
	foreach {lo hi} [split [encoding convertto unicode $string] ""] {
	    append bytes $hi $lo
	}	
    } else {
	set bytes [encoding convertto unicode $string]
    }
    return [asnEncodeString 1e $bytes]
}	
#---------------------------------------------------------------------------
# asnUTF8String: encode tcl string as UTF8 String
#----------------------------------------------------------------------------
proc asn::asnUTF8String {string} {
    return [asnEncodeString 0c [encoding convertto utf-8 $string]]
}
#-----------------------------------------------------------------------------
# asnEncodeString : Encode an RestrictedCharacter String
#-----------------------------------------------------------------------------
proc ::asn::asnEncodeString {tag string} {
    set len [string length $string]
    return [binary format H2a*a$len $tag [asnLength $len] $string]    
}

#-----------------------------------------------------------------------------
# asnObjectIdentifier : Encode an Object Identifier value
#-----------------------------------------------------------------------------
proc ::asn::asnObjectIdentifier {oid} {
    # the object identifier tag is 0x06
    
    if {[llength $oid] < 2} {
        return -code error "OID must have at least two subidentifiers."
    }
    
    # basic check that it is valid
    foreach identifier $oid {
        if {$identifier < 0} {
            return -code error \
		"Malformed OID. Identifiers must be positive Integers."
        }
    }
    
    if {[lindex $oid 0] > 2} {
            return -code error "First subidentifier must be 0,1 or 2"
    }
    if {[lindex $oid 1] > 39} {
            return -code error \
		"Second subidentifier must be between 0 and 39"
    }
    
    # handle the special cases directly
    switch [llength $oid] {
        2  {  return [binary format H2H2c 06 01 \
		[expr {[lindex $oid 0]*40+[lindex $oid 1]}]] }
        default {
              # This can probably be written much shorter. 
              # Just a first try that works...
              #
              set octets [binary format c \
		[expr {[lindex $oid 0]*40+[lindex $oid 1]}]]
              foreach identifier [lrange $oid 2 end] {
                  set d 128
                  if {$identifier < 128} {
                    set subidentifier [list $identifier]
                  } else {  
                    set subidentifier [list]
                    # find the largest divisor
                    
                    while {($identifier / $d) >= 128} { 
			set d [expr {$d * 128}] 
		    }
                    # and construct the subidentifiers
                    set remainder $identifier
                    while {$d >= 128} {
                        set coefficient [expr {($remainder / $d) | 0x80}]
                        set remainder [expr {$remainder % $d}]
                        set d [expr {$d / 128}]
                        lappend subidentifier $coefficient
                    }
                    lappend subidentifier $remainder
                  }
                  append octets [binary format c* $subidentifier]
              }
              return [binary format H2a*a* 06 \
		      [asnLength [string length $octets]] $octets]
        }
    }

}

#-----------------------------------------------------------------------------
# asnGetResponse : Read a ASN response from a channel.
#-----------------------------------------------------------------------------

proc ::asn::asnGetResponse {sock data_var} {
    upvar 1 $data_var data

    # We expect a sequence here (tag 0x30). The code below is an
    # inlined replica of 'asnGetSequence', modified for reading from a
    # channel instead of a string.

    set tag [read $sock 1]

    if {$tag == "\x30"} {
    # The following code is a replica of 'asnGetLength', modified
    # for reading the bytes from the channel instead of a string.

        set len1 [read $sock 1]
        binary scan $len1 c num
        set length [expr {($num + 0x100) % 0x100}]

        if {$length  >= 0x080} {
        # The byte the read is not the length, but a prefix, and
        # the lower nibble tells us how many bytes follow.

            set len_length  [expr {$length & 0x7f}]

        # BUG: We should not perform the value extraction for an
        # BUG: improper length. It wastes cycles, and here it can
        # BUG: cause us trouble, reading more data than there is
        # BUG: on the channel. Depending on the channel
        # BUG: configuration an attacker can induce us to block,
        # BUG: causing a denial of service.
            set lengthBytes [read $sock $len_length]

            switch $len_length {
                1 {
            binary scan $lengthBytes     c length 
            set length [expr {($length + 0x100) % 0x100}]
                }
                2 { binary scan $lengthBytes     S length }
                3 { binary scan \x00$lengthBytes I length }
                4 { binary scan $lengthBytes     I length }
                default {
                    return -code error \
			"length information too long ($len_length)"
                }
            }
        }

    # Now that the length is known we get the remainder,
    # i.e. payload, and construct proper in-memory BER encoded
    # sequence.

        set rest [read $sock $length]
        set data [binary format aa*a$length $tag [asnLength $length] $rest]
    }  else {
    # Generate an error message if the data is not a sequence as
    # we expected.

        set tag_hex ""
        binary scan $tag H2 tag_hex
        return -code error "unknown start tag [string length $tag] $tag_hex"
    }
}

if {[package vsatisfies [package present Tcl] 8.5.0]} {
##############################################################################
# Code for 8.5
##############################################################################
#-----------------------------------------------------------------------------
# asnGetByte (8.5 version) : Retrieve a single byte from the data (unsigned)
#-----------------------------------------------------------------------------

proc ::asn::asnGetByte {data_var byte_var} {
    upvar 1 $data_var data $byte_var byte
    
    binary scan [string index $data 0] cu byte
    set data [string range $data 1 end]

    return
}

#-----------------------------------------------------------------------------
# asnPeekByte (8.5 version) : Retrieve a single byte from the data (unsigned) 
#               without removing it.
#-----------------------------------------------------------------------------

proc ::asn::asnPeekByte {data_var byte_var {offset 0}} {
    upvar 1 $data_var data $byte_var byte
    
    binary scan [string index $data $offset] cu byte

    return
}

#-----------------------------------------------------------------------------
# asnGetLength (8.5 version) : Decode an ASN length value (See notes)
#-----------------------------------------------------------------------------

proc ::asn::asnGetLength {data_var length_var} {
    upvar 1 $data_var data  $length_var length

    asnGetByte data length
    if {$length == 0x080} {
        return -code error "Indefinite length BER encoding not yet supported"
    }
    if {$length > 0x080} {
    # The retrieved byte is a prefix value, and the integer in the
    # lower nibble tells us how many bytes were used to encode the
    # length data following immediately after this prefix.

        set len_length [expr {$length & 0x7f}]
        
        if {[string length $data] < $len_length} {
            return -code error \
		"length information invalid, not enough octets left" 
        }
        
        asnGetBytes data $len_length lengthBytes

        switch $len_length {
            1 { binary scan $lengthBytes     cu length }
            2 { binary scan $lengthBytes     Su length }
            3 { binary scan \x00$lengthBytes Iu length }
            4 { binary scan $lengthBytes     Iu length }
            default {                
                binary scan $lengthBytes H* hexstr
		scan $hexstr %llx length
            }
        }
    }
    return
}

} else {
##############################################################################
# Code for Tcl 8.4
##############################################################################
#-----------------------------------------------------------------------------
# asnGetByte : Retrieve a single byte from the data (unsigned)
#-----------------------------------------------------------------------------

proc ::asn::asnGetByte {data_var byte_var} {
    upvar 1 $data_var data $byte_var byte
    
    binary scan [string index $data 0] c byte
    set byte [expr {($byte + 0x100) % 0x100}]  
    set data [string range $data 1 end]

    return
}

#-----------------------------------------------------------------------------
# asnPeekByte : Retrieve a single byte from the data (unsigned) 
#               without removing it.
#-----------------------------------------------------------------------------

proc ::asn::asnPeekByte {data_var byte_var {offset 0}} {
    upvar 1 $data_var data $byte_var byte
    
    binary scan [string index $data $offset] c byte
    set byte [expr {($byte + 0x100) % 0x100}]  

    return
}

#-----------------------------------------------------------------------------
# asnGetLength : Decode an ASN length value (See notes)
#-----------------------------------------------------------------------------

proc ::asn::asnGetLength {data_var length_var} {
    upvar 1 $data_var data  $length_var length

    asnGetByte data length
    if {$length == 0x080} {
        return -code error "Indefinite length BER encoding not yet supported"
    }
    if {$length > 0x080} {
    # The retrieved byte is a prefix value, and the integer in the
    # lower nibble tells us how many bytes were used to encode the
    # length data following immediately after this prefix.

        set len_length [expr {$length & 0x7f}]
        
        if {[string length $data] < $len_length} {
            return -code error \
		"length information invalid, not enough octets left" 
        }
        
        asnGetBytes data $len_length lengthBytes

        switch $len_length {
            1 {
        # Efficiently coded data will not go through this
        # path, as small length values can be coded directly,
        # without a prefix.

            binary scan $lengthBytes     c length 
            set length [expr {($length + 0x100) % 0x100}]
            }
            2 { binary scan $lengthBytes     S length 
            set length [expr {($length + 0x10000) % 0x10000}]
            }
            3 { binary scan \x00$lengthBytes I length 
            set length [expr {($length + 0x1000000) % 0x1000000}]
            }
            4 { binary scan $lengthBytes     I length 
            set length [expr {(wide($length) + 0x100000000) % 0x100000000}]
            }
            default {                
                binary scan $lengthBytes H* hexstr
                # skip leading zeros which are allowed by BER
                set hexlen [string trimleft $hexstr 0] 
                # check if it fits into a 64-bit signed integer
                if {[string length $hexlen] > 16} {
                    return -code error -errorcode {ARITH IOVERFLOW 
                            {Length value too large for normal use, try asnGetBigLength}} \
			    "Length value to large"
                } elseif {  [string length $hexlen] == 16 \
			&& ([string index $hexlen 0] & 0x8)} { 
                    # check most significant bit, if set we need bignum
                    return -code error -errorcode {ARITH IOVERFLOW 
                            {Length value too large for normal use, try asnGetBigLength}} \
			    "Length value to large"
                } else {
                    scan $hexstr "%lx" length
                }
            }
        }
    }
    return
}

} 

#-----------------------------------------------------------------------------
# asnRetag: Remove an explicit tag with the real newTag
#
#-----------------------------------------------------------------------------
proc ::asn::asnRetag {data_var newTag} {
    upvar 1 $data_var data 
    set tag ""
    set type ""
    set len [asnPeekTag data tag type dummy]	
    asnGetBytes data $len tagbytes
    set data [binary format c* $newTag]$data
}

#-----------------------------------------------------------------------------
# asnGetBytes : Retrieve a block of 'length' bytes from the data.
#-----------------------------------------------------------------------------

proc ::asn::asnGetBytes {data_var length bytes_var} {
    upvar 1 $data_var data  $bytes_var bytes

    incr length -1
    set bytes [string range $data 0 $length]
    incr length
    set data [string range $data $length end]

    return
}

#-----------------------------------------------------------------------------
# asnPeekTag : Decode the tag value
#-----------------------------------------------------------------------------

proc ::asn::asnPeekTag {data_var tag_var tagtype_var constr_var} {
    upvar 1 $data_var data $tag_var tag $tagtype_var tagtype $constr_var constr
    
    set type 0	
    set offset 0
    asnPeekByte data type $offset
    # check if we have a simple tag, < 31, which fits in one byte
     
    set tval [expr {$type & 0x1f}]
    if {$tval == 0x1f} {
	# long tag, max 64-bit with Tcl 8.4, unlimited with 8.5 bignum
	asnPeekByte data tagbyte [incr offset]
	set tval [expr {wide($tagbyte & 0x7f)}]
	while {($tagbyte & 0x80)} {
	    asnPeekByte data tagbyte [incr offset] 
	    set tval [expr {($tval << 7) + ($tagbyte & 0x7f)}]
	}
    } 

    set tagtype [lindex {UNIVERSAL APPLICATION CONTEXT PRIVATE} \
	[expr {($type & 0xc0) >>6}]]
    set tag $tval
    set constr [expr {($type & 0x20) > 0}]

    return [incr offset]	
}

#-----------------------------------------------------------------------------
# asnTag : Build a tag value
#-----------------------------------------------------------------------------

proc ::asn::asnTag {tagnumber {class UNIVERSAL} {tagstyle P}} {
    set first 0
    if {$tagnumber < 31} {
	# encode everything in one byte
	set first $tagnumber	
	set bytes [list]
    } else {
	# multi-byte tag
	set first 31
	set bytes [list [expr {$tagnumber & 0x7f}]]
	set tagnumber [expr {$tagnumber >> 7}]
	while {$tagnumber > 0} {
	    lappend bytes [expr {($tagnumber & 0x7f)+0x80}]
	    set tagnumber [expr {$tagnumber >>7}]	
	}

    }
    
    if {$tagstyle eq "C" || $tagstyle == 1 } {incr first 32}
    switch -glob -- $class {
	U* {		    ;# UNIVERSAL } 
	A* { incr first 64  ;# APPLICATION }
	C* { incr first 128 ;# CONTEXT }
	P* { incr first 192 ;# PRIVATE }
	default {
	    return -code error "Unknown tag class \"$class\""
	}	
    }
    if {[llength $bytes] > 0} {
	# long tag
	set rbytes [list]
	for {set i [expr {[llength $bytes]-1}]} {$i >= 0} {incr i -1} {
	    lappend rbytes [lindex $bytes $i]
	}
	return [binary format cc* $first $rbytes ]
    } 
    return [binary format c $first]
}



#-----------------------------------------------------------------------------
# asnGetBigLength : Retrieve a length that can not be represented in 63-bit
#-----------------------------------------------------------------------------

proc ::asn::asnGetBigLength {data_var biglength_var} {

    # Does any real world code really need this? 
    # If we encounter this, we are doomed to fail anyway, 
    # (there would be an Exabyte inside the data_var, )
    #
    # So i implement it just for completeness.
    # 
    package require math::bignum
    
    upvar 1 $data_var data  $biglength_var length

    asnGetByte data length
    if {$length == 0x080} {
        return -code error "Indefinite length BER encoding not yet supported"
    }
    if {$length > 0x080} {
    # The retrieved byte is a prefix value, and the integer in the
    # lower nibble tells us how many bytes were used to encode the
    # length data following immediately after this prefix.

        set len_length [expr {$length & 0x7f}]
        
        if {[string length $data] < $len_length} {
            return -code error \
		"length information invalid, not enough octets left" 
        }
        
        asnGetBytes data $len_length lengthBytes
        binary scan $lengthBytes H* hexlen
        set length [math::bignum::fromstr $hexlen 16]
    }
    return
}

#-----------------------------------------------------------------------------
# asnGetInteger : Retrieve integer.
#-----------------------------------------------------------------------------

proc ::asn::asnGetInteger {data_var int_var} {
    # Tag is 0x02. 

    upvar 1 $data_var data $int_var int

    asnGetByte   data tag

    if {$tag != 0x02} {
        return -code error \
            [format "Expected Integer (0x02), but got %02x" $tag]
    }

    asnGetLength data len
    asnGetBytes  data $len integerBytes

    set int ?

    switch $len {
        1 { binary scan $integerBytes     c int }
        2 { binary scan $integerBytes     S int }
        3 { 
            # check for negative int and pad 
            scan [string index $integerBytes 0] %c byte
            if {$byte & 128} {
                binary scan \xff$integerBytes I int
            } else {
                binary scan \x00$integerBytes I int 
            }
          }
        4 { binary scan $integerBytes     I int }
        5 -
        6 -
        7 -
        8 {
            # check for negative int and pad
            scan [string index $integerBytes 0] %c byte
            if {$byte & 128} {
                set pad [string repeat \xff [expr {8-$len}]]
            } else {
                set pad [string repeat \x00 [expr {8-$len}]]
            }
            binary scan $pad$integerBytes W int 
        }
        default {
        # Too long, or prefix coding was used.
            return -code error "length information too long"
        }
    }
    return
}

#-----------------------------------------------------------------------------
# asnGetBigInteger : Retrieve a big integer.
#-----------------------------------------------------------------------------

proc ::asn::asnGetBigInteger {data_var bignum_var} {
	# require math::bignum only if it is used
	package require math::bignum

	# Tag is 0x02. We expect that the length of the integer is coded with
	# maximal efficiency, i.e. without a prefix 0x81 prefix. If a prefix
	# is used this decoder will fail.

	upvar $data_var data $bignum_var bignum

	asnGetByte   data tag

	if {$tag != 0x02} {
		return -code error \
			[format "Expected Integer (0x02), but got %02x" $tag]
	}

	asnGetLength data len
	asnGetBytes  data $len integerBytes

	binary scan [string index $integerBytes 0] H* hex_head
	set head [expr 0x$hex_head]
	set replacement_head [expr {$head & 0x7f}]
	set integerBytes [string replace $integerBytes 0 0 [format %c $replacement_head]]

	binary scan $integerBytes H* hex

	set bignum [math::bignum::fromstr $hex 16]

	if {($head >> 7) && 1} {
		set bigsub [math::bignum::pow [::math::bignum::fromstr 2] [::math::bignum::fromstr [expr {($len * 8) - 1}]]]
		set bignum [math::bignum::sub $bignum $bigsub]
	}

	return $bignum
}




#-----------------------------------------------------------------------------
# asnGetEnumeration : Retrieve an enumeration id
#-----------------------------------------------------------------------------

proc ::asn::asnGetEnumeration {data_var enum_var} {
    # This is like 'asnGetInteger', except for a different tag.

    upvar 1 $data_var data $enum_var enum

    asnGetByte   data tag

    if {$tag != 0x0a} {
        return -code error \
            [format "Expected Enumeration (0x0a), but got %02x" $tag]
    }

    asnGetLength data len
    asnGetBytes  data $len integerBytes
    set enum ?

    switch $len {
        1 { binary scan $integerBytes     c enum }
        2 { binary scan $integerBytes     S enum }
        3 { binary scan \x00$integerBytes I enum }
        4 { binary scan $integerBytes     I enum }
        default {
            return -code error "length information too long"
        }
    }
    return
}

#-----------------------------------------------------------------------------
# asnGetOctetString : Retrieve arbitrary string.
#-----------------------------------------------------------------------------

proc ::asn::asnGetOctetString {data_var string_var} {
    # Here we need the full decoder for length data.

    upvar 1 $data_var data $string_var string
    
    asnGetByte data tag
    if {$tag != 0x04} { 
        return -code error \
            [format "Expected Octet String (0x04), but got %02x" $tag]
    }
    asnGetLength data length
    asnGetBytes  data $length temp
    set string $temp
    return
}

#-----------------------------------------------------------------------------
# asnGetSequence : Retrieve Sequence data for further decoding.
#-----------------------------------------------------------------------------

proc ::asn::asnGetSequence {data_var sequence_var} {
    # Here we need the full decoder for length data.

    upvar 1 $data_var data $sequence_var sequence

    asnGetByte data tag
    if {$tag != 0x030} { 
        return -code error \
            [format "Expected Sequence (0x30), but got %02x" $tag]
    }    
    asnGetLength data length
    asnGetBytes  data $length temp
    set sequence $temp
    return
}

#-----------------------------------------------------------------------------
# asnGetSet : Retrieve Set data for further decoding.
#-----------------------------------------------------------------------------

proc ::asn::asnGetSet {data_var set_var} {
    # Here we need the full decoder for length data.

    upvar 1 $data_var data $set_var set

    asnGetByte data tag
    if {$tag != 0x031} { 
        return -code error \
            [format "Expected Set (0x31), but got %02x" $tag]
    }    
    asnGetLength data length
    asnGetBytes  data $length temp
    set set $temp
    return
}

#-----------------------------------------------------------------------------
# asnGetApplication
#-----------------------------------------------------------------------------

proc ::asn::asnGetApplication {data_var appNumber_var {content_var {}} {encodingType_var {}} } {
    upvar 1 $data_var data $appNumber_var appNumber

    asnGetByte   data tag
    asnGetLength data length

    if {($tag & 0xC0) != 0x40} {
        return -code error \
            [format "Expected Application, but got %02x" $tag]
    }    
    if {$encodingType_var != {}} {
	upvar 1 $encodingType_var encodingType
	set encodingType [expr {($tag & 0x20) > 0}]
    }
    set appNumber [expr {$tag & 0x1F}]
	if {[string length $content_var]} {
		upvar 1 $content_var content
		asnGetBytes data $length content
	}	
    return
}

#-----------------------------------------------------------------------------
# asnGetBoolean: decode a boolean value
#-----------------------------------------------------------------------------

proc asn::asnGetBoolean {data_var bool_var} {
    upvar 1 $data_var data $bool_var bool

    asnGetByte data tag
    if {$tag != 0x01} {
        return -code error \
            [format "Expected Boolean (0x01), but got %02x" $tag]
    }

    asnGetLength data length
    asnGetByte data byte
    set bool [expr {$byte == 0 ? 0 : 1}]    
    return
}

#-----------------------------------------------------------------------------
# asnGetUTCTime: Extract an UTC Time string from the data. Returns a string
#                representing an UTC Time.
#
#-----------------------------------------------------------------------------

proc asn::asnGetUTCTime {data_var utc_var} {
    upvar 1 $data_var data $utc_var utc

    asnGetByte data tag
    if {$tag != 0x17} {
        return -code error \
            [format "Expected UTCTime (0x17), but got %02x" $tag]
    }

    asnGetLength data length
    asnGetBytes data $length bytes
    
    # this should be ascii, make it explicit
    set bytes [encoding convertfrom ascii $bytes]
    binary scan $bytes a* utc
    
    return
}


#-----------------------------------------------------------------------------
# asnGetBitString: Extract a Bit String value (a string of 0/1s) from the
#                  ASN.1 data.
#
#-----------------------------------------------------------------------------

proc asn::asnGetBitString {data_var bitstring_var} {
    upvar 1 $data_var data $bitstring_var bitstring

    asnGetByte data tag
    if {$tag != 0x03} {
        return -code error \
            [format "Expected Bit String (0x03), but got %02x" $tag]
    }
    
    asnGetLength data length
    # get the number of padding bits used at the end
    asnGetByte data padding
    incr length -1
    asnGetBytes data $length bytes
    binary scan $bytes B* bits
    
    # cut off the padding bits
    set bits [string range $bits 0 end-$padding]
    set bitstring $bits
}

#-----------------------------------------------------------------------------
# asnGetObjectIdentifier: Decode an ASN.1 Object Identifier (OID) into
#                         a Tcl list of integers.
#-----------------------------------------------------------------------------

proc asn::asnGetObjectIdentifier {data_var oid_var} {
      upvar 1 $data_var data $oid_var oid

      asnGetByte data tag
      if {$tag != 0x06} {
        return -code error \
            [format "Expected Object Identifier (0x06), but got %02x" $tag]  
      }
      asnGetLength data length
      
      # the first byte encodes the OID parts in position 0 and 1
      asnGetByte data val
      set oid [expr {$val / 40}]
      lappend oid [expr {$val % 40}]
      incr length -1
      
      # the next bytes encode the remaining parts of the OID
      set bytes [list]
      set incomplete 0
      while {$length} {
        asnGetByte data octet
        incr length -1
        if {$octet < 128} {
            set oidval $octet
            set mult 128
            foreach byte $bytes {
                if {$byte != {}} {
                incr oidval [expr {$mult*$byte}]    
                set mult [expr {$mult*128}]
                }
            }
            lappend oid $oidval
            set bytes [list]
            set incomplete 0
        } else {
            set byte [expr {$octet-128}]
            set bytes [concat [list $byte] $bytes]
            set incomplete 1
        }                      
      }
      if {$incomplete} {
        return -code error "OID Data is incomplete, not enough octets."
      }
      return
}

#-----------------------------------------------------------------------------
# asnGetContext: Decode an explicit context tag 
#
#-----------------------------------------------------------------------------

proc ::asn::asnGetContext {data_var contextNumber_var {content_var {}} {encodingType_var {}}} {
    upvar 1 $data_var data $contextNumber_var contextNumber 
    
    asnGetByte   data tag
    asnGetLength data length

    if {($tag & 0xC0) != 0x80} {
        return -code error \
            [format "Expected Context, but got %02x" $tag]
    }    
    if {$encodingType_var != {}} { 
	upvar 1 $encodingType_var encodingType 
	set encodingType [expr {($tag & 0x20) > 0}]
    }
    set contextNumber [expr {$tag & 0x1F}]
	if {[string length $content_var]} {
		upvar 1 $content_var content
		asnGetBytes data $length content
	}	
    return
}


#-----------------------------------------------------------------------------
# asnGetNumericString: Decode a Numeric String from the data
#-----------------------------------------------------------------------------

proc ::asn::asnGetNumericString {data_var print_var} {
    upvar 1 $data_var data $print_var print

    asnGetByte data tag
    if {$tag != 0x12} {
        return -code error \
            [format "Expected Numeric String (0x12), but got %02x" $tag]  
    }
    asnGetLength data length 
    asnGetBytes data $length string
    set print [encoding convertfrom ascii $string]
    return
}

#-----------------------------------------------------------------------------
# asnGetPrintableString: Decode a Printable String from the data
#-----------------------------------------------------------------------------

proc ::asn::asnGetPrintableString {data_var print_var} {
    upvar 1 $data_var data $print_var print

    asnGetByte data tag
    if {$tag != 0x13} {
        return -code error \
            [format "Expected Printable String (0x13), but got %02x" $tag]  
    }
    asnGetLength data length 
    asnGetBytes data $length string
    set print [encoding convertfrom ascii $string]
    return
}

#-----------------------------------------------------------------------------
# asnGetIA5String: Decode a IA5(ASCII) String from the data
#-----------------------------------------------------------------------------

proc ::asn::asnGetIA5String {data_var print_var} {
    upvar 1 $data_var data $print_var print

    asnGetByte data tag
    if {$tag != 0x16} {
        return -code error \
            [format "Expected IA5 String (0x16), but got %02x" $tag]  
    }
    asnGetLength data length 
    asnGetBytes data $length string
    set print [encoding convertfrom ascii $string]
    return
}
#------------------------------------------------------------------------
# asnGetBMPString: Decode Basic Multiningval (UCS2 string) from data
#------------------------------------------------------------------------
proc asn::asnGetBMPString {data_var print_var} {
    upvar 1 $data_var data $print_var print
    asnGetByte data tag
    if {$tag != 0x1e} {
        return -code error \
            [format "Expected BMP String (0x1e), but got %02x" $tag]  
    }
    asnGetLength data length 
	asnGetBytes data $length string
	if {$::tcl_platform(byteOrder) eq "littleEndian"} {
		set str2 ""
		foreach {hi lo} [split $string ""] {
			append str2 $lo $hi
		}
	} else {
		set str2 $string
	}
	set print [encoding convertfrom unicode $str2]
	return
}	
#------------------------------------------------------------------------
# asnGetUTF8String: Decode UTF8 string from data
#------------------------------------------------------------------------
proc asn::asnGetUTF8String {data_var print_var} {
    upvar 1 $data_var data $print_var print
    asnGetByte data tag
    if {$tag != 0x0c} {
        return -code error \
            [format "Expected UTF8 String (0x0c), but got %02x" $tag]  
    }
    asnGetLength data length 
	asnGetBytes data $length string
	#there should be some error checking to see if input is
	#properly-formatted utf8
	set print [encoding convertfrom utf-8 $string]
	
	return
}	
#-----------------------------------------------------------------------------
# asnGetNull: decode a NULL value
#-----------------------------------------------------------------------------

proc ::asn::asnGetNull {data_var} {
    upvar 1 $data_var data 

    asnGetByte data tag
    if {$tag != 0x05} {
        return -code error \
            [format "Expected NULL (0x05), but got %02x" $tag]
    }

    asnGetLength data length
    asnGetBytes data $length bytes
    
    # we do not check the null data, all bytes must be 0x00
    
    return
}

#----------------------------------------------------------------------------
# MultiType string routines
#----------------------------------------------------------------------------

namespace eval asn {
	variable stringTypes
	array set stringTypes {
		12 NumericString 
		13 PrintableString 
		16 IA5String 
		1e BMPString 
		0c UTF8String 
		14 T61String
		15 VideotexString
		1a VisibleString
		1b GeneralString
		1c UniversalString
	}	
	variable defaultStringType UTF8
}	
#---------------------------------------------------------------------------
# asnGetString - get readable string automatically detecting its type
#---------------------------------------------------------------------------
proc ::asn::asnGetString {data_var print_var {type_var {}}} {
	variable stringTypes
	upvar 1 $data_var data $print_var print
	asnPeekByte data tag
	set tag [format %02x $tag]
	if {![info exists stringTypes($tag)]} {
		return -code error "Expected one of string types, but got $tag"
	}
	asnGet$stringTypes($tag) data print
	if {[string length $type_var]} {
		upvar $type_var type
		set type $stringTypes($tag)
	}	
}
#---------------------------------------------------------------------
# defaultStringType - set or query default type for unrestricted strings
#---------------------------------------------------------------------
proc ::asn::defaultStringType {{type {}}} {
	variable defaultStringType
	if {![string length $type]} {
		return $defaultStringType
	}
	if {$type ne "BMP" && $type ne "UTF8"} {
		return -code error "Invalid default string type. Should be one of BMP, UTF8"
	}
	set defaultStringType $type
	return
}	

#---------------------------------------------------------------------------
# asnString - encode readable string into most restricted type possible
#---------------------------------------------------------------------------

proc ::asn::asnString {string} {
	variable nonPrintableChars
	variable nonNumericChars
	if {[string length $string]!=[string bytelength $string]} {
	# There are non-ascii character
		variable defaultStringType
		return [asn${defaultStringType}String $string]
	} elseif {![regexp $nonNumericChars $string]} {
		return [asnNumericString $string]
	} elseif {![regexp $nonPrintableChars $string]} {
		return [asnPrintableString $string]
	} else {
		return [asnIA5String $string]
	}	
}

#-----------------------------------------------------------------------------
package provide asn 0.8.4

# aes.tcl - 
#
# Copyright (c) 2005 Thorsten Schloermann
# Copyright (c) 2005 Pat Thoyts <patthoyts@users.sourceforge.net>
# Copyright (c) 2013 Andreas Kupries
#
# A Tcl implementation of the Advanced Encryption Standard (US FIPS PUB 197)
#
# AES is a block cipher with a block size of 128 bits and a variable
# key size of 128, 192 or 256 bits.
# The algorithm works on each block as a 4x4 state array. There are 4 steps
# in each round:
#   SubBytes    a non-linear substitution step using a predefined S-box
#   ShiftRows   cyclic transposition of rows in the state matrix
#   MixColumns  transformation upon columns in the state matrix
#   AddRoundKey application of round specific sub-key
#
# -------------------------------------------------------------------------
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# -------------------------------------------------------------------------

package require Tcl 8.5

namespace eval ::aes {
    variable uid
    if {![info exists uid]} { set uid 0 }

    namespace export aes

    # constants

    # S-box
    variable sbox {
        0x63 0x7c 0x77 0x7b 0xf2 0x6b 0x6f 0xc5 0x30 0x01 0x67 0x2b 0xfe 0xd7 0xab 0x76
        0xca 0x82 0xc9 0x7d 0xfa 0x59 0x47 0xf0 0xad 0xd4 0xa2 0xaf 0x9c 0xa4 0x72 0xc0
        0xb7 0xfd 0x93 0x26 0x36 0x3f 0xf7 0xcc 0x34 0xa5 0xe5 0xf1 0x71 0xd8 0x31 0x15
        0x04 0xc7 0x23 0xc3 0x18 0x96 0x05 0x9a 0x07 0x12 0x80 0xe2 0xeb 0x27 0xb2 0x75
        0x09 0x83 0x2c 0x1a 0x1b 0x6e 0x5a 0xa0 0x52 0x3b 0xd6 0xb3 0x29 0xe3 0x2f 0x84
        0x53 0xd1 0x00 0xed 0x20 0xfc 0xb1 0x5b 0x6a 0xcb 0xbe 0x39 0x4a 0x4c 0x58 0xcf
        0xd0 0xef 0xaa 0xfb 0x43 0x4d 0x33 0x85 0x45 0xf9 0x02 0x7f 0x50 0x3c 0x9f 0xa8
        0x51 0xa3 0x40 0x8f 0x92 0x9d 0x38 0xf5 0xbc 0xb6 0xda 0x21 0x10 0xff 0xf3 0xd2
        0xcd 0x0c 0x13 0xec 0x5f 0x97 0x44 0x17 0xc4 0xa7 0x7e 0x3d 0x64 0x5d 0x19 0x73
        0x60 0x81 0x4f 0xdc 0x22 0x2a 0x90 0x88 0x46 0xee 0xb8 0x14 0xde 0x5e 0x0b 0xdb
        0xe0 0x32 0x3a 0x0a 0x49 0x06 0x24 0x5c 0xc2 0xd3 0xac 0x62 0x91 0x95 0xe4 0x79
        0xe7 0xc8 0x37 0x6d 0x8d 0xd5 0x4e 0xa9 0x6c 0x56 0xf4 0xea 0x65 0x7a 0xae 0x08
        0xba 0x78 0x25 0x2e 0x1c 0xa6 0xb4 0xc6 0xe8 0xdd 0x74 0x1f 0x4b 0xbd 0x8b 0x8a
        0x70 0x3e 0xb5 0x66 0x48 0x03 0xf6 0x0e 0x61 0x35 0x57 0xb9 0x86 0xc1 0x1d 0x9e
        0xe1 0xf8 0x98 0x11 0x69 0xd9 0x8e 0x94 0x9b 0x1e 0x87 0xe9 0xce 0x55 0x28 0xdf
        0x8c 0xa1 0x89 0x0d 0xbf 0xe6 0x42 0x68 0x41 0x99 0x2d 0x0f 0xb0 0x54 0xbb 0x16
    }
    # inverse S-box
    variable xobs {
        0x52 0x09 0x6a 0xd5 0x30 0x36 0xa5 0x38 0xbf 0x40 0xa3 0x9e 0x81 0xf3 0xd7 0xfb
        0x7c 0xe3 0x39 0x82 0x9b 0x2f 0xff 0x87 0x34 0x8e 0x43 0x44 0xc4 0xde 0xe9 0xcb
        0x54 0x7b 0x94 0x32 0xa6 0xc2 0x23 0x3d 0xee 0x4c 0x95 0x0b 0x42 0xfa 0xc3 0x4e
        0x08 0x2e 0xa1 0x66 0x28 0xd9 0x24 0xb2 0x76 0x5b 0xa2 0x49 0x6d 0x8b 0xd1 0x25
        0x72 0xf8 0xf6 0x64 0x86 0x68 0x98 0x16 0xd4 0xa4 0x5c 0xcc 0x5d 0x65 0xb6 0x92
        0x6c 0x70 0x48 0x50 0xfd 0xed 0xb9 0xda 0x5e 0x15 0x46 0x57 0xa7 0x8d 0x9d 0x84
        0x90 0xd8 0xab 0x00 0x8c 0xbc 0xd3 0x0a 0xf7 0xe4 0x58 0x05 0xb8 0xb3 0x45 0x06
        0xd0 0x2c 0x1e 0x8f 0xca 0x3f 0x0f 0x02 0xc1 0xaf 0xbd 0x03 0x01 0x13 0x8a 0x6b
        0x3a 0x91 0x11 0x41 0x4f 0x67 0xdc 0xea 0x97 0xf2 0xcf 0xce 0xf0 0xb4 0xe6 0x73
        0x96 0xac 0x74 0x22 0xe7 0xad 0x35 0x85 0xe2 0xf9 0x37 0xe8 0x1c 0x75 0xdf 0x6e
        0x47 0xf1 0x1a 0x71 0x1d 0x29 0xc5 0x89 0x6f 0xb7 0x62 0x0e 0xaa 0x18 0xbe 0x1b
        0xfc 0x56 0x3e 0x4b 0xc6 0xd2 0x79 0x20 0x9a 0xdb 0xc0 0xfe 0x78 0xcd 0x5a 0xf4
        0x1f 0xdd 0xa8 0x33 0x88 0x07 0xc7 0x31 0xb1 0x12 0x10 0x59 0x27 0x80 0xec 0x5f
        0x60 0x51 0x7f 0xa9 0x19 0xb5 0x4a 0x0d 0x2d 0xe5 0x7a 0x9f 0x93 0xc9 0x9c 0xef
        0xa0 0xe0 0x3b 0x4d 0xae 0x2a 0xf5 0xb0 0xc8 0xeb 0xbb 0x3c 0x83 0x53 0x99 0x61
        0x17 0x2b 0x04 0x7e 0xba 0x77 0xd6 0x26 0xe1 0x69 0x14 0x63 0x55 0x21 0x0c 0x7d
    }
}

# aes::Init --
#
#	Initialise our AES state and calculate the key schedule. An initialization
#	vector is maintained in the state for modes that require one. The key must
#	be binary data of the correct size and the IV must be 16 bytes.
#
#	Nk: columns of the key-array
#	Nr: number of rounds (depends on key-length)
#	Nb: columns of the text-block, is always 4 in AES
#
proc ::aes::Init {mode key iv} {
    switch -exact -- $mode {
        ecb - cbc { }
        cfb - ofb {
            return -code error "$mode mode not implemented"
        }
        default {
            return -code error "invalid mode \"$mode\":\
                must be one of ecb or cbc."
        }
    }

    set size [expr {[string length $key] << 3}]
    switch -exact -- $size {
        128 {set Nk 4; set Nr 10; set Nb 4}
        192 {set Nk 6; set Nr 12; set Nb 4}
        256 {set Nk 8; set Nr 14; set Nb 4}
        default {
            return -code error "invalid key size \"$size\":\
                must be one of 128, 192 or 256."
        }
    }

    variable uid
    set Key [namespace current]::[incr uid]
    upvar #0 $Key state
    if {[binary scan $iv Iu4 state(I)] != 1} {
        return -code error "invalid initialization vector: must be 16 bytes"
    }
    array set state [list M $mode K $key Nk $Nk Nr $Nr Nb $Nb W {}]
    ExpandKey $Key
    return $Key
}

# aes::Reset --
#
#	Reset the initialization vector for the specified key. This permits the
#	key to be reused for encryption or decryption without the expense of
#	re-calculating the key schedule.
#
proc ::aes::Reset {Key iv} {
    upvar #0 $Key state
    if {[binary scan $iv Iu4 state(I)] != 1} {
        return -code error "invalid initialization vector: must be 16 bytes"
    }
    return
}
    
# aes::Final --
#
#	Clean up the key state
#
proc ::aes::Final {Key} {
    # FRINK: nocheck
    unset $Key
}

# -------------------------------------------------------------------------

# 5.1 Cipher:  Encipher a single block of 128 bits.
proc ::aes::EncryptBlock {Key block} {
    upvar #0 $Key state
    if {[binary scan $block Iu4 data] != 1} {
        return -code error "invalid block size: blocks must be 16 bytes"
    }

    if {$state(M) eq {cbc}} {
        # Loop unrolled.
        lassign $data     d0 d1 d2 d3
        lassign $state(I) s0 s1 s2 s3
        set data [list \
                      [expr {$d0 ^ $s0}] \
                      [expr {$d1 ^ $s1}] \
                      [expr {$d2 ^ $s2}] \
                      [expr {$d3 ^ $s3}] ]
    }

    set data [AddRoundKey $Key 0 $data]
    for {set n 1} {$n < $state(Nr)} {incr n} {
        set data [AddRoundKey $Key $n [MixColumns [ShiftRows [SubBytes $data]]]]
    }
    set data [AddRoundKey $Key $n [ShiftRows [SubBytes $data]]]

    # Bug 2993029:
    # Force all elements of data into the 32bit range.
    # Loop unrolled
    set res [Clamp32 $data]

    set state(I) $res
    binary format Iu4 $res
}

# 5.3: Inverse Cipher: Decipher a single 128 bit block.
proc ::aes::DecryptBlock {Key block} {
    upvar #0 $Key state
    if {[binary scan $block Iu4 data] != 1} {
        return -code error "invalid block size: block must be 16 bytes"
    }
    set iv $data

    set n $state(Nr)
    set data [AddRoundKey $Key $state(Nr) $data]
    for {incr n -1} {$n > 0} {incr n -1} {
        set data [InvMixColumns [AddRoundKey $Key $n [InvSubBytes [InvShiftRows $data]]]]
    }
    set data [AddRoundKey $Key $n [InvSubBytes [InvShiftRows $data]]]
    
    if {$state(M) eq {cbc}} {
        lassign $data     d0 d1 d2 d3
        lassign $state(I) s0 s1 s2 s3
        set data [list \
                      [expr {($d0 ^ $s0) & 0xffffffff}] \
                      [expr {($d1 ^ $s1) & 0xffffffff}] \
                      [expr {($d2 ^ $s2) & 0xffffffff}] \
                      [expr {($d3 ^ $s3) & 0xffffffff}] ]
    } else {
        # Bug 2993029:
        # The integrated clamping we see above only happens for CBC mode.
        set data [Clamp32 $data]
    }

    set state(I) $iv
    binary format Iu4 $data
}

proc ::aes::Clamp32 {data} {
    # Force all elements into 32bit range.
    lassign $data d0 d1 d2 d3
    list \
        [expr {$d0 & 0xffffffff}] \
        [expr {$d1 & 0xffffffff}] \
        [expr {$d2 & 0xffffffff}] \
        [expr {$d3 & 0xffffffff}]
}

# 5.2: KeyExpansion
proc ::aes::ExpandKey {Key} {
    upvar #0 $Key state
    set Rcon [list 0x00000000 0x01000000 0x02000000 0x04000000 0x08000000 \
                   0x10000000 0x20000000 0x40000000 0x80000000 0x1b000000 \
                   0x36000000 0x6c000000 0xd8000000 0xab000000 0x4d000000]
    # Split the key into Nk big-endian words
    binary scan $state(K) I* W
    set max [expr {$state(Nb) * ($state(Nr) + 1)}]
    set i $state(Nk)
    set h [expr {$i - 1}]
    set j 0
    for {} {$i < $max} {incr i; incr h; incr j} {
        set temp [lindex $W $h]
        if {($i % $state(Nk)) == 0} {
            set sub [SubWord [RotWord $temp]]
            set rc [lindex $Rcon [expr {$i/$state(Nk)}]]
            set temp [expr {$sub ^ $rc}]
        } elseif {$state(Nk) > 6 && ($i % $state(Nk)) == 4} { 
            set temp [SubWord $temp]
        }
        lappend W [expr {[lindex $W $j] ^ $temp}]
    }
    set state(W) $W
}

# 5.2: Key Expansion: Apply S-box to each byte in the 32 bit word
proc ::aes::SubWord {w} {
    variable sbox
    set s3 [lindex $sbox [expr {($w >> 24) & 255}]]
    set s2 [lindex $sbox [expr {($w >> 16) & 255}]]
    set s1 [lindex $sbox [expr {($w >> 8 ) & 255}]]
    set s0 [lindex $sbox [expr { $w        & 255}]]
    return [expr {($s3 << 24) | ($s2 << 16) | ($s1 << 8) | $s0}]
}

proc ::aes::InvSubWord {w} {
    variable xobs
    set s3 [lindex $xobs [expr {($w >> 24) & 255}]]
    set s2 [lindex $xobs [expr {($w >> 16) & 255}]]
    set s1 [lindex $xobs [expr {($w >> 8 ) & 255}]]
    set s0 [lindex $xobs [expr { $w        & 255}]]
    return [expr {($s3 << 24) | ($s2 << 16) | ($s1 << 8) | $s0}]
}

# 5.2: Key Expansion: Rotate a 32bit word by 8 bits
proc ::aes::RotWord {w} {
    return [expr {(($w << 8) | (($w >> 24) & 0xff)) & 0xffffffff}]
}

# 5.1.1: SubBytes() Transformation
proc ::aes::SubBytes {words} {
    lassign $words w0 w1 w2 w3
    list [SubWord $w0] [SubWord $w1] [SubWord $w2] [SubWord $w3]
}

# 5.3.2: InvSubBytes() Transformation
proc ::aes::InvSubBytes {words} {
    lassign $words w0 w1 w2 w3
    list [InvSubWord $w0] [InvSubWord $w1] [InvSubWord $w2] [InvSubWord $w3]
}

# 5.1.2: ShiftRows() Transformation
proc ::aes::ShiftRows {words} {
    for {set n0 0} {$n0 < 4} {incr n0} {
        set n1 [expr {($n0 + 1) % 4}]
        set n2 [expr {($n0 + 2) % 4}]
        set n3 [expr {($n0 + 3) % 4}]
        lappend r [expr {(  [lindex $words $n0] & 0xff000000)
                         | ([lindex $words $n1] & 0x00ff0000)
                         | ([lindex $words $n2] & 0x0000ff00)
                         | ([lindex $words $n3] & 0x000000ff)
                     }]
    }
    return $r
}


# 5.3.1: InvShiftRows() Transformation
proc ::aes::InvShiftRows {words} {
    for {set n0 0} {$n0 < 4} {incr n0} {
        set n1 [expr {($n0 + 1) % 4}]
        set n2 [expr {($n0 + 2) % 4}]
        set n3 [expr {($n0 + 3) % 4}]
        lappend r [expr {(  [lindex $words $n0] & 0xff000000)
                         | ([lindex $words $n3] & 0x00ff0000)
                         | ([lindex $words $n2] & 0x0000ff00)
                         | ([lindex $words $n1] & 0x000000ff)
                     }]
    }
    return $r
}

# 5.1.3: MixColumns() Transformation
proc ::aes::MixColumns {words} {
    set r {}
    foreach w $words {
        set r0 [expr {(($w >> 24) & 255)}]
        set r1 [expr {(($w >> 16) & 255)}]
        set r2 [expr {(($w >> 8 ) & 255)}]
        set r3 [expr {( $w        & 255)}]

        set s0 [expr {[GFMult2 $r0] ^ [GFMult3 $r1] ^ $r2 ^ $r3}]
        set s1 [expr {$r0 ^ [GFMult2 $r1] ^ [GFMult3 $r2] ^ $r3}]
        set s2 [expr {$r0 ^ $r1 ^ [GFMult2 $r2] ^ [GFMult3 $r3]}]
        set s3 [expr {[GFMult3 $r0] ^ $r1 ^ $r2 ^ [GFMult2 $r3]}]

        lappend r [expr {($s0 << 24) | ($s1 << 16) | ($s2 << 8) | $s3}]
    }
    return $r
}

# 5.3.3: InvMixColumns() Transformation
proc ::aes::InvMixColumns {words} {
    set r {}
    foreach w $words {
        set r0 [expr {(($w >> 24) & 255)}]
        set r1 [expr {(($w >> 16) & 255)}]
        set r2 [expr {(($w >> 8 ) & 255)}]
        set r3 [expr {( $w        & 255)}]

        set s0 [expr {[GFMult0e $r0] ^ [GFMult0b $r1] ^ [GFMult0d $r2] ^ [GFMult09 $r3]}]
        set s1 [expr {[GFMult09 $r0] ^ [GFMult0e $r1] ^ [GFMult0b $r2] ^ [GFMult0d $r3]}]
        set s2 [expr {[GFMult0d $r0] ^ [GFMult09 $r1] ^ [GFMult0e $r2] ^ [GFMult0b $r3]}]
        set s3 [expr {[GFMult0b $r0] ^ [GFMult0d $r1] ^ [GFMult09 $r2] ^ [GFMult0e $r3]}]

        lappend r [expr {($s0 << 24) | ($s1 << 16) | ($s2 << 8) | $s3}]
    }
    return $r
}

# 5.1.4: AddRoundKey() Transformation
proc ::aes::AddRoundKey {Key round words} {
    upvar #0 $Key state
    set r {}
    set n [expr {$round * $state(Nb)}]
    foreach w $words {
        lappend r [expr {$w ^ [lindex $state(W) $n]}]
        incr n
    }
    return $r
}
    
# -------------------------------------------------------------------------
# ::aes::GFMult*
#
#	some needed functions for multiplication in a Galois-field
#
proc ::aes::GFMult2 {number} {
    # this is a tabular representation of xtime (multiplication by 2)
    # it is used instead of calculation to prevent timing attacks
    set xtime {
        0x00 0x02 0x04 0x06 0x08 0x0a 0x0c 0x0e 0x10 0x12 0x14 0x16 0x18 0x1a 0x1c 0x1e
        0x20 0x22 0x24 0x26 0x28 0x2a 0x2c 0x2e 0x30 0x32 0x34 0x36 0x38 0x3a 0x3c 0x3e 
        0x40 0x42 0x44 0x46 0x48 0x4a 0x4c 0x4e 0x50 0x52 0x54 0x56 0x58 0x5a 0x5c 0x5e
        0x60 0x62 0x64 0x66 0x68 0x6a 0x6c 0x6e 0x70 0x72 0x74 0x76 0x78 0x7a 0x7c 0x7e 
        0x80 0x82 0x84 0x86 0x88 0x8a 0x8c 0x8e 0x90 0x92 0x94 0x96 0x98 0x9a 0x9c 0x9e 
        0xa0 0xa2 0xa4 0xa6 0xa8 0xaa 0xac 0xae 0xb0 0xb2 0xb4 0xb6 0xb8 0xba 0xbc 0xbe 
        0xc0 0xc2 0xc4 0xc6 0xc8 0xca 0xcc 0xce 0xd0 0xd2 0xd4 0xd6 0xd8 0xda 0xdc 0xde 
        0xe0 0xe2 0xe4 0xe6 0xe8 0xea 0xec 0xee 0xf0 0xf2 0xf4 0xf6 0xf8 0xfa 0xfc 0xfe 
        0x1b 0x19 0x1f 0x1d 0x13 0x11 0x17 0x15 0x0b 0x09 0x0f 0x0d 0x03 0x01 0x07 0x05 
        0x3b 0x39 0x3f 0x3d 0x33 0x31 0x37 0x35 0x2b 0x29 0x2f 0x2d 0x23 0x21 0x27 0x25 
        0x5b 0x59 0x5f 0x5d 0x53 0x51 0x57 0x55 0x4b 0x49 0x4f 0x4d 0x43 0x41 0x47 0x45 
        0x7b 0x79 0x7f 0x7d 0x73 0x71 0x77 0x75 0x6b 0x69 0x6f 0x6d 0x63 0x61 0x67 0x65 
        0x9b 0x99 0x9f 0x9d 0x93 0x91 0x97 0x95 0x8b 0x89 0x8f 0x8d 0x83 0x81 0x87 0x85 
        0xbb 0xb9 0xbf 0xbd 0xb3 0xb1 0xb7 0xb5 0xab 0xa9 0xaf 0xad 0xa3 0xa1 0xa7 0xa5 
        0xdb 0xd9 0xdf 0xdd 0xd3 0xd1 0xd7 0xd5 0xcb 0xc9 0xcf 0xcd 0xc3 0xc1 0xc7 0xc5 
        0xfb 0xf9 0xff 0xfd 0xf3 0xf1 0xf7 0xf5 0xeb 0xe9 0xef 0xed 0xe3 0xe1 0xe7 0xe5
    }
    lindex $xtime $number
}

proc ::aes::GFMult3 {number} {
    # multliply by 2 (via GFMult2) and add the number again on the result (via XOR)
    expr {$number ^ [GFMult2 $number]}
}

proc ::aes::GFMult09 {number} {
    # 09 is: (02*02*02) + 01
    expr {[GFMult2 [GFMult2 [GFMult2 $number]]] ^ $number}
}

proc ::aes::GFMult0b {number} {
    # 0b is: (02*02*02) + 02 + 01
    #return [expr [GFMult2 [GFMult2 [GFMult2 $number]]] ^ [GFMult2 $number] ^ $number]
    #set g0 [GFMult2 $number]
    expr {[GFMult09 $number] ^ [GFMult2 $number]}
}

proc ::aes::GFMult0d {number} {
    # 0d is: (02*02*02) + (02*02) + 01
    set temp [GFMult2 [GFMult2 $number]]
    expr {[GFMult2 $temp] ^ ($temp ^ $number)}
}

proc ::aes::GFMult0e {number} {
    # 0e is: (02*02*02) + (02*02) + 02
    set temp [GFMult2 [GFMult2 $number]]
    expr {[GFMult2 $temp] ^ ($temp ^ [GFMult2 $number])}
}

# -------------------------------------------------------------------------

# aes::Encrypt --
#
#	Encrypt a blocks of plain text and returns blocks of cipher text.
#	The input data must be a multiple of the block size (16).
#
proc ::aes::Encrypt {Key data} {
    set len [string length $data]
    if {($len % 16) != 0} {
        return -code error "invalid block size: AES requires 16 byte blocks"
    }

    set result {}
    for {set i 0} {$i < $len} {incr i 1} {
        set block [string range $data $i [incr i 15]]
        append result [EncryptBlock $Key $block]
    }
    return $result
}

# aes::Decrypt --
#
#	Decrypt blocks of cipher text and returns blocks of plain text.
#	The input data must be a multiple of the block size (16).
#
proc ::aes::Decrypt {Key data} {
    set len [string length $data]
    if {($len % 16) != 0} {
        return -code error "invalid block size: AES requires 16 byte blocks"
    }

    set result {}
    for {set i 0} {$i < $len} {incr i 1} {
        set block [string range $data $i [incr i 15]]
        append result [DecryptBlock $Key $block]
    }
    return $result
}

# -------------------------------------------------------------------------
# chan event handler for chunked file reading.
#
proc ::aes::Chunk {Key in {out {}} {chunksize 4096}} {
    upvar #0 $Key state

    #puts ||CHUNK.X||i=$in|o=$out|c=$chunksize|eof=[eof $in]
    
    if {[eof $in]} {
        chan event $in readable {}
        set state(reading) 0
    }

    set data [read $in $chunksize]

    #puts ||CHUNK.R||i=$in|o=$out|c=$chunksize|eof=[eof $in]||[string length $data]||$data||

    # Do nothing when data was read at all.
    if {$data eq {}} return

    if {[eof $in]} {
        #puts CHUNK.Z
        set data [Pad $data 16]
    }

    #puts ||CHUNK.P||i=$in|o=$out|c=$chunksize|eof=[eof $in]||[string length $data]||$data||
    
    if {$out eq {}} {
        append state(output) [$state(cmd) $Key $data]
    } else {
        puts -nonewline $out [$state(cmd) $Key $data]
    }
}

proc ::aes::SetOneOf {lst item} {
    set ndx [lsearch -glob $lst "${item}*"]
    if {$ndx == -1} {
        set err [join $lst ", "]
        return -code error "invalid mode \"$item\": must be one of $err"
    }
    lindex $lst $ndx
}

proc ::aes::CheckSize {what size thing} {
    if {[string length $thing] != $size} {
        return -code error "invalid value for $what: must be $size bytes long"
    }
    return $thing
}

proc ::aes::Pad {data blocksize {fill \0}} {
    set len [string length $data]
    if {$len == 0} {
        set data [string repeat $fill $blocksize]
    } elseif {($len % $blocksize) != 0} {
        set pad [expr {$blocksize - ($len % $blocksize)}]
        append data [string repeat $fill $pad]
    }
    return $data
}

proc ::aes::Pop {varname {nth 0}} {
    upvar 1 $varname args
    set r [lindex $args $nth]
    set args [lreplace $args $nth $nth]
    return $r
}

proc ::aes::aes {args} {
    array set opts {-dir encrypt -mode cbc -key {} -in {} -out {} -chunksize 4096 -hex 0}
    set opts(-iv) [string repeat \0 16]
    set modes {ecb cbc}
    set dirs {encrypt decrypt}
    while {([llength $args] > 1) && [string match -* [set option [lindex $args 0]]]} {
        switch -exact -- $option {
            -mode      { set opts(-mode) [SetOneOf $modes [Pop args 1]] }
            -dir       { set opts(-dir) [SetOneOf $dirs [Pop args 1]] }
            -iv        { set opts(-iv) [CheckSize -iv 16 [Pop args 1]] }
            -key       { set opts(-key) [Pop args 1] }
            -in        { set opts(-in) [Pop args 1] }
            -out       { set opts(-out) [Pop args 1] }
            -chunksize { set opts(-chunksize) [Pop args 1] }
            -hex       { set opts(-hex) 1 }
            --         { Pop args ; break }
            default {
                set err [join [lsort [array names opts]] ", "]
                return -code error "bad option \"$option\":\
                    must be one of $err"
            }
        }
        Pop args
    }

    if {$opts(-key) eq {}} {
        return -code error "no key provided: the -key option is required"
    }

    set r {}
    if {$opts(-in) eq {}} {

        if {[llength $args] != 1} {
            return -code error "wrong \# args:\
                should be \"aes ?options...? -key keydata plaintext\""
        }

        set data [Pad [lindex $args 0] 16]
        set Key [Init $opts(-mode) $opts(-key) $opts(-iv)]
        if {[string equal $opts(-dir) "encrypt"]} {
            set r [Encrypt $Key $data]
        } else {
            set r [Decrypt $Key $data]
        }

        if {$opts(-out) ne {}} {
            puts -nonewline $opts(-out) $r
            set r {}
        }
        Final $Key

    } else {

        if {[llength $args] != 0} {
            return -code error "wrong \# args:\
                should be \"aes ?options...? -key keydata -in channel\""
        }

        set Key [Init $opts(-mode) $opts(-key) $opts(-iv)]

        set readcmd [list [namespace origin Chunk] \
                         $Key $opts(-in) $opts(-out) \
                         $opts(-chunksize)]

        upvar 1 $Key state
        set state(reading) 1
        if {[string equal $opts(-dir) "encrypt"]} {
            set state(cmd) Encrypt
        } else {
            set state(cmd) Decrypt
        }
        set state(output) ""
        chan event $opts(-in) readable $readcmd
        if {[info commands ::tkwait] != {}} {
            tkwait variable [subst $Key](reading)
        } else {
            vwait [subst $Key](reading)
        }
        if {$opts(-out) == {}} {
            set r $state(output)
        }
        Final $Key
    }

    if {$opts(-hex)} {
        binary scan $r H* r
    }
    return $r
}

# -------------------------------------------------------------------------

package provide aes 1.2.1

# -------------------------------------------------------------------------
# Local variables:
# mode: tcl
# indent-tabs-mode: nil
# End:
# des.tcl
# $Revision: 1.1 $
# $Date: 2005/09/26 09:16:59 $
#
# Port of Javascript implementation to Tcl 8.4 by Mac A. Cody,
# October, 2002 - February, 2003
# August, 2003 - Separated key set generation from encryption/decryption.
#                Renamed "des" procedure to "block" to differentiate from the
#                "stream" procedure used for CFB and OFB modes.
#                Modified the "encrypt" and "decrypt" procedures to support
#                CFB and OFB modes. Changed the procedure arguments.
#                Added the "stream" procedure to support CFB and OFB modes.
# June, 2004 - Corrected input vector bug in stream-mode processing.  Added
#              support for feedback vector storage and management function.
#              This enables a stream of data to be processed over several calls
#              to the encryptor or decryptor.
# September, 2004 - Added feedback vector to the CBC mode of operation to allow
#                   a large data set to be processed over several calls to the
#                   encryptor or decryptor.
# October, 2004 - Added test for weak keys in the createKeys procedure.
#
# Paul Tero, July 2001
# http://www.shopable.co.uk/des.html
#
# Optimised for performance with large blocks by Michael Hayworth,
# November 2001, http://www.netdealing.com
#
# This software is copyrighted (c) 2003, 2004 by Mac A. Cody.  All rights
# reserved.  The following terms apply to all files associated with
# the software unless explicitly disclaimed in individual files or
# directories.

# The authors hereby grant permission to use, copy, modify, distribute,
# and license this software for any purpose, provided that existing
# copyright notices are retained in all copies and that this notice is
# included verbatim in any distributions. No written agreement, license,
# or royalty fee is required for any of the authorized uses.
# Modifications to this software may be copyrighted by their authors and
# need not follow the licensing terms described here, provided that the
# new terms are clearly indicated on the first page of each file where
# they apply.

# IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY PARTY
# FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
# ARISING OUT OF THE USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY
# DERIVATIVES THEREOF, EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

# THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THIS SOFTWARE
# IS PROVIDED ON AN "AS IS" BASIS, AND THE AUTHORS AND DISTRIBUTORS HAVE
# NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR
# MODIFICATIONS.

# GOVERNMENT USE: If you are acquiring this software on behalf of the
# U.S. government, the Government shall have only "Restricted Rights"
# in the software and related documentation as defined in the Federal 
# Acquisition Regulations (FARs) in Clause 52.227.19 (c) (2).  If you
# are acquiring the software on behalf of the Department of Defense, the
# software shall be classified as "Commercial Computer Software" and the
# Government shall have only "Restricted Rights" as defined in Clause
# 252.227-7013 (c) (1) of DFARs.  Notwithstanding the foregoing, the
# authors grant the U.S. Government and others acting in its behalf
# permission to use and distribute the software in accordance with the
# terms specified in this license. 
namespace eval des {
    variable keysets
    variable WeakKeysError
    if {![info exists WeakKeysError]} { set WeakKeysError 1 }
    set keysets(ndx) 1
    # Produre: keyset - Create or destroy a keyset created from a 64-bit
    #                   DES key or a 192-bit 3DES key.
    # Inputs:
    #   oper  : The operation to be performed.  This will be either "create"
    #           (make a new keyset) or "destroy" (delete an existing keyset).
    #           The meaning of the argument "value" depends of the operation
    #           performed.  An error is generated if "oper" is not "create"
    #           or "destroy".
    #             
    #   value : If the argument "oper" is "create", then "value" is the 64-bit
    #           DES key or the 192-bit 3DES key.  (Note: The lsb of each byte
    #           is ignored; odd parity is not required).  If the argument
    #           "oper" is "destroy", then "value" is a handle to a keyset that
    #           was created previously.
    #
    #   weak:   If true then weak keys are allowed. The default is to raise an
    #           error when a weak key is seen.
    # Output:
    #   If the argument "oper" is "create", then the output is a handle to the
    #   keyset stored in the des namespace.  If the argument "oper" is
    #   "destroy", then nothing is returned.
    proc keyset {oper value {weak 0}} {
	variable keysets
	set newset {}
	switch -exact -- $oper {
	    create {
		# Create a new keyset handle.
		set newset keyset$keysets(ndx)
		# Create key set
		set keysets($newset) [createKeys $value $weak]
		# Never use that keyset handle index again.
		incr keysets(ndx)
	    }
	    destroy {
		# Determine if the keyset handle is valid.
		if {[array names keysets $value] != {}} {
		    # Delete the handle and corresponding keyset.
                    unset keysets($value)
		} else {
		    error "The keyset handle \"$value\" is invalid!"
		}
	    }
	    default {
		error {The operator must be either "create" or "destroy".}
	    }
	}
	return $newset
    }

    # Procedure: encrypt - Encryption front-end for the des procedure
    # Inputs:
    #   keyset  : Handle to an existing keyset.
    #   message : String to be encrypted.
    #   mode    : DES mode ecb (default), cbc, cfb, or ofb.
    #   iv      : Name of the initialization vector used in CBC, CFB,
    #             and OFB modes.
    #   kbits   : Number of bits in a data block (default of 64).
    # Output:
    #   The encrypted data string.
    proc encrypt {keyset message {mode ecb} {iv {}} {kbits 64}} {
	switch -exact -- $mode {
	    ecb {
		return [block $keyset $message 1 0]
	    }
	    cbc -
	    ofb -
	    cfb {
		# Is the initialization/feedback vector variable is valid?
		if {[string length $iv] == 0} {
		    error "An initialization variable must be specified."
		} else {
		    upvar $iv ivec
		    if {![info exists ivec]} {
			error "The variable $iv does not exist."
		    }
		}
		switch -exact -- $mode {
		    cbc {
			return [block $keyset $message 1 1 ivec]
		    }
		    ofb {
			return [stream $keyset $message 1 0 ivec $kbits]
		    }
		    cfb {
			return [stream $keyset $message 1 1 ivec $kbits]
		    }
		}
	    }
	    default {
		error {Mode must be ecb, cbc, cfb, or ofb.}
	    }
	}
    }

    # Procedure: decrypt - Decryption front-end for the des procedure
    # Inputs:
    #   keyset  : Handle to an existing keyset.
    #   message : String to be decrypted.
    #   mode    : DES mode ecb (default), cbc, cfb, or ofb.
    #   iv      : Name of the initialization vector used in CBC, CFB,
    #             and OFB modes.
    #   kbits   : Number of bits in a data block (default of 64).
    # Output:
    #   The encrypted or decrypted data string.
    proc decrypt {keyset message {mode ecb} {iv {}} {kbits 64}} {
	switch -exact -- $mode {
	    ecb {
		return [block $keyset $message 0 0]
	    }
	    cbc -
	    ofb -
	    cfb {
		# Is the initialization/feedback vector variable is valid?
		if {[string length $iv] < 1} {
		    error "An initialization variable must be specified."
		} else {
		    upvar $iv ivec
		    if {![info exists ivec]} {
			error "The variable $iv does not exist."
		    }
		}
		switch -exact -- $mode {
		    cbc {
			return [block $keyset $message 0 1 ivec]
		    }
		    ofb {
			return [stream $keyset $message 0 0 ivec $kbits]
		    }
		    cfb {
			return [stream $keyset $message 0 1 ivec $kbits]
		    }
		}
	    }
	    default {
		error {Mode must be ecb, cbc, cfb, or ofb.}
	    }
	}
    }

    variable spfunction1 [list 0x1010400 0 0x10000 0x1010404 0x1010004 0x10404 0x4 0x10000 0x400 0x1010400 0x1010404 0x400 0x1000404 0x1010004 0x1000000 0x4 0x404 0x1000400 0x1000400 0x10400 0x10400 0x1010000 0x1010000 0x1000404 0x10004 0x1000004 0x1000004 0x10004 0 0x404 0x10404 0x1000000 0x10000 0x1010404 0x4 0x1010000 0x1010400 0x1000000 0x1000000 0x400 0x1010004 0x10000 0x10400 0x1000004 0x400 0x4 0x1000404 0x10404 0x1010404 0x10004 0x1010000 0x1000404 0x1000004 0x404 0x10404 0x1010400 0x404 0x1000400 0x1000400 0 0x10004 0x10400 0 0x1010004];
    variable spfunction2 [list 0x80108020 0x80008000 0x8000 0x108020 0x100000 0x20 0x80100020 0x80008020 0x80000020 0x80108020 0x80108000 0x80000000 0x80008000 0x100000 0x20 0x80100020 0x108000 0x100020 0x80008020 0 0x80000000 0x8000 0x108020 0x80100000 0x100020 0x80000020 0 0x108000 0x8020 0x80108000 0x80100000 0x8020 0 0x108020 0x80100020 0x100000 0x80008020 0x80100000 0x80108000 0x8000 0x80100000 0x80008000 0x20 0x80108020 0x108020 0x20 0x8000 0x80000000 0x8020 0x80108000 0x100000 0x80000020 0x100020 0x80008020 0x80000020 0x100020 0x108000 0 0x80008000 0x8020 0x80000000 0x80100020 0x80108020 0x108000];
    variable spfunction3 [list 0x208 0x8020200 0 0x8020008 0x8000200 0 0x20208 0x8000200 0x20008 0x8000008 0x8000008 0x20000 0x8020208 0x20008 0x8020000 0x208 0x8000000 0x8 0x8020200 0x200 0x20200 0x8020000 0x8020008 0x20208 0x8000208 0x20200 0x20000 0x8000208 0x8 0x8020208 0x200 0x8000000 0x8020200 0x8000000 0x20008 0x208 0x20000 0x8020200 0x8000200 0 0x200 0x20008 0x8020208 0x8000200 0x8000008 0x200 0 0x8020008 0x8000208 0x20000 0x8000000 0x8020208 0x8 0x20208 0x20200 0x8000008 0x8020000 0x8000208 0x208 0x8020000 0x20208 0x8 0x8020008 0x20200];
    variable spfunction4 [list 0x802001 0x2081 0x2081 0x80 0x802080 0x800081 0x800001 0x2001 0 0x802000 0x802000 0x802081 0x81 0 0x800080 0x800001 0x1 0x2000 0x800000 0x802001 0x80 0x800000 0x2001 0x2080 0x800081 0x1 0x2080 0x800080 0x2000 0x802080 0x802081 0x81 0x800080 0x800001 0x802000 0x802081 0x81 0 0 0x802000 0x2080 0x800080 0x800081 0x1 0x802001 0x2081 0x2081 0x80 0x802081 0x81 0x1 0x2000 0x800001 0x2001 0x802080 0x800081 0x2001 0x2080 0x800000 0x802001 0x80 0x800000 0x2000 0x802080];
    variable spfunction5 [list 0x100 0x2080100 0x2080000 0x42000100 0x80000 0x100 0x40000000 0x2080000 0x40080100 0x80000 0x2000100 0x40080100 0x42000100 0x42080000 0x80100 0x40000000 0x2000000 0x40080000 0x40080000 0 0x40000100 0x42080100 0x42080100 0x2000100 0x42080000 0x40000100 0 0x42000000 0x2080100 0x2000000 0x42000000 0x80100 0x80000 0x42000100 0x100 0x2000000 0x40000000 0x2080000 0x42000100 0x40080100 0x2000100 0x40000000 0x42080000 0x2080100 0x40080100 0x100 0x2000000 0x42080000 0x42080100 0x80100 0x42000000 0x42080100 0x2080000 0 0x40080000 0x42000000 0x80100 0x2000100 0x40000100 0x80000 0 0x40080000 0x2080100 0x40000100];
    variable spfunction6 [list 0x20000010 0x20400000 0x4000 0x20404010 0x20400000 0x10 0x20404010 0x400000 0x20004000 0x404010 0x400000 0x20000010 0x400010 0x20004000 0x20000000 0x4010 0 0x400010 0x20004010 0x4000 0x404000 0x20004010 0x10 0x20400010 0x20400010 0 0x404010 0x20404000 0x4010 0x404000 0x20404000 0x20000000 0x20004000 0x10 0x20400010 0x404000 0x20404010 0x400000 0x4010 0x20000010 0x400000 0x20004000 0x20000000 0x4010 0x20000010 0x20404010 0x404000 0x20400000 0x404010 0x20404000 0 0x20400010 0x10 0x4000 0x20400000 0x404010 0x4000 0x400010 0x20004010 0 0x20404000 0x20000000 0x400010 0x20004010];
    variable spfunction7 [list 0x200000 0x4200002 0x4000802 0 0x800 0x4000802 0x200802 0x4200800 0x4200802 0x200000 0 0x4000002 0x2 0x4000000 0x4200002 0x802 0x4000800 0x200802 0x200002 0x4000800 0x4000002 0x4200000 0x4200800 0x200002 0x4200000 0x800 0x802 0x4200802 0x200800 0x2 0x4000000 0x200800 0x4000000 0x200800 0x200000 0x4000802 0x4000802 0x4200002 0x4200002 0x2 0x200002 0x4000000 0x4000800 0x200000 0x4200800 0x802 0x200802 0x4200800 0x802 0x4000002 0x4200802 0x4200000 0x200800 0 0x2 0x4200802 0 0x200802 0x4200000 0x800 0x4000002 0x4000800 0x800 0x200002];
    variable spfunction8 [list 0x10001040 0x1000 0x40000 0x10041040 0x10000000 0x10001040 0x40 0x10000000 0x40040 0x10040000 0x10041040 0x41000 0x10041000 0x41040 0x1000 0x40 0x10040000 0x10000040 0x10001000 0x1040 0x41000 0x40040 0x10040040 0x10041000 0x1040 0 0 0x10040040 0x10000040 0x10001000 0x41040 0x40000 0x41040 0x40000 0x10041000 0x1000 0x40 0x10040040 0x1000 0x41040 0x10001000 0x40 0x10000040 0x10040000 0x10040040 0x10000000 0x40000 0x10001040 0 0x10041040 0x40040 0x10000040 0x10040000 0x10001000 0x10001040 0 0x10041040 0x41000 0x41000 0x1040 0x1040 0x40040 0x10000000 0x10041000];

    variable desEncrypt {0 32 2}
    variable desDecrypt {30 -2 -2}
    variable des3Encrypt {0 32 2 62 30 -2 64 96 2}
    variable des3Decrypt {94 62 -2 32 64 2 30 -2 -2}

    # Procedure: block - DES ECB and CBC mode support
    # Inputs:
    #   keyset   : Handle to an existing keyset.
    #   message  : String to be encrypted or decrypted (Note: For encryption,
    #              the string is extended with null characters to an integral
    #              multiple of eight bytes.  For decryption, the string length
    #              must be an integral multiple of eight bytes.
    #   encrypt  : Perform encryption (1) or decryption (0)
    #   mode     : DES mode 1=CBC, 0=ECB (default).
    #   iv       : Name of the variable containing the initialization vector
    #              used in CBC mode.  The value must be 64 bits in length.
    # Output:
    #   The encrypted or decrypted data string.
    proc block {keyset message encrypt {mode 0} {iv {}}} {
	variable spfunction1
	variable spfunction2
	variable spfunction3
	variable spfunction4
	variable spfunction5
	variable spfunction6
	variable spfunction7
	variable spfunction8
	variable desEncrypt
	variable desDecrypt
	variable des3Encrypt
	variable des3Decrypt
	variable keysets

	# Determine if the keyset handle is valid.
	if {[array names keysets $keyset] != {}} {
	    # Acquire the 16 or 48 subkeys we will need
	    set keys $keysets($keyset)
	} else {
	    error "The keyset handle \"$keyset\" is invalid!"
	}
	set m 0
	set cbcleft 0x00; set cbcleft2 0x00
	set cbcright 0x00; set cbcright2 0x00
	set len [string length $message];
        if {$len == 0} {
            return -code error "invalid message size: the message may not be empty"
        }
	set chunk 0;
	# Set up the loops for single and triple des
	set iterations [expr {[llength $keys] == 32 ? 3 : 9}];
	if {$iterations == 3} {
	    expr {$encrypt ? [set looping $desEncrypt] : \
		      [set looping $desDecrypt]}
	} else {
	    expr {$encrypt ? [set looping $des3Encrypt] : \
		      [set looping $des3Decrypt]}
	}

	# Pad the message out with null bytes.
	append message "\0\0\0\0\0\0\0\0"

	# Store the result here
	set result {};
	set tempresult {};

	# CBC mode
	if {$mode == 1} {
	    # Is the initialization/feedback vector variable is valid?
	    if {[string length $iv] < 1} {
		error "An initialization variable must be specified."
	    } else {
		upvar $iv ivec
		if {![info exists ivec]} {
		    error "The variable $iv does not exist."
		}
                if {[string length $ivec] != 8} {
                    return -code error "invalid initialization vector size:\
                        the initialization vector must be 8 bytes"
                }
	    }
	    # Use the input vector as the intial vector.
	    binary scan $ivec H8H8 cbcleftTemp cbcrightTemp
	    set cbcleft "0x$cbcleftTemp"
	    set cbcright "0x$cbcrightTemp"
	}

	# Loop through each 64 bit chunk of the message
	while {$m < $len} {
	    binary scan $message x${m}H8H8 lefttemp righttemp
	    set left {}
	    append left "0x" $lefttemp
	    set right {}
	    append right "0x" $righttemp
	    incr m 8

	    #puts "Left start: $left";
	    #puts "Right start: $right";
	    # For Cipher Block Chaining mode, xor the
	    # message with the previous result.
	    if {$mode == 1} {
		if {$encrypt} {
		    set left [expr {$left ^ $cbcleft}]
		    set right [expr {$right ^ $cbcright}]
		} else {
		    set cbcleft2 $cbcleft;
		    set cbcright2 $cbcright;
		    set cbcleft $left;
		    set cbcright $right;
		}
	    }

	    #puts "Left mode: $left";
	    #puts "Right mode: $right";
	    #puts "cbcleft: $cbcleft";
	    #puts "cbcleft2: $cbcleft2";
	    #puts "cbcright: $cbcright";
	    #puts "cbcright2: $cbcright2";

	    # First each 64 but chunk of the message
	    # must be permuted according to IP.
	    set temp [expr {(($left >> 4) ^ $right) & 0x0f0f0f0f}];
	    set right [expr {$right ^ $temp}];
	    set left [expr {$left ^ ($temp << 4)}];
	    set temp [expr {(($left >> 16) ^ $right) & 0x0000ffff}];
	    set right [expr {$right ^ $temp}];
	    set left [expr {$left ^ ($temp << 16)}];
	    set temp [expr {(($right >> 2) ^ $left) & 0x33333333}];
	    set left [expr {$left ^ $temp}]
	    set right [expr {$right ^ ($temp << 2)}];

	    set temp [expr {(($right >> 8) ^ $left) & 0x00ff00ff}];
	    set left [expr {$left ^ $temp}];
	    set right [expr {$right ^ ($temp << 8)}];
	    set temp [expr {(($left >> 1) ^ $right) & 0x55555555}];
	    set right [expr {$right ^ $temp}];
	    set left [expr {$left ^ ($temp << 1)}];

	    set left [expr {((($left << 1) & 0xffffffff) | \
				 (($left >> 31) & 0x00000001))}]; 
	    set right [expr {((($right << 1) & 0xffffffff) | \
				  (($right >> 31) & 0x00000001))}]; 

	    #puts "Left IP: [format %x $left]";
	    #puts "Right IP: [format %x $right]";

	    # Do this either 1 or 3 times for each chunk of the message
	    for {set j 0} {$j < $iterations} {incr j 3} {
		set endloop [lindex $looping [expr {$j + 1}]];
		set loopinc [lindex $looping [expr {$j + 2}]];

		#puts "endloop: $endloop";
		#puts "loopinc: $loopinc";

		# Now go through and perform the encryption or decryption  
		for {set i [lindex $looping $j]} \
		    {$i != $endloop} {incr i $loopinc} {
		    # For efficiency
		    set right1 [expr {$right ^ [lindex $keys $i]}]; 
		    set right2 [expr {((($right >> 4) & 0x0fffffff) | \
					   (($right << 28) & 0xffffffff)) ^ \
					  [lindex $keys [expr {$i + 1}]]}];
 
		    # puts "right1: [format %x $right1]";
		    # puts "right2: [format %x $right2]";

		    # The result is attained by passing these
		    # bytes through the S selection functions.
		    set temp $left;
		    set left $right;
		    set right [expr {$temp ^ ([lindex $spfunction2 [expr {($right1 >> 24) & 0x3f}]] | \
                                                  [lindex $spfunction4 [expr {($right1 >> 16) & 0x3f}]] | \
                                                  [lindex $spfunction6 [expr {($right1 >>  8) & 0x3f}]] | \
                                                  [lindex $spfunction8 [expr {$right1 & 0x3f}]] | \
                                                  [lindex $spfunction1 [expr {($right2 >> 24) & 0x3f}]] | \
                                                  [lindex $spfunction3 [expr {($right2 >> 16) & 0x3f}]] | \
                                                  [lindex $spfunction5 [expr {($right2 >>  8) & 0x3f}]] | \
						  [lindex $spfunction7 [expr {$right2 & 0x3f}]])}];
 
		    # puts "Left iter: [format %x $left]";
		    # puts "Right iter: [format %x $right]";

		}
		set temp $left;
		set left $right;
		set right $temp; # Unreverse left and right
	    }; # For either 1 or 3 iterations

	    #puts "Left Iterated: [format %x $left]";
	    #puts "Right Iterated: [format %x $right]";

	    # Move then each one bit to the right
	    set left [expr {((($left >> 1) & 0x7fffffff) \
				 | (($left << 31) & 0xffffffff))}]; 
	    set right [expr {((($right >> 1) & 0x7fffffff) \
				  | (($right << 31) & 0xffffffff))}]; 

	    #puts "Left shifted: [format %x $left]";
	    #puts "Right shifted: [format %x $right]";

	    # Now perform IP-1, which is IP in the opposite direction
	    set temp [expr {((($left >> 1) & 0x7fffffff) ^ $right) & 0x55555555}];
	    set right [expr {$right ^ $temp}];
	    set left [expr {$left ^ ($temp << 1)}];
	    set temp [expr {((($right >> 8) & 0x00ffffff) ^ $left) & 0x00ff00ff}];
	    set left [expr {$left ^ $temp}];
	    set right [expr {$right ^ ($temp << 8)}];
	    set temp [expr {((($right >> 2) & 0x3fffffff) ^ $left) & 0x33333333}]; 
	    set left [expr {$left ^ $temp}];
	    set right [expr {$right ^ ($temp << 2)}];
	    set temp [expr {((($left >> 16) & 0x0000ffff) ^ $right) & 0x0000ffff}];
	    set right [expr {$right ^ $temp}];
	    set left [expr {$left ^ ($temp << 16)}];
	    set temp [expr {((($left >> 4) & 0x0fffffff) ^ $right) & 0x0f0f0f0f}];
	    set right [expr {$right ^ $temp}];
	    set left [expr {$left ^ ($temp << 4)}];

	    #puts "Left IP-1: [format %x $left]";
	    #puts "Right IP-1: [format %x $right]";

	    # For Cipher Block Chaining mode, xor
	    # the message with the previous result.
	    if {$mode == 1} {
		if {$encrypt} {
		    set cbcleft $left;
		    set cbcright $right;
		} else {
		    set left [expr {$left ^ $cbcleft2}];
		    set right [expr {$right ^ $cbcright2}];
		}
	    }

	    append tempresult \
		[binary format H16 [format %08x%08x $left $right]]

	    #puts "Left final: [format %x $left]";
	    #puts "Right final: [format %x $right]";

	    incr chunk 8;
	    if {$chunk == 512} {
		append result $tempresult
		set tempresult {};
		set chunk 0;
	    }
	}; # For every 8 characters, or 64 bits in the message

	if {$mode == 1} {
	    if {$encrypt} {
		# Save the left and right registers to the feedback vector.
		set ivec [binary format H* \
			      [format %08x $left][format %08x $right]]
	    } else {
		set ivec [binary format H* \
			      [format %08x $cbcleft][format %08x $cbcright]]
	    }
	}

	# Return the result as an array
	return ${result}$tempresult
    }; # End of block

    # Procedure: stream - DES CFB and OFB mode support
    # Inputs:
    #   keyset   : Handle to an existing keyset.
    #   message  : String to be encrypted or decrypted (Note: The length of the
    #              string is dependent upon the value of kbits.  Remember that
    #              the string is part of a stream of data, so it must be sized
    #              properly for subsequent encryptions/decryptions to be
    #              correct.  See the man page for correct message lengths for
    #              values of kbits).
    #   encrypt  : Perform encryption (1) or decryption (0)
    #   mode     : DES mode 0=OFB, 1=CFB.
    #   iv       : Name of variable containing the initialization vector.  The
    #              value must be 64 bits in length with the first 64-L bits set
    #              to zero.
    #   kbits    : Number of bits in a data block (default of 64).
    # Output:
    #   The encrypted or decrypted data string.
    proc stream {keyset message encrypt mode iv {kbits 64}} {
	variable spfunction1
	variable spfunction2
	variable spfunction3
	variable spfunction4
	variable spfunction5
	variable spfunction6
	variable spfunction7
	variable spfunction8
	variable desEncrypt
	variable des3Encrypt
	variable keysets

	# Determine if the keyset handle is valid.
	if {[array names keysets $keyset] != {}} {
	    # Acquire the 16 or 48 subkeys we will need.
	    set keys $keysets($keyset)
	} else {
	    error "The keyset handle \"$keyset\" is invalid!"
	}

	# Is the initialization/feedback vector variable is valid?
	if {[string length $iv] < 1} {
	    error "An initialization variable must be specified."
	} else {
	    upvar $iv ivec
	    if {![info exists ivec]} {
		error "The variable $iv does not exist."
	    }
	}

        # Determine if message length (in bits)
	# is not an integral number of kbits.
	set len [string length $message];
        #puts "len: $len, kbits: $kbits"
	if {($kbits < 1) || ($kbits > 64)} {
	    error "The valid values of kbits are 1 through 64."
        } elseif {($kbits % 8) != 0} {
	    set blockSize [expr {$kbits + (8 - ($kbits % 8))}]
	    set fail [expr {(($len * 8) / $blockSize) % $kbits}]
	} else {
	    set blockSize [expr {$kbits / 8}]
	    set fail [expr {$len % $blockSize}]
	}
        if {$fail} {
	    error "Data length (in bits) is not an integral number of kbits."
	}

	set m 0
	set n 0
	set chunk 0;
	# Set up the loops for single and triple des
	set iterations [expr {[llength $keys] == 32 ? 3 : 9}];
	if {$iterations == 3} {
	    set looping $desEncrypt
	} else {
	    set looping $des3Encrypt
	}

        # Set up shifting values.  Used for both CFB and OFB modes.
        if {$kbits < 32} {
	    # Only some bits from left output are needed.
	    set kOutShift [expr {32 - $kbits}]
	    set kOutMask [expr {0x7fffffff >> (31 - $kbits)}]
	    # Determine number of message bytes needed per iteration.
	    set msgBytes [expr {int(ceil(double($kbits) / 8.0))}]
	    # Determine number of message bits needed per iteration.
	    set msgBits [expr {$msgBytes * 8}]
	    set msgBitsSub1 [expr {$msgBits - 1}]
	    # Define bit caches.
	    set bitCacheIn {}
	    set bitCacheOut {}
	    # Variable used to remove bits 0 through
	    # kbits-1 in the input bit cache.
	    set kbitsSub1 [expr {$kbits - 1}]
	    # Variable used to remove leading dummy binary bits.
	    set xbits [expr {32 - $kbits}]
	} elseif {$kbits == 32} {
	    # Only bits of left output are used.
	    # Four messages bytes are needed per iteration.
	    set msgBytes 4
	    set xbits 32
	} elseif {$kbits < 64} {
	    # All bits from left output are needed.
	    set kOutShiftLeft [expr {$kbits - 32}]
	    # Some bits from right output are needed.
	    set kOutShiftRight [expr {64 - $kbits}]
	    set kOutMaskRight [expr {0x7fffffff >> (63 - $kbits)}]
	    # Determine number of message bytes needed per iteration.
	    set msgBytes [expr {int(ceil(double($kbits) / 8.0))}]
	    # Determine number of message bits needed per iteration.
	    set msgBits [expr {$msgBytes * 8}]
	    set msgBitsSub1 [expr {$msgBits - 1}]
	    # Define bit caches.
	    set bitCacheIn {}
	    set bitCacheOut {}
	    # Variable used to remove bits 0 through
	    # kbits-1 in the input bit cache.
	    set kbitsSub1 [expr {$kbits - 1}]
	    # Variable used to remove leading dummy binary bits.
	    set xbits [expr {64 - $kbits}]
	} else {
	    # All 64 bits of output are used.
	    # Eight messages bytes are needed per iteration.
	    set msgBytes 8
	    set xbits 0
	}

	# Store the result here
	set result {}
	set tempresult {}

	# Set up the initialization vector bitstream
	binary scan $ivec H8H8 leftTemp rightTemp
	set left "0x$leftTemp"
	set right "0x$rightTemp"
        #puts "Retrieved Feedback vector: $fbvec"
        #puts "Start: |$left| |$right|"
	
	# Loop through each 64 bit chunk of the message
	while {$m < $len} {
	    # puts "Left start: $left";
	    # puts "Right start: $right";

	    # First each 64 but chunk of the
	    # message must be permuted according to IP.
	    set temp [expr {(($left >> 4) ^ $right) & 0x0f0f0f0f}];
	    set right [expr {$right ^ $temp}];
	    set left [expr {$left ^ ($temp << 4)}];
	    set temp [expr {(($left >> 16) ^ $right) & 0x0000ffff}];
	    set right [expr {$right ^ $temp}];
	    set left [expr {$left ^ ($temp << 16)}];
	    set temp [expr {(($right >> 2) ^ $left) & 0x33333333}];
	    set left [expr {$left ^ $temp}];
	    set right [expr {$right ^ ($temp << 2)}];

	    set temp [expr {(($right >> 8) ^ $left) & 0x00ff00ff}];
	    set left [expr {$left ^ $temp}];
	    set right [expr {$right ^ ($temp << 8)}];
	    set temp [expr {(($left >> 1) ^ $right) & 0x55555555}];
	    set right [expr {$right ^ $temp}];
	    set left [expr {$left ^ ($temp << 1)}];

	    set left [expr {((($left << 1) & 0xffffffff) | \
				 (($left >> 31) & 0x00000001))}]; 
	    set right [expr {((($right << 1) & 0xffffffff) | \
				  (($right >> 31) & 0x00000001))}]; 

	    #puts "Left IP: [format %x $left]";
	    #puts "Right IP: [format %x $right]";

	    # Do this either 1 or 3 times for each chunk of the message
	    for {set j 0} {$j < $iterations} {incr j 3} {
		set endloop [lindex $looping [expr {$j + 1}]];
		set loopinc [lindex $looping [expr {$j + 2}]];

		#puts "endloop: $endloop";
		#puts "loopinc: $loopinc";

		# Now go through and perform the encryption or decryption  
		for {set i [lindex $looping $j]} \
		    {$i != $endloop} {incr i $loopinc} {
		    # For efficiency
		    set right1 [expr {$right ^ [lindex $keys $i]}]; 
		    set right2 [expr {((($right >> 4) & 0x0fffffff) | \
					   (($right << 28) & 0xffffffff)) ^ \
					  [lindex $keys [expr {$i + 1}]]}];
 
		    # puts "right1: [format %x $right1]";
		    # puts "right2: [format %x $right2]";

		    # The result is attained by passing these
		    # bytes through the S selection functions.
		    set temp $left;
		    set left $right;
		    set right [expr {$temp ^ ([lindex $spfunction2 [expr {($right1 >> 24) & 0x3f}]] | \
						  [lindex $spfunction4 [expr {($right1 >> 16) & 0x3f}]] | \
						  [lindex $spfunction6 [expr {($right1 >>  8) & 0x3f}]] | \
						  [lindex $spfunction8 [expr {$right1 & 0x3f}]] | \
						  [lindex $spfunction1 [expr {($right2 >> 24) & 0x3f}]] | \
						  [lindex $spfunction3 [expr {($right2 >> 16) & 0x3f}]] | \
						  [lindex $spfunction5 [expr {($right2 >>  8) & 0x3f}]] | \
						  [lindex $spfunction7 [expr {$right2 & 0x3f}]])}];
 
		    # puts "Left iter: [format %x $left]";
		    # puts "Right iter: [format %x $right]";

		}
		set temp $left;
		set left $right;
		set right $temp; # Unreverse left and right
	    }; # For either 1 or 3 iterations

	    #puts "Left Iterated: [format %x $left]";
	    #puts "Right Iterated: [format %x $right]";

	    # Move then each one bit to the right
	    set left [expr {((($left >> 1) & 0x7fffffff) | \
				 (($left << 31) & 0xffffffff))}]; 
	    set right [expr {((($right >> 1) & 0x7fffffff) | \
				  (($right << 31) & 0xffffffff))}]; 

	    #puts "Left shifted: [format %x $left]";
	    #puts "Right shifted: [format %x $right]";

	    # Now perform IP-1, which is IP in the opposite direction
	    set temp [expr {((($left >> 1) & 0x7fffffff) ^ $right) & 0x55555555}];
	    set right [expr {$right ^ $temp}];
	    set left [expr {$left ^ ($temp << 1)}];
	    set temp [expr {((($right >> 8) & 0x00ffffff) ^ $left) & 0x00ff00ff}];
	    set left [expr {$left ^ $temp}];
	    set right [expr {$right ^ ($temp << 8)}];
	    set temp [expr {((($right >> 2) & 0x3fffffff) ^ $left) & 0x33333333}]; 
	    set left [expr {$left ^ $temp}];
	    set right [expr {$right ^ ($temp << 2)}];
	    set temp [expr {((($left >> 16) & 0x0000ffff) ^ $right) & 0x0000ffff}];
	    set right [expr {$right ^ $temp}];
	    set left [expr {$left ^ ($temp << 16)}];
	    set temp [expr {((($left >> 4) & 0x0fffffff) ^ $right) & 0x0f0f0f0f}];
	    set right [expr {$right ^ $temp}];
	    set left [expr {$left ^ ($temp << 4)}];

	    #puts "Left IP-1: [format %x $left]";
	    #puts "Right IP-1: [format %x $right]";

	    # Extract the "kbits" most significant bits from the output block.
	    if {$kbits < 32} {
		# Only some bits from left output are needed.
		set kData [expr {($left >> $kOutShift) & $kOutMask}]
		set newBits {}
		# If necessary, copy message bytes into input bit cache.
		if {([string length $bitCacheIn] < $kbits) && ($n < $len)} {
		    if {$len - $n < $msgBytes} {
			set lastBits [expr {($len - $n) * 8}]
			###puts -nonewline [binary scan $message x${n}B$lastBits newBits]
			binary scan $message x${n}B$lastBits newBits
		    } else {
			# Extract "msgBytes" whole bytes as bits
			###puts -nonewline [binary scan $message x${n}B$msgBits newBits]
			binary scan $message x${n}B$msgBits newBits
		    }
		    incr n $msgBytes
		    #puts " $newBits  $n [expr {$len - $n}]"
		    # Add the bits to the input bit cache.
		    append bitCacheIn $newBits
		}
		#puts -nonewline "In bit cache: $bitCacheIn"
		# Set up message data from input bit cache.
		binary scan [binary format B32 [format %032s [string range $bitCacheIn 0 $kbitsSub1]]] H8 temp
		set msgData "0x$temp"
		# Mix message bits with crypto bits.
		set mixData [expr {$msgData ^ $kData}]
		# Discard collected bits from the input bit cache.
		set bitCacheIn [string range $bitCacheIn $kbits end]
		#puts "  After: $bitCacheIn"
		# Convert back to a bit stream and append to the output bit cache.
		# Only the lower kbits are wanted.
		binary scan [binary format H8 [format %08x $mixData]] B32 msgOut
		append bitCacheOut [string range $msgOut $xbits end]
		#puts -nonewline "Out bit cache: $bitCacheOut"
		# If there are sufficient bits, move bytes to the temporary holding string.
		if {[string length $bitCacheOut] >= $msgBits} {
		    append tempresult [binary format B$msgBits [string range $bitCacheOut 0 $msgBitsSub1]]
		    set bitCacheOut [string range $bitCacheOut $msgBits end]
                    #puts -nonewline "  After: $bitCacheOut"
		    incr m $msgBytes
		    ###puts "$m bytes output"
		    incr chunk $msgBytes
		}
		#puts ""
		# For CFB mode
		if {$mode == 1} {
		    if {$encrypt} {
			set temp [expr {($right << $kbits) & 0xffffffff}]
			set left [expr {(($left << $kbits) & 0xffffffff) | (($right >> $kOutShift) & $kOutMask)}]
			set right [expr {$temp | $mixData}]
		    } else {
			set temp [expr {($right << $kbits) & 0xffffffff}]
			set left [expr {(($left << $kbits) & 0xffffffff) | (($right >> $kOutShift) & $kOutMask)}]
			set right [expr {$temp | $msgData}]
		    }
		}
	    } elseif {$kbits == 32} {
		# Only bits of left output are used.
		set kData $left
		# Four messages bytes are needed per iteration.
		binary scan $message x${m}H8 temp
		incr m 4
		incr chunk 4
		set msgData "0x$temp"
		# Mix message bits with crypto bits.
		set mixData [expr {$msgData ^ $kData}]
		# Move bytes to the temporary holding string.
		append tempresult [binary format H8 [format %08x $mixData]]
		# For CFB mode
		if {$mode == 1} {
		    set left $right
		    if {$encrypt} {
			set right $mixData
		    } else {
			set right $msgData
		    }
		}
	    } elseif {$kbits < 64} {
		set kDataLeft [expr {($left >> $kOutShiftRight) & $kOutMaskRight}]
		set temp [expr {($left << $kOutShiftLeft) & 0xffffffff}]
		set kDataRight [expr {(($right >> $kOutShiftRight) & $kOutMaskRight) | $temp}]
		# If necessary, copy message bytes into input bit cache.
		if {([string length $bitCacheIn] < $kbits)  && ($n < $len)} {
		    if {$len - $n < $msgBytes} {
			set lastBits [expr {($len - $n) * 8}]
			###puts -nonewline [binary scan $message x${n}B$lastBits newBits]
			binary scan $message x${n}B$lastBits newBits
		    } else {
			# Extract "msgBytes" whole bytes as bits
			###puts -nonewline [binary scan $message x${n}B$msgBits newBits]
			binary scan $message x${n}B$msgBits newBits
		    }
		    incr n $msgBytes
		    # Add the bits to the input bit cache.
		    append bitCacheIn $newBits
		}
		# Set up message data from input bit cache.
		# puts "Bits from cache: [set temp [string range $bitCacheIn 0 $kbitsSub1]]"
		# puts "Length of bit string: [string length $temp]"
		binary scan [binary format B64 [format %064s [string range $bitCacheIn 0 $kbitsSub1]]] H8H8 leftTemp rightTemp
		set msgDataLeft "0x$leftTemp"
		set msgDataRight "0x$rightTemp"
		# puts "msgDataLeft: $msgDataLeft"
		# puts "msgDataRight: $msgDataRight"
		# puts "kDataLeft: [format 0x%08x $kDataLeft]"
		# puts "kDataRight: [format 0x%08x $kDataRight]"
		# Mix message bits with crypto bits.
		set mixDataLeft [expr {$msgDataLeft ^ $kDataLeft}]
		set mixDataRight [expr {$msgDataRight ^ $kDataRight}]
		# puts "mixDataLeft: $mixDataLeft"
		# puts "mixDataRight: $mixDataRight"
		# puts "mixDataLeft: [format 0x%08x $mixDataLeft]"
		# puts "mixDataRight: [format 0x%08x $mixDataRight]"
		# Discard collected bits from the input bit cache.
		set bitCacheIn [string range $bitCacheIn $kbits end]
		# Convert back to a bit stream and
		# append to the output bit cache.
		# Only the lower kbits are wanted.
		binary scan \
		    [binary format H8H8 \
			 [format %08x $mixDataLeft] \
			 [format %08x $mixDataRight]] B64 msgOut
		append bitCacheOut [string range $msgOut $xbits end]
		# If there are sufficient bits, move
		# bytes to the temporary holding string.
		if {[string length $bitCacheOut] >= $msgBits} {
		    append tempresult \
			[binary format B$msgBits \
			     [string range $bitCacheOut 0 $msgBitsSub1]]
		    set bitCacheOut [string range $bitCacheOut $msgBits end]
		    incr m $msgBytes
		    incr chunk $msgBytes
		}
		# For CFB mode
		if {$mode == 1} {
		    if {$encrypt} {
			set temp \
			    [expr {($right << $kOutShiftRight) & 0xffffffff}]
			set left [expr {$temp | $mixDataLeft}]
			set right $mixDataRight
		    } else {
			set temp \
			    [expr {($right << $kOutShiftRight) & 0xffffffff}]
			set left [expr {$temp | $msgDataLeft}]
			set right $msgDataRight
		    }
		}
	    } else {
		# All 64 bits of output are used.
		set kDataLeft $left
		set kDataRight $right
		# Eight messages bytes are needed per iteration.
		binary scan $message x${m}H8H8 leftTemp rightTemp
		incr m 8
		incr chunk 8
		set msgDataLeft "0x$leftTemp"
		set msgDataRight "0x$rightTemp"
		# Mix message bits with crypto bits.
		set mixDataLeft [expr {$msgDataLeft ^ $kDataLeft}]
		set mixDataRight [expr {$msgDataRight ^ $kDataRight}]
		# Move bytes to the temporary holding string.
		append tempresult \
		    [binary format H16 \
			 [format %08x%08x $mixDataLeft $mixDataRight]]
		# For CFB mode
		if {$mode == 1} {
		    if {$encrypt} {
			set left $mixDataLeft
			set right $mixDataRight
		    } else {
			set left $msgDataLeft
			set right $msgDataRight
		    }
		}
	    }

	    #puts "Left final: [format %x $left]";
	    #puts "Right final: [format %x $right]";

	    if {$chunk >= 512} {
		append result $tempresult
		set tempresult {};
		set chunk 0;
	    }
	}; # For every 8 characters, or 64 bits in the message
        #puts "End: |[format 0x%08x $left]| |[format 0x%08x $right]|"
	# Save the left and right registers to the feedback vector.
	set ivec [binary format H* [format %08x $left][format %08x $right]]
	#puts "Saved Feedback vector: $fbvectors($fbvector)"

        append result $tempresult
	if {[string length $result] > $len} {
	    set result [string replace $result $len end]
	}
	# Return the result as an array
	return $result
    }; # End of stream

    variable pc2bytes0 [list 0 0x4 0x20000000 0x20000004 0x10000 0x10004 0x20010000 0x20010004 0x200 0x204 0x20000200 0x20000204 0x10200 0x10204 0x20010200 0x20010204]
    variable pc2bytes1 [list 0 0x1 0x100000 0x100001 0x4000000 0x4000001 0x4100000 0x4100001 0x100 0x101 0x100100 0x100101 0x4000100 0x4000101 0x4100100 0x4100101]
    variable pc2bytes2 [list 0 0x8 0x800 0x808 0x1000000 0x1000008 0x1000800 0x1000808 0 0x8 0x800 0x808 0x1000000 0x1000008 0x1000800 0x1000808]
    variable pc2bytes3 [list 0 0x200000 0x8000000 0x8200000 0x2000 0x202000 0x8002000 0x8202000 0x20000 0x220000 0x8020000 0x8220000 0x22000 0x222000 0x8022000 0x8222000]
    variable pc2bytes4 [list 0 0x40000 0x10 0x40010 0 0x40000 0x10 0x40010 0x1000 0x41000 0x1010 0x41010 0x1000 0x41000 0x1010 0x41010]
    variable pc2bytes5 [list 0 0x400 0x20 0x420 0 0x400 0x20 0x420 0x2000000 0x2000400 0x2000020 0x2000420 0x2000000 0x2000400 0x2000020 0x2000420]
    variable pc2bytes6 [list 0 0x10000000 0x80000 0x10080000 0x2 0x10000002 0x80002 0x10080002 0 0x10000000 0x80000 0x10080000 0x2 0x10000002 0x80002 0x10080002]
    variable pc2bytes7 [list 0 0x10000 0x800 0x10800 0x20000000 0x20010000 0x20000800 0x20010800 0x20000 0x30000 0x20800 0x30800 0x20020000 0x20030000 0x20020800 0x20030800]
    variable pc2bytes8 [list 0 0x40000 0 0x40000 0x2 0x40002 0x2 0x40002 0x2000000 0x2040000 0x2000000 0x2040000 0x2000002 0x2040002 0x2000002 0x2040002]
    variable pc2bytes9 [list 0 0x10000000 0x8 0x10000008 0 0x10000000 0x8 0x10000008 0x400 0x10000400 0x408 0x10000408 0x400 0x10000400 0x408 0x10000408]
    variable pc2bytes10 [list 0 0x20 0 0x20 0x100000 0x100020 0x100000 0x100020 0x2000 0x2020 0x2000 0x2020 0x102000 0x102020 0x102000 0x102020]
    variable pc2bytes11 [list 0 0x1000000 0x200 0x1000200 0x200000 0x1200000 0x200200 0x1200200 0x4000000 0x5000000 0x4000200 0x5000200 0x4200000 0x5200000 0x4200200 0x5200200]
    variable pc2bytes12 [list 0 0x1000 0x8000000 0x8001000 0x80000 0x81000 0x8080000 0x8081000 0x10 0x1010 0x8000010 0x8001010 0x80010 0x81010 0x8080010 0x8081010]
    variable pc2bytes13 [list 0 0x4 0x100 0x104 0 0x4 0x100 0x104 0x1 0x5 0x101 0x105 0x1 0x5 0x101 0x105]

    # Now define the left shifts which need to be done
    variable shifts {0  0  1  1  1  1  1  1  0  1  1  1  1  1  1  0};

    # Procedure: createKeys
    # Input:
    #   key     : The 64-bit DES key or the 192-bit 3DES key
    #             (Note: The lsb of each byte is ignored; odd parity
    #             is not required).
    #
    #   weak:   If true then weak keys are allowed. The default is to raise an
    #           error when a weak key is seen.
    # Output:
    # The 16 (DES) or 48 (3DES) subkeys.
    proc createKeys {key {weak 0}} {
	variable pc2bytes0
	variable pc2bytes1
	variable pc2bytes2
	variable pc2bytes3
	variable pc2bytes4
	variable pc2bytes5
	variable pc2bytes6
	variable pc2bytes7
	variable pc2bytes8
	variable pc2bytes9
	variable pc2bytes10
	variable pc2bytes11
	variable pc2bytes12
	variable pc2bytes13
	variable shifts

	# How many iterations (1 for des, 3 for triple des)
	set iterations [expr {([string length $key] >= 24) ? 3 : 1}];
	# Stores the return keys
	set keys {}
	# Other variables
	set lefttemp {}; set righttemp {}
	set m 0
	# Either 1 or 3 iterations
	for {set j 0} {$j < $iterations} {incr j} {
	    binary scan $key x${m}H8H8 lefttemp righttemp
	    set left {}
	    append left "0x" $lefttemp
	    set right {}
	    append right "0x" $righttemp
	    incr m 8

	    #puts "Left key: $left"
	    #puts "Right key: $right"

	    # Test for weak keys
            if {! $weak} {
                set maskedLeft [expr {$left & 0xfefefefe}]
                set maskedRight [expr {$right & 0xfefefefe}]
                if {($maskedLeft == 0x00000000) \
                        && ($maskedRight == 0x00000000)} {
                    error "Key [expr {$j + 1}] is weak!"
                } elseif {($maskedLeft == 0x1e1e1e1e) \
                              && ($maskedRight == 0x0e0e0e0e)} {
                    error "Key [expr {$j + 1}] is weak!"
                } elseif {($maskedLeft == 0xe0e0e0e0) \
                              && ($maskedRight == 0xf0f0f0f0)} {
                    error "Key [expr {$j + 1}] is weak!"
                } elseif {($maskedLeft == 0xfefefefe) \
                              && ($maskedRight == 0xfefefefe)} {
                    error "Key [expr {$j + 1}] is weak!"
                }
            }

	    set temp [expr {(($left >> 4) ^ $right) & 0x0f0f0f0f}]
	    set right [expr {$right ^ $temp}]
	    set left [expr {$left ^ ($temp << 4)}]
	    set temp [expr {(($right >> 16) ^ $left) & 0x0000ffff}]
	    set left [expr {$left ^ $temp}]
	    set right [expr {$right ^ ($temp << 16)}]
	    set temp [expr {(($left >> 2) ^ $right) & 0x33333333}]
	    set right [expr {$right ^ $temp}]
	    set left [expr {$left ^ ($temp << 2)}]
	    set temp [expr {(($right >> 16) ^ $left) & 0x0000ffff}]
	    set left [expr {$left ^ $temp}]
	    set right [expr {$right ^ ($temp << 16)}]
	    set temp [expr {(($left >> 1) ^ $right) & 0x55555555}]
	    set right [expr {$right ^ $temp}]
	    set left [expr {$left ^ ($temp << 1)}]
	    set temp [expr {(($right >> 8) ^ $left) & 0x00ff00ff}]
	    set left [expr {$left ^ $temp}]
	    set right [expr {$right ^ ($temp << 8)}]
	    set temp [expr {(($left >> 1) ^ $right) & 0x55555555}]
	    set right [expr {$right ^ $temp}]
	    set left [expr {$left ^ ($temp << 1)}]
	    
	    #puts "Left key PC1: [format %x $left]"
	    #puts "Right key PC1: [format %x $right]"

	    # The right side needs to be shifted and to get
	    # the last four bits of the left side
	    set temp [expr {($left << 8) | (($right >> 20) & 0x000000f0)}];
	    # Left needs to be put upside down
	    set left [expr {($right << 24) | (($right << 8) & 0x00ff0000) | \
				(($right >> 8) & 0x0000ff00) \
				| (($right >> 24) & 0x000000f0)}];
	    set right $temp;

	    #puts "Left key juggle: [format %x $left]"
	    #puts "Right key juggle: [format %x $right]"

	    # Now go through and perform these
	    # shifts on the left and right keys.
	    foreach i $shifts  {
		# Shift the keys either one or two bits to the left.
		if {$i} {
		    set left [expr {($left << 2) \
					| (($left >> 26) & 0x0000003f)}];
		    set right [expr {($right << 2) \
					 | (($right >> 26) & 0x0000003f)}];
		} else {
		    set left [expr {($left << 1) \
					| (($left >> 27) & 0x0000001f)}];
		    set right [expr {($right << 1) \
					 | (($right >> 27) & 0x0000001f)}];
		}
		set left [expr {$left & 0xfffffff0}];
		set right [expr {$right & 0xfffffff0}];

		# Now apply PC-2, in such a way that E is easier when
		# encrypting or decrypting this conversion will look like PC-2
		# except only the last 6 bits of each byte are used rather than
		# 48 consecutive bits and the order of lines will be according
		# to how the S selection functions will be applied: S2, S4, S6,
		# S8, S1, S3, S5, S7.
		set lefttemp [expr {[lindex $pc2bytes0 [expr {($left >> 28) & 0x0000000f}]] | \
					[lindex $pc2bytes1 [expr {($left >> 24) & 0x0000000f}]] | \
					[lindex $pc2bytes2 [expr {($left >> 20) & 0x0000000f}]] | \
					[lindex $pc2bytes3 [expr {($left >> 16) & 0x0000000f}]] | \
					[lindex $pc2bytes4 [expr {($left >> 12) & 0x0000000f}]] | \
					[lindex $pc2bytes5 [expr {($left >> 8) & 0x0000000f}]] | \
					[lindex $pc2bytes6 [expr {($left >> 4) & 0x0000000f}]]}];
		set righttemp [expr {[lindex $pc2bytes7 [expr {($right >> 28) & 0x0000000f}]] | \
					 [lindex $pc2bytes8 [expr {($right >> 24) & 0x0000000f}]] | \
					 [lindex $pc2bytes9 [expr {($right >> 20) & 0x0000000f}]] | \
					 [lindex $pc2bytes10 [expr {($right >> 16) & 0x0000000f}]] | \
					 [lindex $pc2bytes11 [expr {($right >> 12) & 0x0000000f}]] | \
					 [lindex $pc2bytes12 [expr {($right >> 8) & 0x0000000f}]] | \
					 [lindex $pc2bytes13 [expr {($right >> 4) & 0x0000000f}]]}];
		set temp [expr {(($righttemp >> 16) ^ $lefttemp) & 0x0000ffff}];
		lappend keys [expr {$lefttemp ^ $temp}];
		lappend keys [expr {$righttemp ^ ($temp << 16)}];
	    }
	}; # For each iteration.
	# Return the keys we've created.
	return $keys;
    }; # End of createKeys.
}; # End of des namespace eval.

package provide tclDES 1.0.0
# des.tcl - Copyright (C) 2005 Pat Thoyts <patthoyts@users.sourceforge.net>
#
# Tcllib wrapper for the DES package. This wrapper provides the same 
# programming API that tcllib uses for AES and Blowfish. We require a
# DES implementation and use either TclDES or TclDESjr to get DES 
# and/or 3DES
#
# -------------------------------------------------------------------------
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# -------------------------------------------------------------------------

package require Tcl 8.2

if {[catch {package require tclDES 1.0.0}]} {
    package require tclDESjr 1.0.0
}

namespace eval DES {
    variable uid
    if {![info exists uid]} { set uid 0 }
}

proc ::DES::Init {mode key iv {weak 0}} {
    variable uid
    set Key [namespace current]::[incr uid]
    upvar #0 $Key state
    if {[string length $key] % 8 != 0} {
        return -code error "invalid key length of\
             [expr {[string length $key] * 8}] bits:\
             DES requires 64 bit keys (56 bits plus parity bits)"
    }
    array set state [list M $mode I $iv K [des::keyset create $key $weak]]
    return $Key
}

proc ::DES::Encrypt {Key data} {
    upvar #0 $Key state
    set iv $state(I)
    set r [des::encrypt $state(K) $data $state(M) iv]
    set state(I) $iv
    return $r
}

proc ::DES::Decrypt {Key data} {
    upvar #0 $Key state
    set iv $state(I)
    set r [des::decrypt $state(K) $data $state(M) iv]
    set state(I) $iv
    return $r
}

proc ::DES::Reset {Key iv} {
    upvar #0 $Key state
    set state(I) $iv
    return
}

proc ::DES::Final {Key} {
    upvar #0 $Key state
    des::keyset destroy $state(K)
    # FRINK: nocheck
    unset $Key
}
# -------------------------------------------------------------------------

# Backwards compatability - here we re-implement the DES 0.8 procs using the
# current implementation.
#
# -- DO NOT USE THESE FUNCTIONS IN NEW CODE--
#
proc ::DES::GetKey {mode keydata keyvarname} {
    set weak 1
    switch -exact -- $mode {
        -encrypt    { set dir encrypt ; set vnc 0 }
        -encryptVNC { set dir encrypt ; set vnc 1 }
        -decrypt    { set dir decrypt ; set vnc 0 }
        -decryptVNC { set dir decrypt ; set vnc 1 }
        default {
            return -code error "invalid mode \"$mode\":\
                must be one of -encrypt, -decrypt, -encryptVNC or -decryptVNC"
        }
    }
    if {$vnc} { set keydata [ReverseBytes $keydata] }
    upvar $keyvarname Key
    set Key [Init ecb $keydata [string repeat \0 8] $weak]
    upvar $Key state
    array set state [list dir $dir]
    return
}

proc ::DES::DesBlock {data keyvarname} {
    upvar $keyvarname Key
    upvar #0 $Key state
    if {[string equal $state(dir) "encrypt"]} {
        set r [Encrypt $Key $data]
    } else {
        set r [Decrypt $Key $data]
    }
    return $r
}

proc ::DES::ReverseBytes {data} {
    binary scan $data b* bin
    return [binary format B* $bin]
}

# -------------------------------------------------------------------------

proc ::DES::SetOneOf {lst item} {
    set ndx [lsearch -glob $lst "${item}*"]
    if {$ndx == -1} {
        set err [join $lst ", "]
        return -code error "invalid mode \"$item\": must be one of $err"
    }
    return [lindex $lst $ndx]
}

proc ::DES::CheckSize {what size thing} {
    if {[string length $thing] != $size} {
        return -code error "invalid value for $what: must be $size bytes long"
    }
    return $thing
}

proc ::DES::Pad {data blocksize {fill \0}} {
    set len [string length $data]
    if {$len == 0} {
        set data [string repeat $fill $blocksize]
    } elseif {($len % $blocksize) != 0} {
        set pad [expr {$blocksize - ($len % $blocksize)}]
        append data [string repeat $fill $pad]
    }
    return $data
}

proc ::DES::Pop {varname {nth 0}} {
    upvar $varname args
    set r [lindex $args $nth]
    set args [lreplace $args $nth $nth]
    return $r
}

proc ::DES::Hex {data} {
    binary scan $data H* r
    return $r 
}

proc ::DES::des {args} {
    array set opts {
        -dir encrypt -mode cbc -key {} -in {} -out {} -chunksize 4096 -hex 0 -weak 0 old 0
    }
    set blocksize 8
    set opts(-iv) [string repeat \0 $blocksize]
    set modes {ecb cbc cfb ofb}
    set dirs {encrypt decrypt}
    while {[string match -* [set option [lindex $args 0]]]} {
        switch -exact -- $option {
            -mode      { 
                set M [Pop args 1]
                if {[catch {set mode [SetOneOf $modes $M]} err]} {
                    if {[catch {SetOneOf {encode decode} $M}]} {
                        return -code error $err
                    } else {
                        # someone is using the old interface, therefore ecb
                        set mode ecb
                        set opts(-weak) 1
                        set opts(old) 1
                        set opts(-dir) [expr {[string match en* $M] ? "encrypt" : "decrypt"}]
                    }
                }
                set opts(-mode) $mode
            }
            -dir       { set opts(-dir) [SetOneOf $dirs [Pop args 1]] }
            -iv        { set opts(-iv) [Pop args 1] }
            -key       { set opts(-key) [Pop args 1] }
            -in        { set opts(-in) [Pop args 1] }
            -out       { set opts(-out) [Pop args 1] }
            -chunksize { set opts(-chunksize) [Pop args 1] }
            -hex       { set opts(-hex) 1 }
            -weak      { set opts(-weak) 1 }
            --         { Pop args ; break }
            default {
                set err [join [lsort [array names opts -*]] ", "]
                return -code error "bad option \"$option\":\
                    must be one of $err"
            }
        }
        Pop args
    }

    if {$opts(-key) == {}} {
        return -code error "no key provided: the -key option is required"
    }

    # pad the key if backwards compat required
    if {$opts(old)} {
        set pad [expr {8 - ([string length $opts(-key)] % 8)}]
        if {$pad != 8} {
            append opts(-key) [string repeat \0 $pad]
        }
    }

    set r {}
    if {$opts(-in) == {}} {

        if {[llength $args] != 1} {
            return -code error "wrong \# args:\
                should be \"des ?options...? -key keydata plaintext\""
        }

        set data [Pad [lindex $args 0] $blocksize]
        set Key [Init $opts(-mode) $opts(-key) $opts(-iv) $opts(-weak)]
        if {[string equal $opts(-dir) "encrypt"]} {
            set r [Encrypt $Key $data]
        } else {
            set r [Decrypt $Key $data]
        }

        if {$opts(-out) != {}} {
            puts -nonewline $opts(-out) $r
            set r {}
        }
        Final $Key

    } else {

        if {[llength $args] != 0} {
            return -code error "wrong \# args:\
                should be \"des ?options...? -key keydata -in channel\""
        }

        set Key [Init $opts(-mode) $opts(-key) $opts(-iv) $opts(-weak)]
        upvar $Key state
        set state(reading) 1
        if {[string equal $opts(-dir) "encrypt"]} {
            set state(cmd) Encrypt
        } else {
            set state(cmd) Decrypt
        }
        set state(output) ""
        fileevent $opts(-in) readable \
            [list [namespace origin Chunk] \
                 $Key $opts(-in) $opts(-out) $opts(-chunksize)]
        if {[info commands ::tkwait] != {}} {
            tkwait variable [subst $Key](reading)
        } else {
            vwait [subst $Key](reading)
        }
        if {$opts(-out) == {}} {
            set r $state(output)
        }
        Final $Key

    }

    if {$opts(-hex)} {
        set r [Hex $r]
    }
    return $r
}

# -------------------------------------------------------------------------

package provide des 1.1.0

# -------------------------------------------------------------------------
#
# Local variables:
#   mode: tcl
#   indent-tabs-mode: nil
# End:
# bignum library in pure Tcl [VERSION 7Sep2004]
# Copyright (C) 2004 Salvatore Sanfilippo <antirez at invece dot org>
# Copyright (C) 2004 Arjen Markus <arjen dot markus at wldelft dot nl>
#
# LICENSE
#
# This software is:
# Copyright (C) 2004 Salvatore Sanfilippo <antirez at invece dot org>
# Copyright (C) 2004 Arjen Markus <arjen dot markus at wldelft dot nl>
# The following terms apply to all files associated with the software
# unless explicitly disclaimed in individual files.
#
# The authors hereby grant permission to use, copy, modify, distribute,
# and license this software and its documentation for any purpose, provided
# that existing copyright notices are retained in all copies and that this
# notice is included verbatim in any distributions. No written agreement,
# license, or royalty fee is required for any of the authorized uses.
# Modifications to this software may be copyrighted by their authors
# and need not follow the licensing terms described here, provided that
# the new terms are clearly indicated on the first page of each file where
# they apply.
#
# IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY PARTY
# FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
# ARISING OUT OF THE USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY
# DERIVATIVES THEREOF, EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#
# THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THIS SOFTWARE
# IS PROVIDED ON AN "AS IS" BASIS, AND THE AUTHORS AND DISTRIBUTORS HAVE
# NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR
# MODIFICATIONS.
#
# GOVERNMENT USE: If you are acquiring this software on behalf of the
# U.S. government, the Government shall have only "Restricted Rights"
# in the software and related documentation as defined in the Federal
# Acquisition Regulations (FARs) in Clause 52.227.19 (c) (2).  If you
# are acquiring the software on behalf of the Department of Defense, the
# software shall be classified as "Commercial Computer Software" and the
# Government shall have only "Restricted Rights" as defined in Clause
# 252.227-7013 (c) (1) of DFARs.  Notwithstanding the foregoing, the
# authors grant the U.S. Government and others acting in its behalf
# permission to use and distribute the software in accordance with the
# terms specified in this license.

# TODO
# - pow and powm should check if the exponent is zero in order to return one

package require Tcl 8.4

namespace eval ::math::bignum {}

#################################### Misc ######################################

# Don't change atombits define if you don't know what you are doing.
# Note that it must be a power of two, and that 16 is too big
# because expr may overflow in the product of two 16 bit numbers.
set ::math::bignum::atombits 16
set ::math::bignum::atombase [expr {1 << $::math::bignum::atombits}]
set ::math::bignum::atommask [expr {$::math::bignum::atombase-1}]

# Note: to change 'atombits' is all you need to change the
# library internal representation base.

# Return the max between a and b (not bignums)
proc ::math::bignum::max {a b} {
    expr {($a > $b) ? $a : $b}
}

# Return the min between a and b (not bignums)
proc ::math::bignum::min {a b} {
    expr {($a < $b) ? $a : $b}
}

############################ Basic bignum operations ###########################

# Returns a new bignum initialized to the value of 0.
#
# The big numbers are represented as a Tcl lists
# The all-is-a-string representation does not pay here
# bignums in Tcl are already slow, we can't slow-down it more.
#
# The bignum representation is [list bignum <sign> <atom0> ... <atomN>]
# Where the atom0 is the least significant. Atoms are the digits
# of a number in base 2^$::math::bignum::atombits
#
# The sign is 0 if the number is positive, 1 for negative numbers.

# Note that the function accepts an argument used in order to
# create a bignum of <atoms> atoms. For default zero is
# represented as a single zero atom.
#
# The function is designed so that "set b [zero [atoms $a]]" will
# produce 'b' with the same number of atoms as 'a'.
proc ::math::bignum::zero {{value 0}} {
    set v [list bignum 0 0]
    while { $value > 1 } {
       lappend v 0
       incr value -1
    }
    return $v
}

# Get the bignum sign
proc ::math::bignum::sign bignum {
    lindex $bignum 1
}

# Get the number of atoms in the bignum
proc ::math::bignum::atoms bignum {
    expr {[llength $bignum]-2}
}

# Get the i-th atom out of a bignum.
# If the bignum is shorter than i atoms, the function
# returns 0.
proc ::math::bignum::atom {bignum i} {
    if {[::math::bignum::atoms $bignum] < [expr {$i+1}]} {
	return 0
    } else {
	lindex $bignum [expr {$i+2}]
    }
}

# Set the i-th atom out of a bignum. If the bignum
# has less than 'i+1' atoms, add zero atoms to reach i.
proc ::math::bignum::setatom {bignumvar i atomval} {
    upvar 1 $bignumvar bignum
    while {[::math::bignum::atoms $bignum] < [expr {$i+1}]} {
	lappend bignum 0
    }
    lset bignum [expr {$i+2}] $atomval
}

# Set the bignum sign
proc ::math::bignum::setsign {bignumvar sign} {
    upvar 1 $bignumvar bignum
    lset bignum 1 $sign
}

# Remove trailing atoms with a value of zero
# The normalized bignum is returned
proc ::math::bignum::normalize bignumvar {
    upvar 1 $bignumvar bignum
    set atoms [expr {[llength $bignum]-2}]
    set i [expr {$atoms+1}]
    while {$atoms && [lindex $bignum $i] == 0} {
	set bignum [lrange $bignum 0 end-1]
	incr atoms -1
	incr i -1
    }
    if {!$atoms} {
	set bignum [list bignum 0 0]
    }
    return $bignum
}

# Return the absolute value of N
proc ::math::bignum::abs n {
    ::math::bignum::setsign n 0
    return $n
}

################################# Comparison ###################################

# Compare by absolute value. Called by ::math::bignum::cmp after the sign check.
#
# Returns 1 if |a| > |b|
#         0 if a == b
#        -1 if |a| < |b|
#
proc ::math::bignum::abscmp {a b} {
    if {[llength $a] > [llength $b]} {
	return 1
    } elseif {[llength $a] < [llength $b]} {
	return -1
    }
    set j [expr {[llength $a]-1}]
    while {$j >= 2} {
	if {[lindex $a $j] > [lindex $b $j]} {
	    return 1
	} elseif {[lindex $a $j] < [lindex $b $j]} {
	    return -1
	}
	incr j -1
    }
    return 0
}

# High level comparison. Return values:
#
#  1 if a > b
# -1 if a < b
#  0 if a == b
#
proc ::math::bignum::cmp {a b} { ; # same sign case
    set a [_treat $a]
    set b [_treat $b]
    if {[::math::bignum::sign $a] == [::math::bignum::sign $b]} {
	if {[::math::bignum::sign $a] == 0} {
	    ::math::bignum::abscmp $a $b
	} else {
	    expr {-([::math::bignum::abscmp $a $b])}
	}
    } else { ; # different sign case
	if {[::math::bignum::sign $a]} {return -1}
	return 1
    }
}

# Return true if 'z' is zero.
proc ::math::bignum::iszero z {
    set z [_treat $z]
    expr {[llength $z] == 3 && [lindex $z 2] == 0}
}

# Comparison facilities
proc ::math::bignum::lt {a b} {expr {[::math::bignum::cmp $a $b] < 0}}
proc ::math::bignum::le {a b} {expr {[::math::bignum::cmp $a $b] <= 0}}
proc ::math::bignum::gt {a b} {expr {[::math::bignum::cmp $a $b] > 0}}
proc ::math::bignum::ge {a b} {expr {[::math::bignum::cmp $a $b] >= 0}}
proc ::math::bignum::eq {a b} {expr {[::math::bignum::cmp $a $b] == 0}}
proc ::math::bignum::ne {a b} {expr {[::math::bignum::cmp $a $b] != 0}}

########################### Addition / Subtraction #############################

# Add two bignums, don't care about the sign.
proc ::math::bignum::rawAdd {a b} {
    while {[llength $a] < [llength $b]} {lappend a 0}
    while {[llength $b] < [llength $a]} {lappend b 0}
    set r [::math::bignum::zero [expr {[llength $a]-1}]]
    set car 0
    for {set i 2} {$i < [llength $a]} {incr i} {
	set sum [expr {[lindex $a $i]+[lindex $b $i]+$car}]
	set car [expr {$sum >> $::math::bignum::atombits}]
	set sum [expr {$sum & $::math::bignum::atommask}]
	lset r $i $sum
    }
    if {$car} {
	lset r $i $car
    }
    ::math::bignum::normalize r
}

# Subtract two bignums, don't care about the sign. a > b condition needed.
proc ::math::bignum::rawSub {a b} {
    set atoms [::math::bignum::atoms $a]
    set r [::math::bignum::zero $atoms]
    while {[llength $b] < [llength $a]} {lappend b 0} ; # b padding
    set car 0
    incr atoms 2
    for {set i 2} {$i < $atoms} {incr i} {
	set sub [expr {[lindex $a $i]-[lindex $b $i]-$car}]
	set car 0
	if {$sub < 0} {
	    incr sub $::math::bignum::atombase
	    set car 1
	}
	lset r $i $sub
    }
    # Note that if a > b there is no car in the last for iteration
    ::math::bignum::normalize r
}

# Higher level addition, care about sign and call rawAdd or rawSub
# as needed.
proc ::math::bignum::add {a b} {
    set a [_treat $a]
    set b [_treat $b]
    # Same sign case
    if {[::math::bignum::sign $a] == [::math::bignum::sign $b]} {
	set r [::math::bignum::rawAdd $a $b]
	::math::bignum::setsign r [::math::bignum::sign $a]
    } else {
	# Different sign case
	set cmp [::math::bignum::abscmp $a $b]
	# 's' is the sign, set accordingly to A or B negative
	set s [expr {[::math::bignum::sign $a] == 1}]
	switch -- $cmp {
	    0 {return [::math::bignum::zero]}
	    1 {
		set r [::math::bignum::rawSub $a $b]
		::math::bignum::setsign r $s
		return $r
	    }
	    -1 {
		set r [::math::bignum::rawSub $b $a]
		::math::bignum::setsign r [expr {!$s}]
		return $r
	    }
	}
    }
    return $r
}

# Higher level subtraction, care about sign and call rawAdd or rawSub
# as needed.
proc ::math::bignum::sub {a b} {
    set a [_treat $a]
    set b [_treat $b]
    # Different sign case
    if {[::math::bignum::sign $a] != [::math::bignum::sign $b]} {
	set r [::math::bignum::rawAdd $a $b]
	::math::bignum::setsign r [::math::bignum::sign $a]
    } else {
	# Same sign case
	set cmp [::math::bignum::abscmp $a $b]
	# 's' is the sign, set accordingly to A and B both negative or positive
	set s [expr {[::math::bignum::sign $a] == 1}]
	switch -- $cmp {
	    0 {return [::math::bignum::zero]}
	    1 {
		set r [::math::bignum::rawSub $a $b]
		::math::bignum::setsign r $s
		return $r
	    }
	    -1 {
		set r [::math::bignum::rawSub $b $a]
		::math::bignum::setsign r [expr {!$s}]
		return $r
	    }
	}
    }
    return $r
}

############################### Multiplication #################################

set ::math::bignum::karatsubaThreshold 32

# Multiplication. Calls Karatsuba that calls Base multiplication under
# a given threshold.
proc ::math::bignum::mul {a b} {
    set a [_treat $a]
    set b [_treat $b]
    set r [::math::bignum::kmul $a $b]
    # The sign is the xor between the two signs
    ::math::bignum::setsign r [expr {[::math::bignum::sign $a]^[::math::bignum::sign $b]}]
}

# Karatsuba Multiplication
proc ::math::bignum::kmul {a b} {
    set n [expr {[::math::bignum::max [llength $a] [llength $b]]-2}]
    set nmin [expr {[::math::bignum::min [llength $a] [llength $b]]-2}]
    if {$nmin < $::math::bignum::karatsubaThreshold} {return [::math::bignum::bmul $a $b]}
    set m [expr {($n+($n&1))/2}]

    set x0 [concat [list bignum 0] [lrange $a 2 [expr {$m+1}]]]
    set y0 [concat [list bignum 0] [lrange $b 2 [expr {$m+1}]]]
    set x1 [concat [list bignum 0] [lrange $a [expr {$m+2}] end]]
    set y1 [concat [list bignum 0] [lrange $b [expr {$m+2}] end]]

    if {0} {
    puts "m: $m"
    puts "x0: $x0"
    puts "x1: $x1"
    puts "y0: $y0"
    puts "y1: $y1"
    }

    set p1 [::math::bignum::kmul $x1 $y1]
    set p2 [::math::bignum::kmul $x0 $y0]
    set p3 [::math::bignum::kmul [::math::bignum::add $x1 $x0] [::math::bignum::add $y1 $y0]]

    set p3 [::math::bignum::sub $p3 $p1]
    set p3 [::math::bignum::sub $p3 $p2]
    set p1 [::math::bignum::lshiftAtoms $p1 [expr {$m*2}]]
    set p3 [::math::bignum::lshiftAtoms $p3 $m]
    set p3 [::math::bignum::add $p3 $p1]
    set p3 [::math::bignum::add $p3 $p2]
    return $p3
}

# Base Multiplication.
proc ::math::bignum::bmul {a b} {
    set r [::math::bignum::zero [expr {[llength $a]+[llength $b]-3}]]
    for {set j 2} {$j < [llength $b]} {incr j} {
	set car 0
	set t [list bignum 0 0]
	for {set i 2} {$i < [llength $a]} {incr i} {
	    # note that A = B * C + D + E
	    # with A of N*2 bits and C,D,E of N bits
	    # can't overflow since:
	    # (2^N-1)*(2^N-1)+(2^N-1)+(2^N-1) == 2^(2*N)-1
	    set t0 [lindex $a $i]
	    set t1 [lindex $b $j]
	    set t2 [lindex $r [expr {$i+$j-2}]]
	    set mul [expr {wide($t0)*$t1+$t2+$car}]
	    set car [expr {$mul >> $::math::bignum::atombits}]
	    set mul [expr {$mul & $::math::bignum::atommask}]
	    lset r [expr {$i+$j-2}] $mul
	}
	if {$car} {
	    lset r [expr {$i+$j-2}] $car
	}
    }
    ::math::bignum::normalize r
}

################################## Shifting ####################################

# Left shift 'z' of 'n' atoms. Low-level function used by ::math::bignum::lshift
# Exploit the internal representation to go faster.
proc ::math::bignum::lshiftAtoms {z n} {
    while {$n} {
	set z [linsert $z 2 0]
	incr n -1
    }
    return $z
}

# Right shift 'z' of 'n' atoms. Low-level function used by ::math::bignum::lshift
# Exploit the internal representation to go faster.
proc ::math::bignum::rshiftAtoms {z n} {
    set z [lreplace $z 2 [expr {$n+1}]]
}

# Left shift 'z' of 'n' bits. Low-level function used by ::math::bignum::lshift.
# 'n' must be <= $::math::bignum::atombits
proc ::math::bignum::lshiftBits {z n} {
    set atoms [llength $z]
    set car 0
    for {set j 2} {$j < $atoms} {incr j} {
	set t [lindex $z $j]
	lset z $j \
	    [expr {wide($car)|((wide($t)<<$n)&$::math::bignum::atommask)}]
	set car [expr {wide($t)>>($::math::bignum::atombits-$n)}]
    }
    if {$car} {
	lappend z 0
	lset z $j $car
    }
    return $z ; # No normalization needed
}

# Right shift 'z' of 'n' bits. Low-level function used by ::math::bignum::rshift.
# 'n' must be <= $::math::bignum::atombits
proc ::math::bignum::rshiftBits {z n} {
    set atoms [llength $z]
    set car 0
    for {set j [expr {$atoms-1}]} {$j >= 2} {incr j -1} {
	set t [lindex $z $j]
	lset z $j [expr {wide($car)|(wide($t)>>$n)}]
	set car \
	    [expr {(wide($t)<<($::math::bignum::atombits-$n))&$::math::bignum::atommask}]
    }
    ::math::bignum::normalize z
}

# Left shift 'z' of 'n' bits.
proc ::math::bignum::lshift {z n} {
    set z [_treat $z]
    set atoms [expr {$n / $::math::bignum::atombits}]
    set bits [expr {$n & ($::math::bignum::atombits-1)}]
    ::math::bignum::lshiftBits [math::bignum::lshiftAtoms $z $atoms] $bits
}

# Right shift 'z' of 'n' bits.
proc ::math::bignum::rshift {z n} {
    set z [_treat $z]
    set atoms [expr {$n / $::math::bignum::atombits}]
    set bits [expr {$n & ($::math::bignum::atombits-1)}]

    #
    # Correct for "arithmetic shift" - signed integers
    #
    set corr 0
    if { [::math::bignum::sign $z] == 1 } {
        for {set j [expr {$atoms+1}]} {$j >= 2} {incr j -1} {
            set t [lindex $z $j]
            if { $t != 0 } {
                set corr 1
            }
        }
        if { $corr == 0 } {
            set t [lindex $z [expr {$atoms+2}]]
            if { ( $t & ~($::math::bignum::atommask<<($bits)) ) != 0 } {
                set corr 1
            }
        }
    }

    set newz [::math::bignum::rshiftBits [math::bignum::rshiftAtoms $z $atoms] $bits]
    if { $corr } {
        set newz [::math::bignum::sub $newz 1]
    }
    return $newz
}

############################## Bit oriented ops ################################

# Set the bit 'n' of 'bignumvar'
proc ::math::bignum::setbit {bignumvar n} {
    upvar 1 $bignumvar z
    set atom [expr {$n / $::math::bignum::atombits}]
    set bit [expr {1 << ($n & ($::math::bignum::atombits-1))}]
    incr atom 2
    while {$atom >= [llength $z]} {lappend z 0}
    lset z $atom [expr {[lindex $z $atom]|$bit}]
}

# Clear the bit 'n' of 'bignumvar'
proc ::math::bignum::clearbit {bignumvar n} {
    upvar 1 $bignumvar z
    set atom [expr {$n / $::math::bignum::atombits}]
    incr atom 2
    if {$atom >= [llength $z]} {return $z}
    set mask [expr {$::math::bignum::atommask^(1 << ($n & ($::math::bignum::atombits-1)))}]
    lset z $atom [expr {[lindex $z $atom]&$mask}]
    ::math::bignum::normalize z
}

# Test the bit 'n' of 'z'. Returns true if the bit is set.
proc ::math::bignum::testbit {z n} {
    set  atom [expr {$n / $::math::bignum::atombits}]
    incr atom 2
    if {$atom >= [llength $z]} {return 0}
    set mask [expr {1 << ($n & ($::math::bignum::atombits-1))}]
    expr {([lindex $z $atom] & $mask) != 0}
}

# does bitwise and between a and b
proc ::math::bignum::bitand {a b} {
    # The internal number rep is little endian. Appending zeros is
    # equivalent to adding leading zeros to a regular big-endian
    # representation. The two numbers are extended to the same length,
    # then the operation is applied to the absolute value.
    set a [_treat $a]
    set b [_treat $b]
    while {[llength $a] < [llength $b]} {lappend a 0}
    while {[llength $b] < [llength $a]} {lappend b 0}
    set r [::math::bignum::zero [expr {[llength $a]-1}]]
    for {set i 2} {$i < [llength $a]} {incr i} {
	set or [expr {[lindex $a $i] & [lindex $b $i]}]
	lset r $i $or
    }
    ::math::bignum::normalize r
}

# does bitwise XOR between a and b
proc ::math::bignum::bitxor {a b} {
    # The internal number rep is little endian. Appending zeros is
    # equivalent to adding leading zeros to a regular big-endian
    # representation. The two numbers are extended to the same length,
    # then the operation is applied to the absolute value.
    set a [_treat $a]
    set b [_treat $b]
    while {[llength $a] < [llength $b]} {lappend a 0}
    while {[llength $b] < [llength $a]} {lappend b 0}
    set r [::math::bignum::zero [expr {[llength $a]-1}]]
    for {set i 2} {$i < [llength $a]} {incr i} {
	set or [expr {[lindex $a $i] ^ [lindex $b $i]}]
	lset r $i $or
    }
    ::math::bignum::normalize r
}

# does bitwise or between a and b
proc ::math::bignum::bitor {a b} {
    # The internal number rep is little endian. Appending zeros is
    # equivalent to adding leading zeros to a regular big-endian
    # representation. The two numbers are extended to the same length,
    # then the operation is applied to the absolute value.
    set a [_treat $a]
    set b [_treat $b]
    while {[llength $a] < [llength $b]} {lappend a 0}
    while {[llength $b] < [llength $a]} {lappend b 0}
    set r [::math::bignum::zero [expr {[llength $a]-1}]]
    for {set i 2} {$i < [llength $a]} {incr i} {
	set or [expr {[lindex $a $i] | [lindex $b $i]}]
	lset r $i $or
    }
    ::math::bignum::normalize r
}

# Return the number of bits needed to represent 'z'.
proc ::math::bignum::bits z {
    set atoms [::math::bignum::atoms $z]
    set bits [expr {($atoms-1)*$::math::bignum::atombits}]
    set atom [lindex $z [expr {$atoms+1}]]
    while {$atom} {
	incr bits
	set atom [expr {$atom >> 1}]
    }
    return $bits
}

################################## Division ####################################

# Division. Returns [list n/d n%d]
#
# I got this algorithm from PGP 2.6.3i (see the mp_udiv function).
# Here is how it works:
#
# Input:  N=(Nn,...,N2,N1,N0)radix2
#         D=(Dn,...,D2,D1,D0)radix2
# Output: Q=(Qn,...,Q2,Q1,Q0)radix2 = N/D
#         R=(Rn,...,R2,R1,R0)radix2 = N%D
#
# Assume: N >= 0, D > 0
#
# For j from 0 to n
#      Qj <- 0
#      Rj <- 0
# For j from n down to 0
#      R <- R*2
#      if Nj = 1 then R0 <- 1
#      if R => D then R <- (R - D), Qn <- 1
#
# Note that the doubling of R is usually done leftshifting one position.
# The only operations needed are bit testing, bit setting and subtraction.
#
# This is the "raw" version, don't care about the sign, returns both
# quotient and rest as a two element list.
# This procedure is used by divqr, div, mod, rem.
proc ::math::bignum::rawDiv {n d} {
    set bit [expr {[::math::bignum::bits $n]-1}]
    set r [list bignum 0 0]
    set q [::math::bignum::zero [expr {[llength $n]-2}]]
    while {$bit >= 0} {
	set b_atom [expr {($bit / $::math::bignum::atombits) + 2}]
	set b_bit [expr {1 << ($bit & ($::math::bignum::atombits-1))}]
	set r [::math::bignum::lshiftBits $r 1]
	if {[lindex $n $b_atom]&$b_bit} {
	    lset r 2 [expr {[lindex $r 2] | 1}]
	}
	if {[::math::bignum::abscmp $r $d] >= 0} {
	    set r [::math::bignum::rawSub $r $d]
	    lset q $b_atom [expr {[lindex $q $b_atom]|$b_bit}]
	}
	incr bit -1
    }
    ::math::bignum::normalize q
    list $q $r
}

# Divide by single-atom immediate. Used to speedup bignum -> string conversion.
# The procedure returns a two-elements list with the bignum quotient and
# the remainder (that's just a number being <= of the max atom value).
proc ::math::bignum::rawDivByAtom {n d} {
    set atoms [::math::bignum::atoms $n]
    set t 0
    set j $atoms
    incr j -1
    for {} {$j >= 0} {incr j -1} {
	set t [expr {($t << $::math::bignum::atombits)+[lindex $n [expr {$j+2}]]}]
	lset n [expr {$j+2}] [expr {$t/$d}]
	set t [expr {$t % $d}]
    }
    ::math::bignum::normalize n
    list $n $t
}

# Higher level division. Returns a list with two bignums, the first
# is the quotient of n/d, the second the remainder n%d.
# Note that if you want the *modulo* operator you should use ::math::bignum::mod
#
# The remainder sign is always the same as the divident.
proc ::math::bignum::divqr {n d} {
    set n [_treat $n]
    set d [_treat $d]
    if {[::math::bignum::iszero $d]} {
	error "Division by zero"
    }
    foreach {q r} [::math::bignum::rawDiv $n $d] break
    ::math::bignum::setsign q [expr {[::math::bignum::sign $n]^[::math::bignum::sign $d]}]
    ::math::bignum::setsign r [::math::bignum::sign $n]
    list $q $r
}

# Like divqr, but only the quotient is returned.
proc ::math::bignum::div {n d} {
    lindex [::math::bignum::divqr $n $d] 0
}

# Like divqr, but only the remainder is returned.
proc ::math::bignum::rem {n d} {
    lindex [::math::bignum::divqr $n $d] 1
}

# Modular reduction. Returns N modulo M
proc ::math::bignum::mod {n m} {
    set n [_treat $n]
    set m [_treat $m]
    set r [lindex [::math::bignum::divqr $n $m] 1]
    if {[::math::bignum::sign $m] != [::math::bignum::sign $r]} {
	set r [::math::bignum::add $r $m]
    }
    return $r
}

# Returns true if n is odd
proc ::math::bignum::isodd n {
    expr {[lindex $n 2]&1}
}

# Returns true if n is even
proc ::math::bignum::iseven n {
    expr {!([lindex $n 2]&1)}
}

############################# Power and Power mod N ############################

# Returns b^e
proc ::math::bignum::pow {b e} {
    set b [_treat $b]
    set e [_treat $e]
    if {[::math::bignum::iszero $e]} {return [list bignum 0 1]}
    # The power is negative is the base is negative and the exponent is odd
    set sign [expr {[::math::bignum::sign $b] && [::math::bignum::isodd $e]}]
    # Set the base to it's abs value, i.e. make it positive
    ::math::bignum::setsign b 0
    # Main loop
    set r [list bignum 0 1]; # Start with result = 1
    while {[::math::bignum::abscmp $e [list bignum 0 1]] > 0} { ;# While the exp > 1
	if {[::math::bignum::isodd $e]} {
	    set r [::math::bignum::mul $r $b]
	}
	set e [::math::bignum::rshiftBits $e 1] ;# exp = exp / 2
	set b [::math::bignum::mul $b $b]
    }
    set r [::math::bignum::mul $r $b]
    ::math::bignum::setsign r $sign
    return $r
}

# Returns b^e mod m
proc ::math::bignum::powm {b e m} {
    set b [_treat $b]
    set e [_treat $e]
    set m [_treat $m]
    if {[::math::bignum::iszero $e]} {return [list bignum 0 1]}
    # The power is negative is the base is negative and the exponent is odd
    set sign [expr {[::math::bignum::sign $b] && [::math::bignum::isodd $e]}]
    # Set the base to it's abs value, i.e. make it positive
    ::math::bignum::setsign b 0
    # Main loop
    set r [list bignum 0 1]; # Start with result = 1
    while {[::math::bignum::abscmp $e [list bignum 0 1]] > 0} { ;# While the exp > 1
	if {[::math::bignum::isodd $e]} {
	    set r [::math::bignum::mod [::math::bignum::mul $r $b] $m]
	}
	set e [::math::bignum::rshiftBits $e 1] ;# exp = exp / 2
	set b [::math::bignum::mod [::math::bignum::mul $b $b] $m]
    }
    set r [::math::bignum::mul $r $b]
    ::math::bignum::setsign r $sign
    set r [::math::bignum::mod $r $m]
    return $r
}

################################## Square Root #################################

# SQRT using the 'binary sqrt algorithm'.
#
# The basic algoritm consists in starting from the higer-bit
# the real square root may have set, down to the bit zero,
# trying to set every bit and checking if guess*guess is not
# greater than 'n'. If it is greater we don't set the bit, otherwise
# we set it. In order to avoid to compute guess*guess a trick
# is used, so only addition and shifting are really required.
proc ::math::bignum::sqrt n {
    if {[lindex $n 1]} {
	error "Square root of a negative number"
    }
    set i [expr {(([::math::bignum::bits $n]-1)/2)+1}]
    set b [expr {$i*2}] ; # Bit to set to get 2^i*2^i

    set r [::math::bignum::zero] ; # guess
    set x [::math::bignum::zero] ; # guess^2
    set s [::math::bignum::zero] ; # guess^2 backup
    set t [::math::bignum::zero] ; # intermediate result
    for {} {$i >= 0} {incr i -1; incr b -2} {
	::math::bignum::setbit t $b
	set x [::math::bignum::rawAdd $s $t]
	::math::bignum::clearbit t $b
	if {[::math::bignum::abscmp $x $n] <= 0} {
	    set s $x
	    ::math::bignum::setbit r $i
	    ::math::bignum::setbit t [expr {$b+1}]
	}
	set t [::math::bignum::rshiftBits $t 1]
    }
    return $r
}

################################## Random Number ###############################

# Returns a random number in the range [0,2^n-1]
proc ::math::bignum::rand bits {
    set atoms [expr {($bits+$::math::bignum::atombits-1)/$::math::bignum::atombits}]
    set shift [expr {($atoms*$::math::bignum::atombits)-$bits}]
    set r [list bignum 0]
    while {$atoms} {
	lappend r [expr {int(rand()*(1<<$::math::bignum::atombits))}]
	incr atoms -1
    }
    set r [::math::bignum::rshiftBits $r $shift]
    return $r
}

############################ Convertion to/from string #########################

# The string representation charset. Max base is 36
set ::math::bignum::cset "0123456789abcdefghijklmnopqrstuvwxyz"

# Convert 'z' to a string representation in base 'base'.
# Note that this is missing a simple but very effective optimization
# that's to divide by the biggest power of the base that fits
# in a Tcl plain integer, and then to perform divisions with [expr].
proc ::math::bignum::tostr {z {base 10}} {
    if {[string length $::math::bignum::cset] < $base} {
	error "base too big for string convertion"
    }
    if {[::math::bignum::iszero $z]} {return 0}
    set sign [::math::bignum::sign $z]
    set str {}
    while {![::math::bignum::iszero $z]} {
	foreach {q r} [::math::bignum::rawDivByAtom $z $base] break
	append str [string index $::math::bignum::cset $r]
	set z $q
    }
    if {$sign} {append str -}
    # flip the resulting string
    set flipstr {}
    set i [string length $str]
    incr i -1
    while {$i >= 0} {
	append flipstr [string index $str $i]
	incr i -1
    }
    return $flipstr
}

# Create a bignum from a string representation in base 'base'.
proc ::math::bignum::fromstr {str {base 0}} {
    set z [::math::bignum::zero]
    set str [string trim $str]
    set sign 0
    if {[string index $str 0] eq {-}} {
	set str [string range $str 1 end]
	set sign 1
    }
    if {$base == 0} {
	switch -- [string tolower [string range $str 0 1]] {
	    0x {set base 16; set str [string range $str 2 end]}
	    ox {set base 8 ; set str [string range $str 2 end]}
	    bx {set base 2 ; set str [string range $str 2 end]}
	    default {set base 10}
	}
    }
    if {[string length $::math::bignum::cset] < $base} {
	error "base too big for string convertion"
    }
    set bigbase [list bignum 0 $base] ; # Build a bignum with the base value
    set basepow [list bignum 0 1] ; # multiply every digit for a succ. power
    set i [string length $str]
    incr i -1
    while {$i >= 0} {
	set digitval [string first [string index $str $i] $::math::bignum::cset]
	if {$digitval == -1} {
	    error "Illegal char '[string index $str $i]' for base $base"
	}
	set bigdigitval [list bignum 0 $digitval]
	set z [::math::bignum::rawAdd $z [::math::bignum::mul $basepow $bigdigitval]]
	set basepow [::math::bignum::mul $basepow $bigbase]
	incr i -1
    }
    if {![::math::bignum::iszero $z]} {
	::math::bignum::setsign z $sign
    }
    return $z
}

#
# Pre-treatment of some constants : 0 and 1
# Updated 19/11/2005 : abandon the 'upvar' command and its cost
#
proc ::math::bignum::_treat {num} {
    if {[llength $num]<2} {
        if {[string equal $num 0]} {
            # set to the bignum 0
            return {bignum 0 0}
        } elseif {[string equal $num 1]} {
            # set to the bignum 1
            return {bignum 0 1}
        }
    }
    return $num
}

namespace eval ::math::bignum {
    namespace export *
}

# Announce the package

package provide math::bignum 3.1.1
# md5.tcl - Copyright (C) 2003 Pat Thoyts <patthoyts@users.sourceforge.net>
#
# MD5  defined by RFC 1321, "The MD5 Message-Digest Algorithm"
# HMAC defined by RFC 2104, "Keyed-Hashing for Message Authentication"
#
# This is an implementation of MD5 based upon the example code given in
# RFC 1321 and upon the tcllib MD4 implementation and taking some ideas
# from the earlier tcllib md5 version by Don Libes.
#
# This implementation permits incremental updating of the hash and 
# provides support for external compiled implementations either using
# critcl (md5c) or Trf.
#
# -------------------------------------------------------------------------
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# -------------------------------------------------------------------------

package require Tcl 8.2;                # tcl minimum version

namespace eval ::md5 {
    variable  accel
    array set accel {critcl 0 cryptkit 0 trf 0}

    namespace export md5 hmac MD5Init MD5Update MD5Final

    variable uid
    if {![info exists uid]} {
        set uid 0
    }
}

# -------------------------------------------------------------------------

# MD5Init --
#
#   Create and initialize an MD5 state variable. This will be
#   cleaned up when we call MD5Final
#
proc ::md5::MD5Init {} {
    variable accel
    variable uid
    set token [namespace current]::[incr uid]
    upvar #0 $token state

    # RFC1321:3.3 - Initialize MD5 state structure
    array set state \
        [list \
             A [expr {0x67452301}] \
             B [expr {0xefcdab89}] \
             C [expr {0x98badcfe}] \
             D [expr {0x10325476}] \
             n 0 i "" ]
    if {$accel(cryptkit)} {
        cryptkit::cryptCreateContext state(ckctx) CRYPT_UNUSED CRYPT_ALGO_MD5
    } elseif {$accel(trf)} {
        set s {}
        switch -exact -- $::tcl_platform(platform) {
            windows { set s [open NUL w] }
            unix    { set s [open /dev/null w] }
        }
        if {$s != {}} {
            fconfigure $s -translation binary -buffering none
            ::md5 -attach $s -mode write \
                -read-type variable \
                -read-destination [subst $token](trfread) \
                -write-type variable \
                -write-destination [subst $token](trfwrite)
            array set state [list trfread 0 trfwrite 0 trf $s]
        }
    }
    return $token
}

# MD5Update --
#
#   This is called to add more data into the hash. You may call this
#   as many times as you require. Note that passing in "ABC" is equivalent
#   to passing these letters in as separate calls -- hence this proc 
#   permits hashing of chunked data
#
#   If we have a C-based implementation available, then we will use
#   it here in preference to the pure-Tcl implementation.
#
proc ::md5::MD5Update {token data} {
    variable accel
    upvar #0 $token state

    if {$accel(critcl)} {
        if {[info exists state(md5c)]} {
            set state(md5c) [md5c $data $state(md5c)]
        } else {
            set state(md5c) [md5c $data]
        }
        return
    } elseif {[info exists state(ckctx)]} {
        if {[string length $data] > 0} {
            cryptkit::cryptEncrypt $state(ckctx) $data
        }
        return
    } elseif {[info exists state(trf)]} {
        puts -nonewline $state(trf) $data
        return
    }

    # Update the state values
    incr state(n) [string length $data]
    append state(i) $data

    # Calculate the hash for any complete blocks
    set len [string length $state(i)]
    for {set n 0} {($n + 64) <= $len} {} {
        MD5Hash $token [string range $state(i) $n [incr n 64]]
    }

    # Adjust the state for the blocks completed.
    set state(i) [string range $state(i) $n end]
    return
}

# MD5Final --
#
#    This procedure is used to close the current hash and returns the
#    hash data. Once this procedure has been called the hash context
#    is freed and cannot be used again.
#
#    Note that the output is 128 bits represented as binary data.
#
proc ::md5::MD5Final {token} {
    upvar #0 $token state

    # Check for either of the C-compiled versions.
    if {[info exists state(md5c)]} {
        set r $state(md5c)
        unset state
        return $r
    } elseif {[info exists state(ckctx)]} {
        cryptkit::cryptEncrypt $state(ckctx) ""
        cryptkit::cryptGetAttributeString $state(ckctx) \
            CRYPT_CTXINFO_HASHVALUE r 16
        cryptkit::cryptDestroyContext $state(ckctx)
        # If nothing was hashed, we get no r variable set!
        if {[info exists r]} {
            unset state
            return $r
        }
    } elseif {[info exists state(trf)]} {
        close $state(trf)
        set r $state(trfwrite)
        unset state
        return $r
    }

    # RFC1321:3.1 - Padding
    #
    set len [string length $state(i)]
    set pad [expr {56 - ($len % 64)}]
    if {$len % 64 > 56} {
        incr pad 64
    }
    if {$pad == 0} {
        incr pad 64
    }
    append state(i) [binary format a$pad \x80]

    # RFC1321:3.2 - Append length in bits as little-endian wide int.
    append state(i) [binary format ii [expr {8 * $state(n)}] 0]

    # Calculate the hash for the remaining block.
    set len [string length $state(i)]
    for {set n 0} {($n + 64) <= $len} {} {
        MD5Hash $token [string range $state(i) $n [incr n 64]]
    }

    # RFC1321:3.5 - Output
    set r [bytes $state(A)][bytes $state(B)][bytes $state(C)][bytes $state(D)]
    unset state
    return $r
}

# -------------------------------------------------------------------------
# HMAC Hashed Message Authentication (RFC 2104)
#
# hmac = H(K xor opad, H(K xor ipad, text))
#

# HMACInit --
#
#    This is equivalent to the MD5Init procedure except that a key is
#    added into the algorithm
#
proc ::md5::HMACInit {K} {

    # Key K is adjusted to be 64 bytes long. If K is larger, then use
    # the MD5 digest of K and pad this instead.
    set len [string length $K]
    if {$len > 64} {
        set tok [MD5Init]
        MD5Update $tok $K
        set K [MD5Final $tok]
        set len [string length $K]
    }
    set pad [expr {64 - $len}]
    append K [string repeat \0 $pad]

    # Cacluate the padding buffers.
    set Ki {}
    set Ko {}
    binary scan $K i16 Ks
    foreach k $Ks {
        append Ki [binary format i [expr {$k ^ 0x36363636}]]
        append Ko [binary format i [expr {$k ^ 0x5c5c5c5c}]]
    }

    set tok [MD5Init]
    MD5Update $tok $Ki;                 # initialize with the inner pad
    
    # preserve the Ko value for the final stage.
    # FRINK: nocheck
    set [subst $tok](Ko) $Ko

    return $tok
}

# HMACUpdate --
#
#    Identical to calling MD5Update
#
proc ::md5::HMACUpdate {token data} {
    MD5Update $token $data
    return
}

# HMACFinal --
#
#    This is equivalent to the MD5Final procedure. The hash context is
#    closed and the binary representation of the hash result is returned.
#
proc ::md5::HMACFinal {token} {
    upvar #0 $token state

    set tok [MD5Init];                  # init the outer hashing function
    MD5Update $tok $state(Ko);          # prepare with the outer pad.
    MD5Update $tok [MD5Final $token];   # hash the inner result
    return [MD5Final $tok]
}

# -------------------------------------------------------------------------
# Description:
#  This is the core MD5 algorithm. It is a lot like the MD4 algorithm but
#  includes an extra round and a set of constant modifiers throughout.
# 
# Note:
#  This function body is substituted later on to inline some of the 
#  procedures and to make is a bit more comprehensible.
#
set ::md5::MD5Hash_body {
    variable $token
    upvar 0 $token state

    # RFC1321:3.4 - Process Message in 16-Word Blocks
    binary scan $msg i* blocks
    foreach {X0 X1 X2 X3 X4 X5 X6 X7 X8 X9 X10 X11 X12 X13 X14 X15} $blocks {
        set A $state(A)
        set B $state(B)
        set C $state(C)
        set D $state(D)

        # Round 1
        # Let [abcd k s i] denote the operation
        #   a = b + ((a + F(b,c,d) + X[k] + T[i]) <<< s).
        # Do the following 16 operations.
        # [ABCD  0  7  1]  [DABC  1 12  2]  [CDAB  2 17  3]  [BCDA  3 22  4]
        set A [expr {$B + (($A + [F $B $C $D] + $X0 + $T01) <<< 7)}]
        set D [expr {$A + (($D + [F $A $B $C] + $X1 + $T02) <<< 12)}]
        set C [expr {$D + (($C + [F $D $A $B] + $X2 + $T03) <<< 17)}]
        set B [expr {$C + (($B + [F $C $D $A] + $X3 + $T04) <<< 22)}]
        # [ABCD  4  7  5]  [DABC  5 12  6]  [CDAB  6 17  7]  [BCDA  7 22  8]
        set A [expr {$B + (($A + [F $B $C $D] + $X4 + $T05) <<< 7)}]
        set D [expr {$A + (($D + [F $A $B $C] + $X5 + $T06) <<< 12)}]
        set C [expr {$D + (($C + [F $D $A $B] + $X6 + $T07) <<< 17)}]
        set B [expr {$C + (($B + [F $C $D $A] + $X7 + $T08) <<< 22)}]
        # [ABCD  8  7  9]  [DABC  9 12 10]  [CDAB 10 17 11]  [BCDA 11 22 12]
        set A [expr {$B + (($A + [F $B $C $D] + $X8 + $T09) <<< 7)}]
        set D [expr {$A + (($D + [F $A $B $C] + $X9 + $T10) <<< 12)}]
        set C [expr {$D + (($C + [F $D $A $B] + $X10 + $T11) <<< 17)}]
        set B [expr {$C + (($B + [F $C $D $A] + $X11 + $T12) <<< 22)}]
        # [ABCD 12  7 13]  [DABC 13 12 14]  [CDAB 14 17 15]  [BCDA 15 22 16]
        set A [expr {$B + (($A + [F $B $C $D] + $X12 + $T13) <<< 7)}]
        set D [expr {$A + (($D + [F $A $B $C] + $X13 + $T14) <<< 12)}]
        set C [expr {$D + (($C + [F $D $A $B] + $X14 + $T15) <<< 17)}]
        set B [expr {$C + (($B + [F $C $D $A] + $X15 + $T16) <<< 22)}]

        # Round 2.
        # Let [abcd k s i] denote the operation
        #   a = b + ((a + G(b,c,d) + X[k] + Ti) <<< s)
        # Do the following 16 operations.
        # [ABCD  1  5 17]  [DABC  6  9 18]  [CDAB 11 14 19]  [BCDA  0 20 20]
        set A [expr {$B + (($A + [G $B $C $D] + $X1  + $T17) <<<  5)}]
        set D [expr {$A + (($D + [G $A $B $C] + $X6  + $T18) <<<  9)}]
        set C [expr {$D + (($C + [G $D $A $B] + $X11 + $T19) <<< 14)}]
        set B [expr {$C + (($B + [G $C $D $A] + $X0  + $T20) <<< 20)}]
        # [ABCD  5  5 21]  [DABC 10  9 22]  [CDAB 15 14 23]  [BCDA  4 20 24]
        set A [expr {$B + (($A + [G $B $C $D] + $X5  + $T21) <<<  5)}]
        set D [expr {$A + (($D + [G $A $B $C] + $X10 + $T22) <<<  9)}]
        set C [expr {$D + (($C + [G $D $A $B] + $X15 + $T23) <<< 14)}]
        set B [expr {$C + (($B + [G $C $D $A] + $X4  + $T24) <<< 20)}]
        # [ABCD  9  5 25]  [DABC 14  9 26]  [CDAB  3 14 27]  [BCDA  8 20 28]
        set A [expr {$B + (($A + [G $B $C $D] + $X9  + $T25) <<<  5)}]
        set D [expr {$A + (($D + [G $A $B $C] + $X14 + $T26) <<<  9)}]
        set C [expr {$D + (($C + [G $D $A $B] + $X3  + $T27) <<< 14)}]
        set B [expr {$C + (($B + [G $C $D $A] + $X8  + $T28) <<< 20)}]
        # [ABCD 13  5 29]  [DABC  2  9 30]  [CDAB  7 14 31]  [BCDA 12 20 32]
        set A [expr {$B + (($A + [G $B $C $D] + $X13 + $T29) <<<  5)}]
        set D [expr {$A + (($D + [G $A $B $C] + $X2  + $T30) <<<  9)}]
        set C [expr {$D + (($C + [G $D $A $B] + $X7  + $T31) <<< 14)}]
        set B [expr {$C + (($B + [G $C $D $A] + $X12 + $T32) <<< 20)}]
        
        # Round 3.
        # Let [abcd k s i] denote the operation
        #   a = b + ((a + H(b,c,d) + X[k] + T[i]) <<< s)
        # Do the following 16 operations.
        # [ABCD  5  4 33]  [DABC  8 11 34]  [CDAB 11 16 35]  [BCDA 14 23 36]
        set A [expr {$B + (($A + [H $B $C $D] + $X5  + $T33) <<<  4)}]
        set D [expr {$A + (($D + [H $A $B $C] + $X8  + $T34) <<< 11)}]
        set C [expr {$D + (($C + [H $D $A $B] + $X11 + $T35) <<< 16)}]
        set B [expr {$C + (($B + [H $C $D $A] + $X14 + $T36) <<< 23)}]
        # [ABCD  1  4 37]  [DABC  4 11 38]  [CDAB  7 16 39]  [BCDA 10 23 40]
        set A [expr {$B + (($A + [H $B $C $D] + $X1  + $T37) <<<  4)}]
        set D [expr {$A + (($D + [H $A $B $C] + $X4  + $T38) <<< 11)}]
        set C [expr {$D + (($C + [H $D $A $B] + $X7  + $T39) <<< 16)}]
        set B [expr {$C + (($B + [H $C $D $A] + $X10 + $T40) <<< 23)}]
        # [ABCD 13  4 41]  [DABC  0 11 42]  [CDAB  3 16 43]  [BCDA  6 23 44]
        set A [expr {$B + (($A + [H $B $C $D] + $X13 + $T41) <<<  4)}]
        set D [expr {$A + (($D + [H $A $B $C] + $X0  + $T42) <<< 11)}]
        set C [expr {$D + (($C + [H $D $A $B] + $X3  + $T43) <<< 16)}]
        set B [expr {$C + (($B + [H $C $D $A] + $X6  + $T44) <<< 23)}]
        # [ABCD  9  4 45]  [DABC 12 11 46]  [CDAB 15 16 47]  [BCDA  2 23 48]
        set A [expr {$B + (($A + [H $B $C $D] + $X9  + $T45) <<<  4)}]
        set D [expr {$A + (($D + [H $A $B $C] + $X12 + $T46) <<< 11)}]
        set C [expr {$D + (($C + [H $D $A $B] + $X15 + $T47) <<< 16)}]
        set B [expr {$C + (($B + [H $C $D $A] + $X2  + $T48) <<< 23)}]

        # Round 4.
        # Let [abcd k s i] denote the operation
        #   a = b + ((a + I(b,c,d) + X[k] + T[i]) <<< s)
        # Do the following 16 operations.
        # [ABCD  0  6 49]  [DABC  7 10 50]  [CDAB 14 15 51]  [BCDA  5 21 52]
        set A [expr {$B + (($A + [I $B $C $D] + $X0  + $T49) <<<  6)}]
        set D [expr {$A + (($D + [I $A $B $C] + $X7  + $T50) <<< 10)}]
        set C [expr {$D + (($C + [I $D $A $B] + $X14 + $T51) <<< 15)}]
        set B [expr {$C + (($B + [I $C $D $A] + $X5  + $T52) <<< 21)}]
        # [ABCD 12  6 53]  [DABC  3 10 54]  [CDAB 10 15 55]  [BCDA  1 21 56]
        set A [expr {$B + (($A + [I $B $C $D] + $X12 + $T53) <<<  6)}]
        set D [expr {$A + (($D + [I $A $B $C] + $X3  + $T54) <<< 10)}]
        set C [expr {$D + (($C + [I $D $A $B] + $X10 + $T55) <<< 15)}]
        set B [expr {$C + (($B + [I $C $D $A] + $X1  + $T56) <<< 21)}]
        # [ABCD  8  6 57]  [DABC 15 10 58]  [CDAB  6 15 59]  [BCDA 13 21 60]
        set A [expr {$B + (($A + [I $B $C $D] + $X8  + $T57) <<<  6)}]
        set D [expr {$A + (($D + [I $A $B $C] + $X15 + $T58) <<< 10)}]
        set C [expr {$D + (($C + [I $D $A $B] + $X6  + $T59) <<< 15)}]
        set B [expr {$C + (($B + [I $C $D $A] + $X13 + $T60) <<< 21)}]
        # [ABCD  4  6 61]  [DABC 11 10 62]  [CDAB  2 15 63]  [BCDA  9 21 64]
        set A [expr {$B + (($A + [I $B $C $D] + $X4  + $T61) <<<  6)}]
        set D [expr {$A + (($D + [I $A $B $C] + $X11 + $T62) <<< 10)}]
        set C [expr {$D + (($C + [I $D $A $B] + $X2  + $T63) <<< 15)}]
        set B [expr {$C + (($B + [I $C $D $A] + $X9  + $T64) <<< 21)}]

        # Then perform the following additions. (That is, increment each
        # of the four registers by the value it had before this block
        # was started.)
        incr state(A) $A
        incr state(B) $B
        incr state(C) $C
        incr state(D) $D
    }

    return
}

proc ::md5::byte {n v} {expr {((0xFF << (8 * $n)) & $v) >> (8 * $n)}}
proc ::md5::bytes {v} { 
    #format %c%c%c%c [byte 0 $v] [byte 1 $v] [byte 2 $v] [byte 3 $v]
    format %c%c%c%c \
        [expr {0xFF & $v}] \
        [expr {(0xFF00 & $v) >> 8}] \
        [expr {(0xFF0000 & $v) >> 16}] \
        [expr {((0xFF000000 & $v) >> 24) & 0xFF}]
}

# 32bit rotate-left
proc ::md5::<<< {v n} {
    return [expr {((($v << $n) \
                        | (($v >> (32 - $n)) \
                               & (0x7FFFFFFF >> (31 - $n))))) \
                      & 0xFFFFFFFF}]
}

# Convert our <<< pseudo-operator into a procedure call.
regsub -all -line \
    {\[expr {(\$[ABCD]) \+ \(\((.*)\)\s+<<<\s+(\d+)\)}\]} \
    $::md5::MD5Hash_body \
    {[expr {int(\1 + [<<< [expr {\2}] \3])}]} \
    ::md5::MD5Hash_body

# RFC1321:3.4 - function F
proc ::md5::F {X Y Z} {
    return [expr {($X & $Y) | ((~$X) & $Z)}]
}

# Inline the F function
regsub -all -line \
    {\[F (\$[ABCD]) (\$[ABCD]) (\$[ABCD])\]} \
    $::md5::MD5Hash_body \
    {( (\1 \& \2) | ((~\1) \& \3) )} \
    ::md5::MD5Hash_body
    
# RFC1321:3.4 - function G
proc ::md5::G {X Y Z} {
    return [expr {(($X & $Z) | ($Y & (~$Z)))}]
}

# Inline the G function
regsub -all -line \
    {\[G (\$[ABCD]) (\$[ABCD]) (\$[ABCD])\]} \
    $::md5::MD5Hash_body \
    {(((\1 \& \3) | (\2 \& (~\3))))} \
    ::md5::MD5Hash_body

# RFC1321:3.4 - function H
proc ::md5::H {X Y Z} {
    return [expr {$X ^ $Y ^ $Z}]
}

# Inline the H function
regsub -all -line \
    {\[H (\$[ABCD]) (\$[ABCD]) (\$[ABCD])\]} \
    $::md5::MD5Hash_body \
    {(\1 ^ \2 ^ \3)} \
    ::md5::MD5Hash_body

# RFC1321:3.4 - function I
proc ::md5::I {X Y Z} {
    return [expr {$Y ^ ($X | (~$Z))}]
}

# Inline the I function
regsub -all -line \
    {\[I (\$[ABCD]) (\$[ABCD]) (\$[ABCD])\]} \
    $::md5::MD5Hash_body \
    {(\2 ^ (\1 | (~\3)))} \
    ::md5::MD5Hash_body


# RFC 1321:3.4 step 4: inline the set of constant modifiers.
namespace eval md5 {
    foreach tName {
        T01 T02 T03 T04 T05 T06 T07 T08 T09 T10 
        T11 T12 T13 T14 T15 T16 T17 T18 T19 T20 
        T21 T22 T23 T24 T25 T26 T27 T28 T29 T30 
        T31 T32 T33 T34 T35 T36 T37 T38 T39 T40 
        T41 T42 T43 T44 T45 T46 T47 T48 T49 T50 
        T51 T52 T53 T54 T55 T56 T57 T58 T59 T60 
        T61 T62 T63 T64 
    }  tVal {
        0xd76aa478 0xe8c7b756 0x242070db 0xc1bdceee
        0xf57c0faf 0x4787c62a 0xa8304613 0xfd469501
        0x698098d8 0x8b44f7af 0xffff5bb1 0x895cd7be
        0x6b901122 0xfd987193 0xa679438e 0x49b40821
        
        0xf61e2562 0xc040b340 0x265e5a51 0xe9b6c7aa
        0xd62f105d 0x2441453  0xd8a1e681 0xe7d3fbc8
        0x21e1cde6 0xc33707d6 0xf4d50d87 0x455a14ed
        0xa9e3e905 0xfcefa3f8 0x676f02d9 0x8d2a4c8a
        
        0xfffa3942 0x8771f681 0x6d9d6122 0xfde5380c
        0xa4beea44 0x4bdecfa9 0xf6bb4b60 0xbebfbc70
        0x289b7ec6 0xeaa127fa 0xd4ef3085 0x4881d05
        0xd9d4d039 0xe6db99e5 0x1fa27cf8 0xc4ac5665
        
        0xf4292244 0x432aff97 0xab9423a7 0xfc93a039
        0x655b59c3 0x8f0ccc92 0xffeff47d 0x85845dd1
        0x6fa87e4f 0xfe2ce6e0 0xa3014314 0x4e0811a1
        0xf7537e82 0xbd3af235 0x2ad7d2bb 0xeb86d391
    } {
        lappend map \$$tName $tVal
    }
    set ::md5::MD5Hash_body [string map $map $::md5::MD5Hash_body]
    unset map tName tVal
}

# Define the MD5 hashing procedure with inline functions.
proc ::md5::MD5Hash {token msg} $::md5::MD5Hash_body
unset ::md5::MD5Hash_body

# -------------------------------------------------------------------------

if {[package provide Trf] != {}} {
    interp alias {} ::md5::Hex {} ::hex -mode encode --
} else {
    proc ::md5::Hex {data} {
        binary scan $data H* result
        return [string toupper $result]
    }
}

# -------------------------------------------------------------------------

# LoadAccelerator --
#
#	This package can make use of a number of compiled extensions to
#	accelerate the digest computation. This procedure manages the
#	use of these extensions within the package. During normal usage
#	this should not be called, but the test package manipulates the
#	list of enabled accelerators.
#
proc ::md5::LoadAccelerator {name} {
    variable accel
    set r 0
    switch -exact -- $name {
        critcl {
            if {![catch {package require tcllibc}]
                || ![catch {package require md5c}]} {
                set r [expr {[info command ::md5::md5c] != {}}]
            }
        }
        cryptkit {
            if {![catch {package require cryptkit}]} {
                set r [expr {![catch {cryptkit::cryptInit}]}]
            }
        }
        trf {
            if {![catch {package require Trf}]} {
                set r [expr {![catch {::md5 aa} msg]}]
            }
        }
        default {
            return -code error "invalid accelerator package:\
                must be one of [join [array names accel] {, }]"
        }
    }
    set accel($name) $r
}

# -------------------------------------------------------------------------

# Description:
#  Pop the nth element off a list. Used in options processing.
#
proc ::md5::Pop {varname {nth 0}} {
    upvar $varname args
    set r [lindex $args $nth]
    set args [lreplace $args $nth $nth]
    return $r
}

# -------------------------------------------------------------------------

# fileevent handler for chunked file hashing.
#
proc ::md5::Chunk {token channel {chunksize 4096}} {
    upvar #0 $token state
    
    if {[eof $channel]} {
        fileevent $channel readable {}
        set state(reading) 0
    }
        
    MD5Update $token [read $channel $chunksize]
}

# -------------------------------------------------------------------------

proc ::md5::md5 {args} {
    array set opts {-hex 0 -filename {} -channel {} -chunksize 4096}
    while {[string match -* [set option [lindex $args 0]]]} {
        switch -glob -- $option {
            -hex       { set opts(-hex) 1 }
            -file*     { set opts(-filename) [Pop args 1] }
            -channel   { set opts(-channel) [Pop args 1] }
            -chunksize { set opts(-chunksize) [Pop args 1] }
            default {
                if {[llength $args] == 1} { break }
                if {[string compare $option "--"] == 0} { Pop args; break }
                set err [join [lsort [array names opts]] ", "]
                return -code error "bad option $option:\
                    must be one of $err\nlen: [llength $args]"
            }
        }
        Pop args
    }

    if {$opts(-filename) != {}} {
        set opts(-channel) [open $opts(-filename) r]
        fconfigure $opts(-channel) -translation binary
    }

    if {$opts(-channel) == {}} {

        if {[llength $args] != 1} {
            return -code error "wrong # args:\
                should be \"md5 ?-hex? -filename file | string\""
        }
        set tok [MD5Init]
        MD5Update $tok [lindex $args 0]
        set r [MD5Final $tok]

    } else {

        set tok [MD5Init]
        # FRINK: nocheck
        set [subst $tok](reading) 1
        fileevent $opts(-channel) readable \
            [list [namespace origin Chunk] \
                 $tok $opts(-channel) $opts(-chunksize)]
        vwait [subst $tok](reading)
        set r [MD5Final $tok]

        # If we opened the channel - we should close it too.
        if {$opts(-filename) != {}} {
            close $opts(-channel)
        }
    }
    
    if {$opts(-hex)} {
        set r [Hex $r]
    }
    return $r
}

# -------------------------------------------------------------------------

proc ::md5::hmac {args} {
    array set opts {-hex 0 -filename {} -channel {} -chunksize 4096}
    while {[string match -* [set option [lindex $args 0]]]} {
        switch -glob -- $option {
            -key       { set opts(-key) [Pop args 1] }
            -hex       { set opts(-hex) 1 }
            -file*     { set opts(-filename) [Pop args 1] }
            -channel   { set opts(-channel) [Pop args 1] }
            -chunksize { set opts(-chunksize) [Pop args 1] }
            default {
                if {[llength $args] == 1} { break }
                if {[string compare $option "--"] == 0} { Pop args; break }
                set err [join [lsort [array names opts]] ", "]
                return -code error "bad option $option:\
                    must be one of $err"
            }
        }
        Pop args
    }

    if {![info exists opts(-key)]} {
        return -code error "wrong # args:\
            should be \"hmac ?-hex? -key key -filename file | string\""
    }

    if {$opts(-filename) != {}} {
        set opts(-channel) [open $opts(-filename) r]
        fconfigure $opts(-channel) -translation binary
    }

    if {$opts(-channel) == {}} {

        if {[llength $args] != 1} {
            return -code error "wrong # args:\
                should be \"hmac ?-hex? -key key -filename file | string\""
        }
        set tok [HMACInit $opts(-key)]
        HMACUpdate $tok [lindex $args 0]
        set r [HMACFinal $tok]

    } else {

        set tok [HMACInit $opts(-key)]
        # FRINK: nocheck
        set [subst $tok](reading) 1
        fileevent $opts(-channel) readable \
            [list [namespace origin Chunk] \
                 $tok $opts(-channel) $opts(-chunksize)]
        vwait [subst $tok](reading)
        set r [HMACFinal $tok]

        # If we opened the channel - we should close it too.
        if {$opts(-filename) != {}} {
            close $opts(-channel)
        }
    }
    
    if {$opts(-hex)} {
        set r [Hex $r]
    }
    return $r
}

# -------------------------------------------------------------------------

# Try and load a compiled extension to help.
namespace eval ::md5 {
    variable e
    foreach  e {critcl cryptkit trf} { if {[LoadAccelerator $e]} { break } }
    unset    e
}

package provide md5 2.0.7

# -------------------------------------------------------------------------
# Local Variables:
#   mode: tcl
#   indent-tabs-mode: nil
# End:


# sha256.tcl - Copyright (C) 2005 Pat Thoyts <patthoyts@users.sourceforge.net>
#
# SHA1 defined by FIPS 180-2, "The Secure Hash Standard"
# HMAC defined by RFC 2104, "Keyed-Hashing for Message Authentication"
#
# This is an implementation of the secure hash algorithms specified in the
# FIPS 180-2 document.
#
# This implementation permits incremental updating of the hash and 
# provides support for external compiled implementations using critcl.
#
# This implementation permits incremental updating of the hash and 
# provides support for external compiled implementations either using
# critcl (sha256c).
#
# Ref: http://csrc.nist.gov/publications/fips/fips180-2/fips180-2.pdf
#      http://csrc.nist.gov/publications/fips/fips180-2/fips180-2withchangenotice.pdf
#
# -------------------------------------------------------------------------
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# -------------------------------------------------------------------------
# @mdgen EXCLUDE: sha256c.tcl

package require Tcl 8.2;                # tcl minimum version

namespace eval ::sha2 {
    variable  accel
    array set accel {tcl 0 critcl 0}
    variable  loaded {}

    namespace export sha256 hmac \
            SHA256Init SHA256Update SHA256Final

    variable uid
    if {![info exists uid]} {
        set uid 0
    }

    variable K
    if {![info exists K]} {
        # FIPS 180-2: 4.2.2 SHA-256 constants
        set K [list \
                   0x428a2f98 0x71374491 0xb5c0fbcf 0xe9b5dba5 \
                   0x3956c25b 0x59f111f1 0x923f82a4 0xab1c5ed5 \
                   0xd807aa98 0x12835b01 0x243185be 0x550c7dc3 \
                   0x72be5d74 0x80deb1fe 0x9bdc06a7 0xc19bf174 \
                   0xe49b69c1 0xefbe4786 0x0fc19dc6 0x240ca1cc \
                   0x2de92c6f 0x4a7484aa 0x5cb0a9dc 0x76f988da \
                   0x983e5152 0xa831c66d 0xb00327c8 0xbf597fc7 \
                   0xc6e00bf3 0xd5a79147 0x06ca6351 0x14292967 \
                   0x27b70a85 0x2e1b2138 0x4d2c6dfc 0x53380d13 \
                   0x650a7354 0x766a0abb 0x81c2c92e 0x92722c85 \
                   0xa2bfe8a1 0xa81a664b 0xc24b8b70 0xc76c51a3 \
                   0xd192e819 0xd6990624 0xf40e3585 0x106aa070 \
                   0x19a4c116 0x1e376c08 0x2748774c 0x34b0bcb5 \
                   0x391c0cb3 0x4ed8aa4a 0x5b9cca4f 0x682e6ff3 \
                   0x748f82ee 0x78a5636f 0x84c87814 0x8cc70208 \
                   0x90befffa 0xa4506ceb 0xbef9a3f7 0xc67178f2 \
                  ]
    }
    
}

# -------------------------------------------------------------------------
# Management of sha256 implementations.

# LoadAccelerator --
#
#	This package can make use of a number of compiled extensions to
#	accelerate the digest computation. This procedure manages the
#	use of these extensions within the package. During normal usage
#	this should not be called, but the test package manipulates the
#	list of enabled accelerators.
#
proc ::sha2::LoadAccelerator {name} {
    variable accel
    set r 0
    switch -exact -- $name {
        tcl {
            # Already present (this file)
            set r 1
        }
        critcl {
            if {![catch {package require tcllibc}]
                || ![catch {package require sha256c}]} {
                set r [expr {[info command ::sha2::sha256c_update] != {}}]
            }
        }
        default {
            return -code error "invalid accelerator $key:\
                must be one of [join [KnownImplementations] {, }]"
        }
    }
    set accel($name) $r
    return $r
}

# ::sha2::Implementations --
#
#	Determines which implementations are
#	present, i.e. loaded.
#
# Arguments:
#	None.
#
# Results:
#	A list of implementation keys.

proc ::sha2::Implementations {} {
    variable accel
    set res {}
    foreach n [array names accel] {
	if {!$accel($n)} continue
	lappend res $n
    }
    return $res
}

# ::sha2::KnownImplementations --
#
#	Determines which implementations are known
#	as possible implementations.
#
# Arguments:
#	None.
#
# Results:
#	A list of implementation keys. In the order
#	of preference, most prefered first.

proc ::sha2::KnownImplementations {} {
    return {critcl tcl}
}

proc ::sha2::Names {} {
    return {
	critcl   {tcllibc based}
	tcl      {pure Tcl}
    }
}

# ::sha2::SwitchTo --
#
#	Activates a loaded named implementation.
#
# Arguments:
#	key	Name of the implementation to activate.
#
# Results:
#	None.

proc ::sha2::SwitchTo {key} {
    variable accel
    variable loaded

    if {[string equal $key $loaded]} {
	# No change, nothing to do.
	return
    } elseif {![string equal $key ""]} {
	# Validate the target implementation of the switch.

	if {![info exists accel($key)]} {
	    return -code error "Unable to activate unknown implementation \"$key\""
	} elseif {![info exists accel($key)] || !$accel($key)} {
	    return -code error "Unable to activate missing implementation \"$key\""
	}
    }

    # Deactivate the previous implementation, if there was any.

    if {![string equal $loaded ""]} {
        foreach c {
            SHA256Init   SHA224Init
            SHA256Final  SHA224Final
            SHA256Update
        } {
            rename ::sha2::$c ::sha2::${c}-${loaded}
        }
    }

    # Activate the new implementation, if there is any.

    if {![string equal $key ""]} {
        foreach c {
            SHA256Init   SHA224Init
            SHA256Final  SHA224Final
            SHA256Update
        } {
            rename ::sha2::${c}-${key} ::sha2::$c
        }
    }

    # Remember the active implementation, for deactivation by future
    # switches.

    set loaded $key
    return
}

# -------------------------------------------------------------------------

# SHA256Init --
#
#   Create and initialize an SHA256 state variable. This will be
#   cleaned up when we call SHA256Final
#

proc ::sha2::SHA256Init-tcl {} {
    variable uid
    set token [namespace current]::[incr uid]
    upvar #0 $token tok

    # FIPS 180-2: 5.3.2 Setting the initial hash value
    array set tok \
            [list \
            A [expr {int(0x6a09e667)}] \
            B [expr {int(0xbb67ae85)}] \
            C [expr {int(0x3c6ef372)}] \
            D [expr {int(0xa54ff53a)}] \
            E [expr {int(0x510e527f)}] \
            F [expr {int(0x9b05688c)}] \
            G [expr {int(0x1f83d9ab)}] \
            H [expr {int(0x5be0cd19)}] \
            n 0 i "" v 256]
    return $token
}

proc ::sha2::SHA256Init-critcl {} {
    variable uid
    set token [namespace current]::[incr uid]
    upvar #0 $token tok

    # FIPS 180-2: 5.3.2 Setting the initial hash value
    set tok(sha256c) [sha256c_init256]
    return $token
}

# SHA256Update --
#
#   This is called to add more data into the hash. You may call this
#   as many times as you require. Note that passing in "ABC" is equivalent
#   to passing these letters in as separate calls -- hence this proc 
#   permits hashing of chunked data
#
#   If we have a C-based implementation available, then we will use
#   it here in preference to the pure-Tcl implementation.
#

proc ::sha2::SHA256Update-tcl {token data} {
    upvar #0 $token state

    # Update the state values
    incr   state(n) [string length $data]
    append state(i) $data

    # Calculate the hash for any complete blocks
    set len [string length $state(i)]
    for {set n 0} {($n + 64) <= $len} {} {
        SHA256Transform $token [string range $state(i) $n [incr n 64]]
    }

    # Adjust the state for the blocks completed.
    set state(i) [string range $state(i) $n end]
    return
}

proc ::sha2::SHA256Update-critcl {token data} {
    upvar #0 $token state

    set state(sha256c) [sha256c_update $data $state(sha256c)]
    return
}

# SHA256Final --
#
#    This procedure is used to close the current hash and returns the
#    hash data. Once this procedure has been called the hash context
#    is freed and cannot be used again.
#
#    Note that the output is 256 bits represented as binary data.
#

proc ::sha2::SHA256Final-tcl {token} {
    upvar #0 $token state
    SHA256Penultimate $token
    
    # Output
    set r [bytes $state(A)][bytes $state(B)][bytes $state(C)][bytes $state(D)][bytes $state(E)][bytes $state(F)][bytes $state(G)][bytes $state(H)]
    unset state
    return $r
}

proc ::sha2::SHA256Final-critcl {token} {
    upvar #0 $token state
    set r $state(sha256c)
    unset  state
    return $r
}

# SHA256Penultimate --
#
#
proc ::sha2::SHA256Penultimate {token} {
    upvar #0 $token state

    # FIPS 180-2: 5.1.1: Padding the message
    #
    set len [string length $state(i)]
    set pad [expr {56 - ($len % 64)}]
    if {$len % 64 > 56} {
        incr pad 64
    }
    if {$pad == 0} {
        incr pad 64
    }
    append state(i) [binary format a$pad \x80]

    # Append length in bits as big-endian wide int.
    set dlen [expr {8 * $state(n)}]
    append state(i) [binary format II 0 $dlen]

    # Calculate the hash for the remaining block.
    set len [string length $state(i)]
    for {set n 0} {($n + 64) <= $len} {} {
        SHA256Transform $token [string range $state(i) $n [incr n 64]]
    }
}

# -------------------------------------------------------------------------

proc ::sha2::SHA224Init-tcl {} {
    variable uid
    set token [namespace current]::[incr uid]
    upvar #0 $token tok

    # FIPS 180-2 (change notice 1) (1): SHA-224 initialization values
    array set tok \
            [list \
            A [expr {int(0xc1059ed8)}] \
            B [expr {int(0x367cd507)}] \
            C [expr {int(0x3070dd17)}] \
            D [expr {int(0xf70e5939)}] \
            E [expr {int(0xffc00b31)}] \
            F [expr {int(0x68581511)}] \
            G [expr {int(0x64f98fa7)}] \
            H [expr {int(0xbefa4fa4)}] \
            n 0 i "" v 224]
    return $token
}

proc ::sha2::SHA224Init-critcl {} {
    variable uid
    set token [namespace current]::[incr uid]
    upvar #0 $token tok

    # FIPS 180-2 (change notice 1) (1): SHA-224 initialization values
    set tok(sha256c) [sha256c_init224]
    return $token
}

interp alias {} ::sha2::SHA224Update {} ::sha2::SHA256Update

proc ::sha2::SHA224Final-tcl {token} {
    upvar #0 $token state
    SHA256Penultimate $token
    
    # Output
    set r [bytes $state(A)][bytes $state(B)][bytes $state(C)][bytes $state(D)][bytes $state(E)][bytes $state(F)][bytes $state(G)]
    unset state
    return $r
}

proc ::sha2::SHA224Final-critcl {token} {
    upvar #0 $token state
    # Trim result down to 224 bits (by 4 bytes).
    # See output below, A..G, not A..H
    set r [string range $state(sha256c) 0 end-4]
    unset state
    return $r
}

# -------------------------------------------------------------------------
# HMAC Hashed Message Authentication (RFC 2104)
#
# hmac = H(K xor opad, H(K xor ipad, text))
#

# HMACInit --
#
#    This is equivalent to the SHA1Init procedure except that a key is
#    added into the algorithm
#
proc ::sha2::HMACInit {K} {

    # Key K is adjusted to be 64 bytes long. If K is larger, then use
    # the SHA1 digest of K and pad this instead.
    set len [string length $K]
    if {$len > 64} {
        set tok [SHA256Init]
        SHA256Update $tok $K
        set K [SHA256Final $tok]
        set len [string length $K]
    }
    set pad [expr {64 - $len}]
    append K [string repeat \0 $pad]

    # Cacluate the padding buffers.
    set Ki {}
    set Ko {}
    binary scan $K i16 Ks
    foreach k $Ks {
        append Ki [binary format i [expr {$k ^ 0x36363636}]]
        append Ko [binary format i [expr {$k ^ 0x5c5c5c5c}]]
    }

    set tok [SHA256Init]
    SHA256Update $tok $Ki;                 # initialize with the inner pad
    
    # preserve the Ko value for the final stage.
    # FRINK: nocheck
    set [subst $tok](Ko) $Ko

    return $tok
}

# HMACUpdate --
#
#    Identical to calling SHA256Update
#
proc ::sha2::HMACUpdate {token data} {
    SHA256Update $token $data
    return
}

# HMACFinal --
#
#    This is equivalent to the SHA256Final procedure. The hash context is
#    closed and the binary representation of the hash result is returned.
#
proc ::sha2::HMACFinal {token} {
    upvar #0 $token state

    set tok [SHA256Init];                 # init the outer hashing function
    SHA256Update $tok $state(Ko);         # prepare with the outer pad.
    SHA256Update $tok [SHA256Final $token]; # hash the inner result
    return [SHA256Final $tok]
}

# -------------------------------------------------------------------------
# Description:
#  This is the core SHA1 algorithm. It is a lot like the MD4 algorithm but
#  includes an extra round and a set of constant modifiers throughout.
#
set ::sha2::SHA256Transform_body {
    variable K
    upvar #0 $token state

    # FIPS 180-2: 6.2.2 SHA-256 Hash computation.
    binary scan $msg I* blocks
    set blockLen [llength $blocks]
    for {set i 0} {$i < $blockLen} {incr i 16} {
        set W [lrange $blocks $i [expr {$i+15}]]

        # FIPS 180-2: 6.2.2 (1) Prepare the message schedule
        # For t = 16 to 64 
        #   let Wt = (sigma1(Wt-2) + Wt-7 + sigma0(Wt-15) + Wt-16)
        set t2  13
        set t7   8
        set t15  0
        set t16 -1
        for {set t 16} {$t < 64} {incr t} {
            lappend W [expr {([sigma1 [lindex $W [incr t2]]] \
                                 + [lindex $W [incr t7]] \
                                 + [sigma0 [lindex $W [incr t15]]] \
                                 + [lindex $W [incr t16]]) & 0xffffffff}]
        }
        
        # FIPS 180-2: 6.2.2 (2) Initialise the working variables
        set A $state(A)
        set B $state(B)
        set C $state(C)
        set D $state(D)
        set E $state(E)
        set F $state(F)
        set G $state(G)
        set H $state(H)

        # FIPS 180-2: 6.2.2 (3) Do permutation rounds
        # For t = 0 to 63 do
        #   T1 = h + SIGMA1(e) + Ch(e,f,g) + Kt + Wt
        #   T2 = SIGMA0(a) + Maj(a,b,c)
        #   h = g; g = f;  f = e;  e = d + T1;  d = c;  c = b; b = a;
        #   a = T1 + T2
        #
        for {set t 0} {$t < 64} {incr t} {
            set T1 [expr {($H + [SIGMA1 $E] + [Ch $E $F $G] 
                          + [lindex $K $t] + [lindex $W $t]) & 0xffffffff}]
            set T2 [expr {([SIGMA0 $A] + [Maj $A $B $C]) & 0xffffffff}]
            set H $G
            set G $F
            set F $E
            set E [expr {($D + $T1) & 0xffffffff}]
            set D $C
            set C $B
            set B $A
            set A [expr {($T1 + $T2) & 0xffffffff}]
        }

        # FIPS 180-2: 6.2.2 (4) Compute the intermediate hash
        incr state(A) $A
        incr state(B) $B
        incr state(C) $C
        incr state(D) $D
        incr state(E) $E
        incr state(F) $F
        incr state(G) $G
        incr state(H) $H
    }

    return
}

# -------------------------------------------------------------------------

# FIPS 180-2: 4.1.2 equation 4.2
proc ::sha2::Ch {x y z} {
    return [expr {($x & $y) ^ (~$x & $z)}]
}

# FIPS 180-2: 4.1.2 equation 4.3
proc ::sha2::Maj {x y z} {
    return [expr {($x & $y) ^ ($x & $z) ^ ($y & $z)}]
}

# FIPS 180-2: 4.1.2 equation 4.4
#  (x >>> 2) ^ (x >>> 13) ^ (x >>> 22)
proc ::sha2::SIGMA0 {x} {
    return [expr {[>>> $x 2] ^ [>>> $x 13] ^ [>>> $x 22]}]
}

# FIPS 180-2: 4.1.2 equation 4.5
#  (x >>> 6) ^ (x >>> 11) ^ (x >>> 25)
proc ::sha2::SIGMA1 {x} {
    return [expr {[>>> $x 6] ^ [>>> $x 11] ^ [>>> $x 25]}]
}

# FIPS 180-2: 4.1.2 equation 4.6
#  s0 = (x >>> 7)  ^ (x >>> 18) ^ (x >> 3)
proc ::sha2::sigma0 {x} {
    #return [expr {[>>> $x 7] ^ [>>> $x 18] ^ (($x >> 3) & 0x1fffffff)}]
    return [expr {((($x<<25) | (($x>>7) & (0x7FFFFFFF>>6))) \
                 ^ (($x<<14) | (($x>>18) & (0x7FFFFFFF>>17))) & 0xFFFFFFFF) \
                 ^ (($x>>3) & 0x1fffffff)}]
}

# FIPS 180-2: 4.1.2 equation 4.7
#  s1 = (x >>> 17) ^ (x >>> 19) ^ (x >> 10)
proc ::sha2::sigma1 {x} {
    #return [expr {[>>> $x 17] ^ [>>> $x 19] ^ (($x >> 10) & 0x003fffff)}]
    return [expr {((($x<<15) | (($x>>17) & (0x7FFFFFFF>>16))) \
                 ^ (($x<<13) | (($x>>19) & (0x7FFFFFFF>>18))) & 0xFFFFFFFF) \
                 ^ (($x >> 10) & 0x003fffff)}]
}

# 32bit rotate-right
proc ::sha2::>>> {v n} {
    return [expr {(($v << (32 - $n)) \
                       | (($v >> $n) & (0x7FFFFFFF >> ($n - 1)))) \
                      & 0xFFFFFFFF}]
}

# 32bit rotate-left
proc ::sha2::<<< {v n} {
    return [expr {((($v << $n) \
                        | (($v >> (32 - $n)) \
                               & (0x7FFFFFFF >> (31 - $n))))) \
                      & 0xFFFFFFFF}]
}

# -------------------------------------------------------------------------
# We speed up the SHA256Transform code while maintaining readability in the
# source code by substituting inline for a number of functions.
# The idea is to reduce the number of [expr] calls.

# Inline the Ch function
regsub -all -line \
    {\[Ch (\$[ABCDEFGH]) (\$[ABCDEFGH]) (\$[ABCDEFGH])\]} \
    $::sha2::SHA256Transform_body \
    {((\1 \& \2) ^ ((~\1) \& \3))} \
    ::sha2::SHA256Transform_body

# Inline the Maj function
regsub -all -line \
    {\[Maj (\$[ABCDEFGH]) (\$[ABCDEFGH]) (\$[ABCDEFGH])\]} \
    $::sha2::SHA256Transform_body \
    {((\1 \& \2) ^ (\1 \& \3) ^ (\2 \& \3))} \
    ::sha2::SHA256Transform_body


# Inline the SIGMA0 function
regsub -all -line \
    {\[SIGMA0 (\$[ABCDEFGH])\]} \
    $::sha2::SHA256Transform_body \
    {((((\1<<30) | ((\1>>2) \& (0x7FFFFFFF>>1))) \& 0xFFFFFFFF) \
          ^ (((\1<<19) | ((\1>>13) \& (0x7FFFFFFF>>12))) \& 0xFFFFFFFF) \
          ^ (((\1<<10) | ((\1>>22) \& (0x7FFFFFFF>>21))) \& 0xFFFFFFFF) \
          )} \
    ::sha2::SHA256Transform_body

# Inline the SIGMA1 function
regsub -all -line \
    {\[SIGMA1 (\$[ABCDEFGH])\]} \
    $::sha2::SHA256Transform_body \
    {((((\1<<26) | ((\1>>6) \& (0x7FFFFFFF>>5))) \& 0xFFFFFFFF) \
          ^ (((\1<<21) | ((\1>>11) \& (0x7FFFFFFF>>10))) \& 0xFFFFFFFF) \
          ^ (((\1<<7) | ((\1>>25) \& (0x7FFFFFFF>>24))) \& 0xFFFFFFFF) \
          )} \
    ::sha2::SHA256Transform_body

proc ::sha2::SHA256Transform {token msg} $::sha2::SHA256Transform_body

# -------------------------------------------------------------------------

# Convert a integer value into a binary string in big-endian order.
proc ::sha2::byte {n v} {expr {((0xFF << (8 * $n)) & $v) >> (8 * $n)}}
proc ::sha2::bytes {v} { 
    #format %c%c%c%c [byte 3 $v] [byte 2 $v] [byte 1 $v] [byte 0 $v]
    format %c%c%c%c \
        [expr {((0xFF000000 & $v) >> 24) & 0xFF}] \
        [expr {(0xFF0000 & $v) >> 16}] \
        [expr {(0xFF00 & $v) >> 8}] \
        [expr {0xFF & $v}]
}

# -------------------------------------------------------------------------

proc ::sha2::Hex {data} {
    binary scan $data H* result
    return $result
}

# -------------------------------------------------------------------------

# Description:
#  Pop the nth element off a list. Used in options processing.
#
proc ::sha2::Pop {varname {nth 0}} {
    upvar $varname args
    set r [lindex $args $nth]
    set args [lreplace $args $nth $nth]
    return $r
}

# -------------------------------------------------------------------------

# fileevent handler for chunked file hashing.
#
proc ::sha2::Chunk {token channel {chunksize 4096}} {
    upvar #0 $token state
    
    if {[eof $channel]} {
        fileevent $channel readable {}
        set state(reading) 0
    }
        
    SHA256Update $token [read $channel $chunksize]
}

# -------------------------------------------------------------------------

proc ::sha2::_sha256 {ver args} {
    array set opts {-hex 0 -filename {} -channel {} -chunksize 4096}
    if {[llength $args] == 1} {
        set opts(-hex) 1
    } else {
        while {[string match -* [set option [lindex $args 0]]]} {
            switch -glob -- $option {
                -hex       { set opts(-hex) 1 }
                -bin       { set opts(-hex) 0 }
                -file*     { set opts(-filename) [Pop args 1] }
                -channel   { set opts(-channel) [Pop args 1] }
                -chunksize { set opts(-chunksize) [Pop args 1] }
                default {
                    if {[llength $args] == 1} { break }
                    if {[string compare $option "--"] == 0} { Pop args; break }
                    set err [join [lsort [concat -bin [array names opts]]] ", "]
                    return -code error "bad option $option:\
                    must be one of $err"
                }
            }
            Pop args
        }
    }

    if {$opts(-filename) != {}} {
        set opts(-channel) [open $opts(-filename) r]
        fconfigure $opts(-channel) -translation binary
    }

    if {$opts(-channel) == {}} {

        if {[llength $args] != 1} {
            return -code error "wrong # args: should be\
                \"[namespace current]::sha$ver ?-hex|-bin? -filename file\
                | -channel channel | string\""
        }
        set tok [SHA${ver}Init]
        SHA${ver}Update $tok [lindex $args 0]
        set r [SHA${ver}Final $tok]

    } else {

        set tok [SHA${ver}Init]
        # FRINK: nocheck
        set [subst $tok](reading) 1
        fileevent $opts(-channel) readable \
            [list [namespace origin Chunk] \
                 $tok $opts(-channel) $opts(-chunksize)]
        # FRINK: nocheck
        vwait [subst $tok](reading)
        set r [SHA${ver}Final $tok]

        # If we opened the channel - we should close it too.
        if {$opts(-filename) != {}} {
            close $opts(-channel)
        }
    }
    
    if {$opts(-hex)} {
        set r [Hex $r]
    }
    return $r
}

interp alias {} ::sha2::sha256 {} ::sha2::_sha256 256
interp alias {} ::sha2::sha224 {} ::sha2::_sha256 224

# -------------------------------------------------------------------------

proc ::sha2::hmac {args} {
    array set opts {-hex 1 -filename {} -channel {} -chunksize 4096}
    if {[llength $args] != 2} {
        while {[string match -* [set option [lindex $args 0]]]} {
            switch -glob -- $option {
                -key       { set opts(-key) [Pop args 1] }
                -hex       { set opts(-hex) 1 }
                -bin       { set opts(-hex) 0 }
                -file*     { set opts(-filename) [Pop args 1] }
                -channel   { set opts(-channel) [Pop args 1] }
                -chunksize { set opts(-chunksize) [Pop args 1] }
                default {
                    if {[llength $args] == 1} { break }
                    if {[string compare $option "--"] == 0} { Pop args; break }
                    set err [join [lsort [array names opts]] ", "]
                    return -code error "bad option $option:\
                    must be one of $err"
                }
            }
            Pop args
        }
    }

    if {[llength $args] == 2} {
        set opts(-key) [Pop args]
    }

    if {![info exists opts(-key)]} {
        return -code error "wrong # args:\
            should be \"hmac ?-hex? -key key -filename file | string\""
    }

    if {$opts(-filename) != {}} {
        set opts(-channel) [open $opts(-filename) r]
        fconfigure $opts(-channel) -translation binary
    }

    if {$opts(-channel) == {}} {

        if {[llength $args] != 1} {
            return -code error "wrong # args:\
                should be \"hmac ?-hex? -key key -filename file | string\""
        }
        set tok [HMACInit $opts(-key)]
        HMACUpdate $tok [lindex $args 0]
        set r [HMACFinal $tok]

    } else {

        set tok [HMACInit $opts(-key)]
        # FRINK: nocheck
        set [subst $tok](reading) 1
        fileevent $opts(-channel) readable \
            [list [namespace origin Chunk] \
                 $tok $opts(-channel) $opts(-chunksize)]
        # FRINK: nocheck
        vwait [subst $tok](reading)
        set r [HMACFinal $tok]

        # If we opened the channel - we should close it too.
        if {$opts(-filename) != {}} {
            close $opts(-channel)
        }
    }
    
    if {$opts(-hex)} {
        set r [Hex $r]
    }
    return $r
}

# -------------------------------------------------------------------------

# Try and load a compiled extension to help.
namespace eval ::sha2 {
    variable e {}
    foreach e [KnownImplementations] {
	if {[LoadAccelerator $e]} {
	    SwitchTo $e
	    break
	}
    }
    unset e
}

package provide sha256 1.0.3

# -------------------------------------------------------------------------
# Local Variables:
#   mode: tcl
#   indent-tabs-mode: nil
# End:
# base64.tcl --
#
# Encode/Decode base64 for a string
# Stephen Uhler / Brent Welch (c) 1997 Sun Microsystems
# The decoder was done for exmh by Chris Garrigues
#
# Copyright (c) 1998-2000 by Ajuba Solutions.
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# 
# RCS: @(#) $Id: base64.tcl,v 1.32 2010/07/06 19:15:40 andreas_kupries Exp $

# Version 1.0   implemented Base64_Encode, Base64_Decode
# Version 2.0   uses the base64 namespace
# Version 2.1   fixes various decode bugs and adds options to encode
# Version 2.2   is much faster, Tcl8.0 compatible
# Version 2.2.1 bugfixes
# Version 2.2.2 bugfixes
# Version 2.3   bugfixes and extended to support Trf

# @mdgen EXCLUDE: base64c.tcl

package require Tcl 8.2
namespace eval ::base64 {
    namespace export encode decode
}

if {![catch {package require Trf 2.0}]} {
    # Trf is available, so implement the functionality provided here
    # in terms of calls to Trf for speed.

    # ::base64::encode --
    #
    #	Base64 encode a given string.
    #
    # Arguments:
    #	args	?-maxlen maxlen? ?-wrapchar wrapchar? string
    #	
    #		If maxlen is 0, the output is not wrapped.
    #
    # Results:
    #	A Base64 encoded version of $string, wrapped at $maxlen characters
    #	by $wrapchar.
    
    proc ::base64::encode {args} {
	# Set the default wrapchar and maximum line length to match
	# the settings for MIME encoding (RFC 3548, RFC 2045). These
	# are the settings used by Trf as well. Various RFCs allow for
	# different wrapping characters and wraplengths, so these may
	# be overridden by command line options.
	set wrapchar "\n"
	set maxlen 76

	if { [llength $args] == 0 } {
	    error "wrong # args: should be \"[lindex [info level 0] 0]\
		    ?-maxlen maxlen? ?-wrapchar wrapchar? string\""
	}

	set optionStrings [list "-maxlen" "-wrapchar"]
	for {set i 0} {$i < [llength $args] - 1} {incr i} {
	    set arg [lindex $args $i]
	    set index [lsearch -glob $optionStrings "${arg}*"]
	    if { $index == -1 } {
		error "unknown option \"$arg\": must be -maxlen or -wrapchar"
	    }
	    incr i
	    if { $i >= [llength $args] - 1 } {
		error "value for \"$arg\" missing"
	    }
	    set val [lindex $args $i]

	    # The name of the variable to assign the value to is extracted
	    # from the list of known options, all of which have an
	    # associated variable of the same name as the option without
	    # a leading "-". The [string range] command is used to strip
	    # of the leading "-" from the name of the option.
	    #
	    # FRINK: nocheck
	    set [string range [lindex $optionStrings $index] 1 end] $val
	}
    
	# [string is] requires Tcl8.2; this works with 8.0 too
	if {[catch {expr {$maxlen % 2}}]} {
	    return -code error "expected integer but got \"$maxlen\""
	} elseif {$maxlen < 0} {
	    return -code error "expected positive integer but got \"$maxlen\""
	}

	set string [lindex $args end]
	set result [::base64 -mode encode -- $string]

	# Trf's encoder implicitly uses the settings -maxlen 76,
	# -wrapchar \n for its output. We may have to reflow this for
	# the settings chosen by the user. A second difference is that
	# Trf closes the output with the wrap char sequence,
	# always. The code here doesn't. Therefore 'trimright' is
	# needed in the fast cases.

	if {($maxlen == 76) && [string equal $wrapchar \n]} {
	    # Both maxlen and wrapchar are identical to Trf's
	    # settings. This is the super-fast case, because nearly
	    # nothing has to be done. Only thing to do is strip a
	    # terminating wrapchar.
	    set result [string trimright $result]
	} elseif {$maxlen == 76} {
	    # wrapchar has to be different here, length is the
	    # same. We can use 'string map' to transform the wrap
	    # information.
	    set result [string map [list \n $wrapchar] \
			    [string trimright $result]]
	} elseif {$maxlen == 0} {
	    # Have to reflow the output to no wrapping. Another fast
	    # case using only 'string map'. 'trimright' is not needed
	    # here.

	    set result [string map [list \n ""] $result]
	} else {
	    # Have to reflow the output from 76 to the chosen maxlen,
	    # and possibly change the wrap sequence as well.

	    # Note: After getting rid of the old wrap sequence we
	    # extract the relevant segments from the string without
	    # modifying the string. Modification, i.e. removal of the
	    # processed part, means 'shifting down characters in
	    # memory', making the algorithm O(n^2). By avoiding the
	    # modification we stay in O(n).
	    
	    set result [string map [list \n ""] $result]
	    set l [expr {[string length $result]-$maxlen}]
	    for {set off 0} {$off < $l} {incr off $maxlen} {
		append res [string range $result $off [expr {$off+$maxlen-1}]] $wrapchar
	    }
	    append res [string range $result $off end]
	    set result $res
	}

	return $result
    }

    # ::base64::decode --
    #
    #	Base64 decode a given string.
    #
    # Arguments:
    #	string	The string to decode.  Characters not in the base64
    #		alphabet are ignored (e.g., newlines)
    #
    # Results:
    #	The decoded value.

    proc ::base64::decode {string} {
	regsub -all {\s} $string {} string
	::base64 -mode decode -- $string
    }

} else {
    # Without Trf use a pure tcl implementation

    namespace eval base64 {
	variable base64 {}
	variable base64_en {}

	# We create the auxiliary array base64_tmp, it will be unset later.
	variable base64_tmp
	variable i

	set i 0
	foreach char {A B C D E F G H I J K L M N O P Q R S T U V W X Y Z \
		a b c d e f g h i j k l m n o p q r s t u v w x y z \
		0 1 2 3 4 5 6 7 8 9 + /} {
	    set base64_tmp($char) $i
	    lappend base64_en $char
	    incr i
	}

	#
	# Create base64 as list: to code for instance C<->3, specify
	# that [lindex $base64 67] be 3 (C is 67 in ascii); non-coded
	# ascii chars get a {}. we later use the fact that lindex on a
	# non-existing index returns {}, and that [expr {} < 0] is true
	#

	# the last ascii char is 'z'
	variable char
	variable len
	variable val

	scan z %c len
	for {set i 0} {$i <= $len} {incr i} {
	    set char [format %c $i]
	    set val {}
	    if {[info exists base64_tmp($char)]} {
		set val $base64_tmp($char)
	    } else {
		set val {}
	    }
	    lappend base64 $val
	}

	# code the character "=" as -1; used to signal end of message
	scan = %c i
	set base64 [lreplace $base64 $i $i -1]

	# remove unneeded variables
	unset base64_tmp i char len val

	namespace export encode decode
    }

    # ::base64::encode --
    #
    #	Base64 encode a given string.
    #
    # Arguments:
    #	args	?-maxlen maxlen? ?-wrapchar wrapchar? string
    #	
    #		If maxlen is 0, the output is not wrapped.
    #
    # Results:
    #	A Base64 encoded version of $string, wrapped at $maxlen characters
    #	by $wrapchar.
    
    proc ::base64::encode {args} {
	set base64_en $::base64::base64_en
	
	# Set the default wrapchar and maximum line length to match
	# the settings for MIME encoding (RFC 3548, RFC 2045). These
	# are the settings used by Trf as well. Various RFCs allow for
	# different wrapping characters and wraplengths, so these may
	# be overridden by command line options.
	set wrapchar "\n"
	set maxlen 76

	if { [llength $args] == 0 } {
	    error "wrong # args: should be \"[lindex [info level 0] 0]\
		    ?-maxlen maxlen? ?-wrapchar wrapchar? string\""
	}

	set optionStrings [list "-maxlen" "-wrapchar"]
	for {set i 0} {$i < [llength $args] - 1} {incr i} {
	    set arg [lindex $args $i]
	    set index [lsearch -glob $optionStrings "${arg}*"]
	    if { $index == -1 } {
		error "unknown option \"$arg\": must be -maxlen or -wrapchar"
	    }
	    incr i
	    if { $i >= [llength $args] - 1 } {
		error "value for \"$arg\" missing"
	    }
	    set val [lindex $args $i]

	    # The name of the variable to assign the value to is extracted
	    # from the list of known options, all of which have an
	    # associated variable of the same name as the option without
	    # a leading "-". The [string range] command is used to strip
	    # of the leading "-" from the name of the option.
	    #
	    # FRINK: nocheck
	    set [string range [lindex $optionStrings $index] 1 end] $val
	}
    
	# [string is] requires Tcl8.2; this works with 8.0 too
	if {[catch {expr {$maxlen % 2}}]} {
	    return -code error "expected integer but got \"$maxlen\""
	} elseif {$maxlen < 0} {
	    return -code error "expected positive integer but got \"$maxlen\""
	}

	set string [lindex $args end]

	set result {}
	set state 0
	set length 0


	# Process the input bytes 3-by-3

	binary scan $string c* X

	foreach {x y z} $X {
	    ADD [lindex $base64_en [expr {($x >>2) & 0x3F}]]
	    if {$y != {}} {
		ADD [lindex $base64_en [expr {(($x << 4) & 0x30) | (($y >> 4) & 0xF)}]]
		if {$z != {}} {
		    ADD [lindex $base64_en [expr {(($y << 2) & 0x3C) | (($z >> 6) & 0x3)}]]
		    ADD [lindex $base64_en [expr {($z & 0x3F)}]]
		} else {
		    set state 2
		    break
		}
	    } else {
		set state 1
		break
	    }
	}
	if {$state == 1} {
	    ADD [lindex $base64_en [expr {(($x << 4) & 0x30)}]]
	    ADD =
	    ADD =
	} elseif {$state == 2} {
	    ADD [lindex $base64_en [expr {(($y << 2) & 0x3C)}]]
	    ADD =
	}
	return $result
    }

    proc ::base64::ADD {x} {
	# The line length check is always done before appending so
	# that we don't get an extra newline if the output is a
	# multiple of $maxlen chars long.

	upvar 1 maxlen maxlen length length result result wrapchar wrapchar
	if {$maxlen && $length >= $maxlen} {
	    append result $wrapchar
	    set length 0
	}
	append result $x
	incr length
	return
    }

    # ::base64::decode --
    #
    #	Base64 decode a given string.
    #
    # Arguments:
    #	string	The string to decode.  Characters not in the base64
    #		alphabet are ignored (e.g., newlines)
    #
    # Results:
    #	The decoded value.

    proc ::base64::decode {string} {
	if {[string length $string] == 0} {return ""}

	set base64 $::base64::base64
	set output "" ; # Fix for [Bug 821126]

	binary scan $string c* X
	foreach x $X {
	    set bits [lindex $base64 $x]
	    if {$bits >= 0} {
		if {[llength [lappend nums $bits]] == 4} {
		    foreach {v w z y} $nums break
		    set a [expr {($v << 2) | ($w >> 4)}]
		    set b [expr {(($w & 0xF) << 4) | ($z >> 2)}]
		    set c [expr {(($z & 0x3) << 6) | $y}]
		    append output [binary format ccc $a $b $c]
		    set nums {}
		}		
	    } elseif {$bits == -1} {
		# = indicates end of data.  Output whatever chars are left.
		# The encoding algorithm dictates that we can only have 1 or 2
		# padding characters.  If x=={}, we must (*) have 12 bits of input 
		# (enough for 1 8-bit output).  If x!={}, we have 18 bits of
		# input (enough for 2 8-bit outputs).
		#
		# (*) If we don't then the input is broken (bug 2976290).

		foreach {v w z} $nums break

		# Bug 2976290
		if {$w == {}} {
		    return -code error "Not enough data to process padding"
		}

		set a [expr {($v << 2) | (($w & 0x30) >> 4)}]
		if {$z == {}} {
		    append output [binary format c $a ]
		} else {
		    set b [expr {(($w & 0xF) << 4) | (($z & 0x3C) >> 2)}]
		    append output [binary format cc $a $b]
		}		
		break
	    } else {
		# RFC 2045 says that line breaks and other characters not part
		# of the Base64 alphabet must be ignored, and that the decoder
		# can optionally emit a warning or reject the message.  We opt
		# not to do so, but to just ignore the character. 
		continue
	    }
	}
	return $output
    }
}

package provide base64 2.4.2
#! /usr/bin/env tclsh
# -*- tcl -*-
# RSA
#
# (c) 2010, 2011, 2012, 2013 Roy Keene.
#	 BSD Licensed.

# # ## ### ##### ######## #############
## Requisites

package require Tcl 8.5

## Versions of asn lower than 0.8.4 are known to have defects
package require asn 0.8.4

## Further dependencies
package require aes
package require des
package require math::bignum
package require md5 2
package require sha1
package require sha256

# # ## ### ##### ######## #############
## Requisites

namespace eval ::pki {
	variable oids
	array set oids {
		1.2.840.113549.1.1.1           rsaEncryption
		1.2.840.113549.1.1.5           sha1WithRSAEncryption
		1.2.840.113549.2.5             md5
		1.3.14.3.2.26                  sha1
		2.16.840.1.101.3.4.2.1         sha256
		0.9.2342.19200300.100.1.1      uid
		0.9.2342.19200300.100.1.10     manager
		0.9.2342.19200300.100.1.11     documentIdentifier
		0.9.2342.19200300.100.1.12     documentTitle
		0.9.2342.19200300.100.1.13     documentVersion
		0.9.2342.19200300.100.1.14     documentAuthor
		0.9.2342.19200300.100.1.15     documentLocation
		0.9.2342.19200300.100.1.2      textEncodedORAddress
		0.9.2342.19200300.100.1.20     homePhone
		0.9.2342.19200300.100.1.21     secretary
		0.9.2342.19200300.100.1.22     otherMailbox
		0.9.2342.19200300.100.1.25     dc
		0.9.2342.19200300.100.1.26     aRecord
		0.9.2342.19200300.100.1.27     mDRecord
		0.9.2342.19200300.100.1.28     mXRecord
		0.9.2342.19200300.100.1.29     nSRecord
		0.9.2342.19200300.100.1.3      mail
		0.9.2342.19200300.100.1.30     sOARecord
		0.9.2342.19200300.100.1.31     cNAMERecord
		0.9.2342.19200300.100.1.37     associatedDomain
		0.9.2342.19200300.100.1.38     associatedName
		0.9.2342.19200300.100.1.39     homePostalAddress
		0.9.2342.19200300.100.1.4      info
		0.9.2342.19200300.100.1.40     personalTitle
		0.9.2342.19200300.100.1.41     mobile
		0.9.2342.19200300.100.1.42     pager
		0.9.2342.19200300.100.1.43     co
		0.9.2342.19200300.100.1.43     friendlyCountryName
		0.9.2342.19200300.100.1.44     uniqueIdentifier
		0.9.2342.19200300.100.1.45     organizationalStatus
		0.9.2342.19200300.100.1.46     janetMailbox
		0.9.2342.19200300.100.1.47     mailPreferenceOption
		0.9.2342.19200300.100.1.48     buildingName
		0.9.2342.19200300.100.1.49     dSAQuality
		0.9.2342.19200300.100.1.5      drink
		0.9.2342.19200300.100.1.50     singleLevelQuality
		0.9.2342.19200300.100.1.51     subtreeMinimumQuality
		0.9.2342.19200300.100.1.52     subtreeMaximumQuality
		0.9.2342.19200300.100.1.53     personalSignature
		0.9.2342.19200300.100.1.54     dITRedirect
		0.9.2342.19200300.100.1.55     audio
		0.9.2342.19200300.100.1.56     documentPublisher
		0.9.2342.19200300.100.1.6      roomNumber
		0.9.2342.19200300.100.1.60     jpegPhoto
		0.9.2342.19200300.100.1.7      photo
		0.9.2342.19200300.100.1.8      userClass
		0.9.2342.19200300.100.1.9      host
		1.2.840.113549.1.9.1           email
		1.3.6.1.4.1.2428.90.1.1        norEduOrgUniqueNumber
		1.3.6.1.4.1.2428.90.1.11       norEduOrgSchemaVersion
		1.3.6.1.4.1.2428.90.1.12       norEduOrgNIN
		1.3.6.1.4.1.2428.90.1.2        norEduOrgUnitUniqueNumber
		1.3.6.1.4.1.2428.90.1.3        norEduPersonBirthDate
		1.3.6.1.4.1.2428.90.1.4        norEduPersonLIN
		1.3.6.1.4.1.2428.90.1.5        norEduPersonNIN
		1.3.6.1.4.1.2428.90.1.6        norEduOrgAcronym
		1.3.6.1.4.1.2428.90.1.7        norEduOrgUniqueIdentifier
		1.3.6.1.4.1.2428.90.1.8        norEduOrgUnitUniqueIdentifier
		1.3.6.1.4.1.2428.90.1.9        federationFeideSchemaVersion
		1.3.6.1.4.1.250.1.57           labeledURI
		1.3.6.1.4.1.5923.1.1.1.1       eduPersonAffiliation
		1.3.6.1.4.1.5923.1.1.1.10      eduPersonTargetedID
		1.3.6.1.4.1.5923.1.1.1.2       eduPersonNickname
		1.3.6.1.4.1.5923.1.1.1.3       eduPersonOrgDN
		1.3.6.1.4.1.5923.1.1.1.4       eduPersonOrgUnitDN
		1.3.6.1.4.1.5923.1.1.1.5       eduPersonPrimaryAffiliation
		1.3.6.1.4.1.5923.1.1.1.6       eduPersonPrincipalName
		1.3.6.1.4.1.5923.1.1.1.7       eduPersonEntitlement
		1.3.6.1.4.1.5923.1.1.1.8       eduPersonPrimaryOrgUnitDN
		1.3.6.1.4.1.5923.1.1.1.9       eduPersonScopedAffiliation
		1.3.6.1.4.1.5923.1.2.1.2       eduOrgHomePageURI
		1.3.6.1.4.1.5923.1.2.1.3       eduOrgIdentityAuthNPolicyURI
		1.3.6.1.4.1.5923.1.2.1.4       eduOrgLegalName
		1.3.6.1.4.1.5923.1.2.1.5       eduOrgSuperiorURI
		1.3.6.1.4.1.5923.1.2.1.6       eduOrgWhitePagesURI
		1.3.6.1.4.1.5923.1.5.1.1       isMemberOf
		2.16.840.1.113730.3.1.1        carLicense
		2.16.840.1.113730.3.1.2        departmentNumber
		2.16.840.1.113730.3.1.216      userPKCS12
		2.16.840.1.113730.3.1.241      displayName
		2.16.840.1.113730.3.1.3        employeeNumber
		2.16.840.1.113730.3.1.39       preferredLanguage
		2.16.840.1.113730.3.1.4        employeeType
		2.16.840.1.113730.3.1.40       userSMIMECertificate
		2.5.4.0                        objectClass
		2.5.4.1                        aliasedEntryName
		2.5.4.10                       o
		2.5.4.11                       ou
		2.5.4.12                       title
		2.5.4.13                       description
		2.5.4.14                       searchGuide
		2.5.4.15                       businessCategory
		2.5.4.16                       postalAddress
		2.5.4.17                       postalCode
		2.5.4.18                       postOfficeBox
		2.5.4.19                       physicalDeliveryOfficeName
		2.5.4.2                        knowledgeInformation
		2.5.4.20                       telephoneNumber
		2.5.4.21                       telexNumber
		2.5.4.22                       teletexTerminalIdentifier
		2.5.4.23                       facsimileTelephoneNumber
		2.5.4.23                       fax
		2.5.4.24                       x121Address
		2.5.4.25                       internationaliSDNNumber
		2.5.4.26                       registeredAddress
		2.5.4.27                       destinationIndicator
		2.5.4.28                       preferredDeliveryMethod
		2.5.4.29                       presentationAddress
		2.5.4.3                        cn
		2.5.4.30                       supportedApplicationContext
		2.5.4.31                       member
		2.5.4.32                       owner
		2.5.4.33                       roleOccupant
		2.5.4.34                       seeAlso
		2.5.4.35                       userPassword
		2.5.4.36                       userCertificate
		2.5.4.37                       cACertificate
		2.5.4.38                       authorityRevocationList
		2.5.4.39                       certificateRevocationList
		2.5.4.4                        sn
		2.5.4.40                       crossCertificatePair
		2.5.4.41                       name
		2.5.4.42                       gn
		2.5.4.43                       initials
		2.5.4.44                       generationQualifier
		2.5.4.45                       x500UniqueIdentifier
		2.5.4.46                       dnQualifier
		2.5.4.47                       enhancedSearchGuide
		2.5.4.48                       protocolInformation
		2.5.4.49                       distinguishedName
		2.5.4.5                        serialNumber
		2.5.4.50                       uniqueMember
		2.5.4.51                       houseIdentifier
		2.5.4.52                       supportedAlgorithms
		2.5.4.53                       deltaRevocationList
		2.5.4.54                       dmdName
		2.5.4.6                        c
		2.5.4.65                       pseudonym
		2.5.4.7                        l
		2.5.4.8                        st
		2.5.4.9                        street
		2.5.29.14                      id-ce-subjectKeyIdentifier
		2.5.29.15                      id-ce-keyUsage
		2.5.29.16                      id-ce-privateKeyUsagePeriod
		2.5.29.17                      id-ce-subjectAltName
		2.5.29.18                      id-ce-issuerAltName
		2.5.29.19                      id-ce-basicConstraints
		2.5.29.20                      id-ce-cRLNumber
		2.5.29.32                      id-ce-certificatePolicies
		2.5.29.33                      id-ce-cRLDistributionPoints
		2.5.29.35                      id-ce-authorityKeyIdentifier
	}

	variable handlers
	array set handlers {
		rsa                            {::pki::rsa::encrypt ::pki::rsa::decrypt ::pki::rsa::generate ::pki::rsa::serialize_key}
	}

	variable INT_MAX [expr {[format "%u" -1] / 2}]
}

namespace eval ::pki::rsa {}
namespace eval ::pki::x509 {}
namespace eval ::pki::pkcs {}

# # ## ### ##### ######## #############
## Implementation

proc ::pki::_dec_to_hex {num} {
	set retval [format %llx $num]
	return $retval
}

proc ::pki::_dec_to_ascii {num {bitlen -1}} {
	set retval ""

	while {$num} {
		set currchar [expr {$num & 0xff}]
		set retval "[format %c $currchar]$retval"
		set num [expr {$num >> 8}]
	}

	if {$bitlen != -1} {
		set bytelen [expr {$bitlen / 8}]
		while {[string length $retval] < $bytelen} {
			set retval "\x00$retval"
		}
	}

	return $retval
}

proc ::pki::_powm {x y m} {
	if {$y == 0} {
		return 1
	}

	set retval 1

	while {$y > 0} {
		if {($y & 1) == 1} {
			set retval [expr {($retval * $x) % $m}]
		}

		set y [expr {$y >> 1}]
		set x [expr {($x * $x) % $m}]
	}

	return $retval
}

## **NOTE** Requires that "m" be prime
### a^-1 === a^(m-2)    (all mod m)
proc ::pki::_modi {a m} {
	return [_powm $a [expr {$m - 2}] $m]
}

proc ::pki::_oid_number_to_name {oid} {
	set oid [join $oid .]

	if {[info exists ::pki::oids($oid)]} {
		return $::pki::oids($oid)
	}

	return $oid
}

proc ::pki::_oid_name_to_number {name} {
	foreach {chkoid chkname} [array get ::pki::oids] {
		if {[string equal -nocase $chkname $name]} {
			return [split $chkoid .]
		}
	}

	return -code error
}

proc ::pki::rsa::_encrypt_num {input exponent mod} {
	set ret [::pki::_powm $input $exponent $mod]

	return $ret
}

proc ::pki::rsa::_decrypt_num {input exponent mod} {
	set ret [::pki::_powm $input $exponent $mod]

	return $ret
}

proc ::pki::_pad_pkcs {data bitlength {blocktype 2}} {
	set ret ""

	set bytes_to_pad [expr {($bitlength / 8) - 3 - [string length $data]}]
	if {$bytes_to_pad < 0} {
		return $data
	}

	switch -- $blocktype {
		0 {
		}
		1 {
			append ret "\x00\x01"
			append ret [string repeat "\xff" $bytes_to_pad]
			append ret "\x00"
		}
		2 {
			append ret "\x00\x02"
			for {set idx 0} {$idx < $bytes_to_pad} {incr idx} {
				append ret [format %c [expr {int(rand() * 255 + 1)}]]
			}
			append ret "\x00"
		}
	}

	append ret $data

	return $ret
}

proc ::pki::_unpad_pkcs {data} {
	set check [string index $data 0]
	binary scan [string index $data 1] H* blocktype
	set datalen [string length $data]

	if {$check != "\x00"} {
		return $data
	}

	switch -- $blocktype {
		"00" {
			# Padding Scheme 1, the first non-zero byte is the start of data
			for {set idx 2} {$idx < $datalen} {incr idx} {
				set char [string index $data $idx]
				if {$char != "\x00"} {
					set ret [string range $data $idx end]
				}
			}
		}
		"01" {
			# Padding Scheme 2, pad bytes are 0xFF followed by 0x00
			for {set idx 2} {$idx < $datalen} {incr idx} {
				set char [string index $data $idx]
				if {$char != "\xff"} {
					if {$char == "\x00"} {
						set ret [string range $data [expr {$idx + 1}] end]

						break
					} else {
						return -code error "Invalid padding, seperator byte is not 0x00"
					}
				}
			}
		}
		"02" {
			# Padding Scheme 3, pad bytes are random, followed by 0x00
			for {set idx 2} {$idx < $datalen} {incr idx} {
				set char [string index $data $idx]
				if {$char == "\x00"} {
					set ret [string range $data [expr {$idx + 1}] end]

					break
				}
			}
		}
		default {
			return $data
		}
	}

	if {![info exists ret]} {
		return -code error "Invalid padding, no seperator byte found"
	}

	return $ret
}

proc ::pki::rsa::encrypt {mode input keylist} {
	switch -- $mode {
		"pub" {
			set exponent_ent e
		}
		"priv" {
			set exponent_ent d
		}
	}

	array set key $keylist

	set exponent $key($exponent_ent)
	set mod $key(n)

	## RSA requires that the input be no larger than the key
	set input_len_bits [expr {[string length $input] * 8}]
	if {$key(l) < $input_len_bits} {
		return -code error "Message length exceeds key length"
	}

	binary scan $input H* input_num

	set input_num "0x${input_num}"

	set retval_num [_encrypt_num $input_num $exponent $mod]

	set retval [::pki::_dec_to_ascii $retval_num $key(l)]

	return $retval
}

proc ::pki::rsa::decrypt {mode input keylist} {
	switch -- $mode {
		"pub" {
			set exponent_ent e
		}
		"priv" {
			set exponent_ent d
		}
	}

	array set key $keylist

	set exponent $key($exponent_ent)
	set mod $key(n)

	binary scan $input H* input_num

	set input_num "0x${input_num}"

	set retval_num [_decrypt_num $input_num $exponent $mod]

	set retval [::pki::_dec_to_ascii $retval_num $key(l)]

	return $retval
}

proc ::pki::rsa::serialize_key {keylist} {
	array set key $keylist

	foreach entry [list n e d p q] {
		if {![info exists key($entry)]} {
			return -code error "Key does not contain an element $entry"
		}
	}

	# Exponent 1
	## d (mod p-1)
	set e1 [expr {$key(d) % ($key(p) - 1)}]

	# Exponent 2
	#set e2 [expr d mod (q-1)]
	set e2 [expr {$key(d) % ($key(q) - 1)}]

	# Coefficient
	## Modular multiplicative inverse of q mod p
	set c [::pki::_modi $key(q) $key(p)]

	set ret [::asn::asnSequence \
			[::asn::asnBigInteger [::math::bignum::fromstr 0]] \
			[::asn::asnBigInteger [::math::bignum::fromstr $key(n)]] \
			[::asn::asnBigInteger [::math::bignum::fromstr $key(e)]] \
			[::asn::asnBigInteger [::math::bignum::fromstr $key(d)]] \
			[::asn::asnBigInteger [::math::bignum::fromstr $key(p)]] \
			[::asn::asnBigInteger [::math::bignum::fromstr $key(q)]] \
			[::asn::asnBigInteger [::math::bignum::fromstr $e1]] \
			[::asn::asnBigInteger [::math::bignum::fromstr $e2]] \
			[::asn::asnBigInteger [::math::bignum::fromstr $c]] \
	]

	return [list data $ret begin "-----BEGIN RSA PRIVATE KEY-----" end "-----END RSA PRIVATE KEY-----"]
}

proc ::pki::_lookup_command {action keylist} {
	array set key $keylist

	set type $key(type)

	switch -- $action {
		"encrypt" {
			set idx 0
		}
		"decrypt" {
			set idx 1
		}
		"generate" {
			set idx 2
		}
		"serialize_key" {
			set idx 3
		}
	}

	set cmdlist $::pki::handlers($type)

	set ret [lindex $cmdlist $idx]

	return $ret
}

proc ::pki::encrypt args {
	set outmode "hex"
	set enablepad 1

	set argsmode 0
	set newargs [list]
	foreach arg $args {
		if {![string match "-*" $arg]} {
			set argsmode 1
		}

		if {$argsmode} {
			lappend newargs $arg
			continue
		}

		switch -- $arg {
			"-pub" {
				set mode pub
				set padmode 2
			}
			"-priv" {
				set mode priv
				set padmode 1
			}
			"-hex" {
				set outmode "hex"
			}
			"-binary" {
				set outmode "bin"
			}
			"-pad" {
				set enablepad 1
			}
			"-nopad" {
				set enablepad 0
			}
			"--" {
				set argsmode 1
			}
			default {
				return -code error "usage: encrypt ?-binary? ?-hex? ?-pad? ?-nopad? -priv|-pub ?--? input key"
			}
		}
	}
	set args $newargs

	if {[llength $args] != 2 || ![info exists mode]} {
		return -code error "usage: encrypt ?-binary? ?-hex? ?-pad? ?-nopad? -priv|-pub ?--? input key"
	}

	set input [lindex $args 0]
	set keylist [lindex $args 1]
	array set key $keylist

	if {$enablepad} {
		set input [::pki::_pad_pkcs $input $key(l) $padmode]
	}

	set encrypt [::pki::_lookup_command encrypt $keylist]

	set retval [$encrypt $mode $input $keylist]

	switch -- $outmode {
		"hex" {
			binary scan $retval H* retval
		}
	}

	return $retval
}

proc ::pki::decrypt args {
	set inmode "hex"
	set enableunpad 1

	set argsmode 0
	set newargs [list]
	foreach arg $args {
		if {![string match "-*" $arg]} {
			set argsmode 1
		}

		if {$argsmode} {
			lappend newargs $arg
			continue
		}

		switch -- $arg {
			"-pub" {
				set mode pub
			}
			"-priv" {
				set mode priv
			}
			"-hex" {
				set inmode "hex"
			}
			"-binary" {
				set inmode "bin"
			}
			"-unpad" {
				set enableunpad 1
			}
			"-nounpad" {
				set enableunpad 0
			}
			"--" {
				set argsmode 1
			}
			default {
				return -code error "usage: decrypt ?-binary? ?-hex? ?-unpad? ?-nounpad? -priv|-pub ?--? input key"
			}
		}
	}
	set args $newargs

	if {[llength $args] != 2 || ![info exists mode]} {
		return -code error "usage: decrypt ?-binary? ?-hex? ?-unpad? ?-nounpad? -priv|-pub ?--? input key"
	}

	set input [lindex $args 0]
	set keylist [lindex $args 1]
	array set key $keylist

	switch -- $inmode {
		"hex" {
			set input [binary format H* $input]
		}
	}

	set decrypt [::pki::_lookup_command decrypt $keylist]

	set retval [$decrypt $mode $input $keylist]

	if {$enableunpad} {
		set retval [::pki::_unpad_pkcs $retval]
	}

	return $retval
}

# Hash and encrypt with private key
proc ::pki::sign {input keylist {algo "sha1"}} {
	switch -- $algo {
		"md5" {
			package require md5

			set header "\x30\x20\x30\x0c\x06\x08\x2a\x86\x48\x86\xf7\x0d\x02\x05\x05\x00\x04\x10"
			set hash [md5::md5 $input]
		}
		"sha1" {
			package require sha1

			set header "\x30\x21\x30\x09\x06\x05\x2b\x0e\x03\x02\x1a\x05\x00\x04\x14"
			set hash [sha1::sha1 -bin $input]
		}
		"sha256" {
			package require sha256

			set header "\x30\x31\x30\x0d\x06\x09\x60\x86\x48\x01\x65\x03\x04\x02\x01\x05\x00\x04\x20"
			set hash [sha2::sha256 -bin $input]
		}
		"raw" {
			set header ""
			set hash $input
		}
		default {
			return -code error "Invalid algorithm selected, must be one of: md5, sha1, sha256, raw"
		}
	}

	set plaintext "${header}${hash}"

	array set key $keylist

	set padded [::pki::_pad_pkcs $plaintext $key(l) 1]

	return [::pki::encrypt -binary -nopad -priv -- $padded $keylist]
}

# Verify known-plaintext with signature
proc ::pki::verify {signedmessage checkmessage keylist {algo default}} {
	package require asn

	if {[catch {
		set plaintext [::pki::decrypt -binary -unpad -pub -- $signedmessage $keylist]
	}]} {
		return false
	}

	if {$algo == "default"} {
		set algoId "unknown"
		set digest ""

		catch {
			::asn::asnGetSequence plaintext message
			::asn::asnGetSequence message digestInfo
			::asn::asnGetObjectIdentifier digestInfo algoId
			::asn::asnGetOctetString message digest
		}

		set algoId [::pki::_oid_number_to_name $algoId]
	} else {
		set algoId $algo
		set digest $plaintext
	}

	switch -- $algoId {
		"md5" - "md5WithRSAEncryption" {
			set checkdigest [md5::md5 $checkmessage]
		}
		"sha1" - "sha1WithRSAEncryption" {
			set checkdigest [sha1::sha1 -bin $checkmessage]
		}
		"sha256" - "sha256WithRSAEncryption" {
			set checkdigest [sha2::sha256 -bin $checkmessage]
		}
		default {
			return -code error "Unknown hashing algorithm: $algoId"
		}
	}

	if {$checkdigest != $digest} {
		return false
	}

	return true
}

proc ::pki::key {keylist {password ""} {encodePem 1}} {
	set serialize_key [::pki::_lookup_command serialize_key $keylist]

	if {$serialize_key eq ""} {
		array set key $keylist

		return -code error "Do not know how to serialize an $key(type) key"
	}

	array set retval_parts [$serialize_key $keylist]

	if {$encodePem} {
		set retval [::pki::_encode_pem $retval_parts(data) $retval_parts(begin) $retval_parts(end) $password]
	} else {
		if {$password != ""} {
			return -code error "DER encoded keys may not be password protected"
		}

		set retval $retval_parts(data)
	}

	return $retval
}

proc ::pki::_parse_init {} {
	if {[info exists ::pki::_parse_init_done]} {
		return
	}

	package require asn

	set test "FAIL"
	catch {
		set test [binary decode base64 "UEFTUw=="]
	}

	switch -- $test {
		"PASS" {
			set ::pki::rsa::base64_binary 1
		}
		"FAIL" {
			set ::pki::rsa::base64_binary 0

			package require base64
		}
	}

	set ::pki::_parse_init_done 1
	return
}

proc ::pki::_getopensslkey {password salt bytes} {
	package require md5

	set salt [string range $salt 0 7]

	set saltedkey "${password}${salt}"
	for {set ret ""} {[string length $ret] < $bytes} {} {
		if {![info exists hash]} {
			set hash $saltedkey
		} else {
			set hash "${hash}${saltedkey}"
		}

		set hash [md5::md5 $hash]

		append ret $hash
	}

	if {[string length $ret] < $bytes} {
		set bytes_to_add [expr $bytes - [string length $ret]]
		set ret "[string repeat "\x00" $bytes_to_add]${ret}"
	}

	set ret [string range $ret 0 [expr {$bytes - 1}]]

	return $ret
}

proc ::pki::_encode_pem {data begin end {password ""} {algo "aes-256-cbc"}} {
	set ret ""

	append ret "${begin}\n"
	if {$password != ""} {
		switch -glob -- $algo {
			"aes-*" {
				set algostr [string toupper $algo]
				set work [split $algo "-"]
				set algo "aes"
				set keysize [lindex $work 1]
				set mode [lindex $work 2]
				set blocksize 16
				set ivsize [expr {$blocksize * 8}]
			}
			default {
				return -code error "Only AES is currently supported"
			}
		}

		set keybytesize [expr {$keysize / 8}]
		set ivbytesize [expr {$ivsize / 8}]

		set iv ""
		while {[string length $iv] < $ivbytesize} {
			append iv [::pki::_random -binary]
		}
		set iv [string range $iv 0 [expr {$ivbytesize - 1}]]

		set password_key [::pki::_getopensslkey $password $iv $keybytesize]

		set pad [expr {$blocksize - ([string length $data] % $blocksize)}]
		append data [string repeat "\x09" $pad]

		switch -- $algo {
			"aes" {
				set data [aes::aes -dir encrypt -mode $mode -iv $iv -key $password_key -- $data]
			}
		}

		binary scan $iv H* iv
		set iv [string toupper $iv]

		append ret "Proc-Type: 4,ENCRYPTED\n"
		append ret "DEK-Info: $algostr,$iv\n"
		append ret "\n"
	}

	if {$::pki::rsa::base64_binary} {
		append ret [binary encode base64 -maxlen 64 $data]
	} else {
		append ret [::base64::encode -maxlen 64 $data]
	}
	append ret "\n"
	append ret "${end}\n"

	return $ret
}

proc ::pki::_parse_pem {pem begin end {password ""}} {
	# Unencode a PEM-encoded object
	set testpem [split $pem \n]
	set pem_startidx [lsearch -exact $testpem $begin]
	set pem_endidx [lsearch -exact -start $pem_startidx $testpem $end]

	if {$pem_startidx == -1 || $pem_endidx == -1} {
		return [list data $pem]
	}

	set pem $testpem

	incr pem_startidx
	incr pem_endidx -1

	array set ret [list]

	set newpem ""
	foreach line [lrange $pem $pem_startidx $pem_endidx] {
		if {[string match "*:*" $line]} {
			set work [split $line :]

			set var [string toupper [lindex $work 0]]
			set val [string trim [join [lrange $work 1 end] :]]

			set ret($var) $val

			continue
		}

		set line [string trim $line]

		append newpem $line
	}

	if {$newpem != ""} {
		if {$::pki::rsa::base64_binary} {
			set pem [binary decode base64 $newpem]
		} else {
			set pem [::base64::decode $newpem]
		}
	}

	if {[info exists ret(PROC-TYPE)] && [info exists ret(DEK-INFO)]} {
		if {$ret(PROC-TYPE) == "4,ENCRYPTED"} {
			if {$password == ""} {
				return [list error "ENCRYPTED"]
			}

			switch -glob -- $ret(DEK-INFO) {
				"DES-EDE3-*" {
					package require des

					# DES-EDE3-CBC,03B1F1883BFA4412
					set keyinfo $ret(DEK-INFO)

					set work [split $keyinfo ,]
					set cipher [lindex $work 0]
					set iv [lindex $work 1]

					set work [split $cipher -]
					set algo [lindex $work 0]
					set mode [string tolower [lindex $work 2]]

					set iv [binary format H* $iv]
					set password_key [::pki::_getopensslkey $password $iv 24]

					set pem [DES::des -dir decrypt -mode $mode -iv $iv -key $password_key -- $pem]
				}
				"AES-*" {
					package require aes

					# AES-256-CBC,AF517BA39E94FF39D1395C63F6DE9657
					set keyinfo $ret(DEK-INFO)

					set work [split $keyinfo ,]
					set cipher [lindex $work 0]
					set iv [lindex $work 1]

					set work [split $cipher -]
					set algo [lindex $work 0]
					set keysize [lindex $work 1]
					set mode [string tolower [lindex $work 2]]

					set iv [binary format H* $iv]
					set password_key [::pki::_getopensslkey $password $iv [expr $keysize / 8]]

					set pem [aes::aes -dir decrypt -mode $mode -iv $iv -key $password_key -- $pem]
				}
			}
		}
	}

	set ret(data) $pem

	return [array get ret]
}

proc ::pki::pkcs::parse_key {key {password ""}} {
	array set parsed_key [::pki::_parse_pem $key "-----BEGIN RSA PRIVATE KEY-----" "-----END RSA PRIVATE KEY-----" $password]

	set key_seq $parsed_key(data)

	::asn::asnGetSequence key_seq key
	::asn::asnGetBigInteger key version
	::asn::asnGetBigInteger key ret(n)
	::asn::asnGetBigInteger key ret(e)
	::asn::asnGetBigInteger key ret(d)
	::asn::asnGetBigInteger key ret(p)
	::asn::asnGetBigInteger key ret(q)

	set ret(n) [::math::bignum::tostr $ret(n)]
	set ret(e) [::math::bignum::tostr $ret(e)]
	set ret(d) [::math::bignum::tostr $ret(d)]
	set ret(p) [::math::bignum::tostr $ret(p)]
	set ret(q) [::math::bignum::tostr $ret(q)]
	set ret(l) [expr {int([::pki::_bits $ret(n)] / 8.0000 + 0.5) * 8}]
	set ret(type) rsa

	return [array get ret]
}

proc ::pki::x509::_dn_to_list {dn} {
	set ret ""

	while {$dn != ""} {
		::asn::asnGetSet dn dn_parts
		::asn::asnGetSequence dn_parts curr_part
		::asn::asnGetObjectIdentifier curr_part label
		::asn::asnGetString curr_part value

		set label [::pki::_oid_number_to_name $label]
		lappend ret $label $value
	}

	return $ret
}

proc ::pki::x509::_list_to_dn {name} {
	set ret ""
	foreach {oid_name value} $name {
		if {![regexp {[^ A-Za-z0-9'()+,.:/?=-]} $value]} {
			set asnValue [::asn::asnPrintableString $value]
		} else {
			set asnValue [::asn::asnUTF8String $value]
		}

		append ret [::asn::asnSet \
			[::asn::asnSequence \
				[::asn::asnObjectIdentifier [::pki::_oid_name_to_number $oid_name]] \
				$asnValue \
			] \
		] \
	}

	return $ret
}

proc ::pki::x509::_dn_to_string {dn} {
	set ret [list]

	foreach {label value} [_dn_to_list $dn] {
		set label [string toupper $label]

		lappend ret "$label=$value"
	}

	set ret [join $ret {, }]

	return $ret
}

proc ::pki::x509::_string_to_dn {string} {
	foreach {label value} [split $string ",="] {
		set label [string trim $label]
		set value [string trim $value]

		lappend namelist $label $value
	}

	return [_list_to_dn $namelist]
}

proc ::pki::x509::_dn_to_cn {dn} {
	foreach {label value} [split $dn ",="] {
		set label [string toupper [string trim $label]]
		set value [string trim $value]

		if {$label == "CN"} {
			return $value
		}
	}

	return ""
}

proc ::pki::x509::_utctime_to_native {utctime} {
	return [clock scan $utctime -format {%y%m%d%H%M%SZ} -gmt true]
}

proc ::pki::x509::_native_to_utctime {time} {
	return [clock format $time -format {%y%m%d%H%M%SZ} -gmt true]
}

proc ::pki::x509::parse_cert {cert} {
	array set parsed_cert [::pki::_parse_pem $cert "-----BEGIN CERTIFICATE-----" "-----END CERTIFICATE-----"]
	set cert_seq $parsed_cert(data)

	array set ret [list]

	# Decode X.509 certificate, which is an ASN.1 sequence
	::asn::asnGetSequence cert_seq wholething
	::asn::asnGetSequence wholething cert

	set ret(cert) $cert
	set ret(cert) [::asn::asnSequence $ret(cert)]
	binary scan $ret(cert) H* ret(cert)

	::asn::asnPeekByte cert peek_tag
	if {$peek_tag != 0x02} {
		# Version number is optional, if missing assumed to be value of 0
		::asn::asnGetContext cert - asn_version
		::asn::asnGetInteger asn_version ret(version)
		incr ret(version)
	} else {
		set ret(version) 1
	}

	::asn::asnGetBigInteger cert ret(serial_number)
	::asn::asnGetSequence cert data_signature_algo_seq
		::asn::asnGetObjectIdentifier data_signature_algo_seq ret(data_signature_algo)
	::asn::asnGetSequence cert issuer
	::asn::asnGetSequence cert validity
		::asn::asnGetUTCTime validity ret(notBefore)
		::asn::asnGetUTCTime validity ret(notAfter)
	::asn::asnGetSequence cert subject
	::asn::asnGetSequence cert pubkeyinfo
		::asn::asnGetSequence pubkeyinfo pubkey_algoid
			::asn::asnGetObjectIdentifier pubkey_algoid ret(pubkey_algo)
		::asn::asnGetBitString pubkeyinfo pubkey

	set extensions_list [list]
	while {$cert != ""} {
		::asn::asnPeekByte cert peek_tag

		switch -- [format {0x%02x} $peek_tag] {
			"0xa1" {
				::asn::asnGetContext cert - issuerUniqID
			}
			"0xa2" {
				::asn::asnGetContext cert - subjectUniqID
			}
			"0xa3" {
				::asn::asnGetContext cert - extensions_ctx
				::asn::asnGetSequence extensions_ctx extensions
				while {$extensions != ""} {
					::asn::asnGetSequence extensions extension
						::asn::asnGetObjectIdentifier extension ext_oid

						::asn::asnPeekByte extension peek_tag
						if {$peek_tag == 0x1} {
							::asn::asnGetBoolean extension ext_critical
						} else {
							set ext_critical false
						}

						::asn::asnGetOctetString extension ext_value_seq

					set ext_oid [::pki::_oid_number_to_name $ext_oid]

					set ext_value [list $ext_critical]

					switch -- $ext_oid {
						id-ce-basicConstraints {
							::asn::asnGetSequence ext_value_seq ext_value_bin

							if {$ext_value_bin != ""} {
								::asn::asnGetBoolean ext_value_bin allowCA
							} else {
								set allowCA "false"
							}

							if {$ext_value_bin != ""} {
								::asn::asnGetInteger ext_value_bin caDepth
							} else {
								set caDepth -1
							}
						
							lappend ext_value $allowCA $caDepth
						}
						default {
							binary scan $ext_value_seq H* ext_value_seq_hex
							lappend ext_value $ext_value_seq_hex
						}
					}

					lappend extensions_list $ext_oid $ext_value
				}
			}
		}
	}
	set ret(extensions) $extensions_list

	::asn::asnGetSequence wholething signature_algo_seq
	::asn::asnGetObjectIdentifier signature_algo_seq ret(signature_algo)
	::asn::asnGetBitString wholething ret(signature)

	# Convert values from ASN.1 decoder to usable values if needed
	set ret(notBefore) [::pki::x509::_utctime_to_native $ret(notBefore)]
	set ret(notAfter) [::pki::x509::_utctime_to_native $ret(notAfter)]
	set ret(serial_number) [::math::bignum::tostr $ret(serial_number)]
	set ret(data_signature_algo) [::pki::_oid_number_to_name $ret(data_signature_algo)]
	set ret(signature_algo) [::pki::_oid_number_to_name $ret(signature_algo)]
	set ret(pubkey_algo) [::pki::_oid_number_to_name $ret(pubkey_algo)]
	set ret(issuer) [_dn_to_string $issuer]
	set ret(subject) [_dn_to_string $subject]
	set ret(signature) [binary format B* $ret(signature)]
	binary scan $ret(signature) H* ret(signature)

	# Handle RSA public keys by extracting N and E
	switch -- $ret(pubkey_algo) {
		"rsaEncryption" {
			set pubkey [binary format B* $pubkey]
			binary scan $pubkey H* ret(pubkey)

			::asn::asnGetSequence pubkey pubkey_parts
			::asn::asnGetBigInteger pubkey_parts ret(n)
			::asn::asnGetBigInteger pubkey_parts ret(e)

			set ret(n) [::math::bignum::tostr $ret(n)]
			set ret(e) [::math::bignum::tostr $ret(e)]
			set ret(l) [expr {int([::pki::_bits $ret(n)] / 8.0000 + 0.5) * 8}]
			set ret(type) rsa
		}
	}

	return [array get ret]
}

# Verify whether a cert is valid, regardless of trust
proc ::pki::x509::validate_cert {cert args} {
	# Verify arguments and load options
	for {set idx 0} {$idx < [llength $args]} {incr idx} {
		set arg [lindex $args $idx]

		switch -- $arg {
			"-sign_message" {
				incr idx
				set dn [lindex $args $idx]
				set cn [_dn_to_cn $dn]

				set opts(sign_message) $cn
			}
			"-encrypt_message" {
				incr idx
				set dn [lindex $args $idx]
				set cn [_dn_to_cn $dn]

				set opts(encrypt_message) $cn
			}
			"-sign_cert" {
				incr idx
				set dn [lindex $args $idx]
				if {$dn == "ALL" || $dn == "ANY"} {
					set cn $dn
				} else {
					set cn [_dn_to_cn $dn]
				}

				incr idx
				set currdepth [lindex $args $idx]

				set opts(sign_cert) [list $cn $currdepth]
			}
			"-ssl" {
				incr idx
				set dn [lindex $args $idx]
				set cn [_dn_to_cn $dn]

				set opts(ssl) $cn
			}
			default {
				return -code error {wrong # args: should be "validate_cert cert ?-sign_message dn_of_signer? ?-encrypt_message dn_of_signer? ?-sign_cert [dn_to_be_signed | ANY | ALL] ca_depth? ?-ssl dn?"}
			}
		}
	}

	# Load cert
	array set cert_arr $cert

	# Validate certificate
	## Validate times
	if {![info exists cert_arr(notBefore)] || ![info exists cert_arr(notAfter)]} {
		return false
	}

	set currtime [clock seconds]
	if {$currtime < $cert_arr(notBefore) || $currtime > $cert_arr(notAfter)} {
		return false
	}

	# Check for extensions and process them
	## Critical extensions must be understood, non-critical extensions may be ignored if not understood
	set CA 0
	set CAdepth -1
	foreach {ext_id ext_val} $cert_arr(extensions) {
		set critical [lindex $ext_val 0]

		switch -- $ext_id {
			id-ce-basicConstraints {
				set CA [lindex $ext_val 1]
				set CAdepth [lindex $ext_val 2]
			}
			default {
				### If this extensions is critical and not understood, we must reject it
				if {$critical} {
					return false
				}
			}
		}
	}

	if {[info exists opts(sign_cert)]} {
		if {!$CA} {
			return false
		}

		if {$CAdepth >= 0} {
			set sign_depth [lindex $opts(sign_cert) 1]
			if {$sign_depth > $CAdepth} {
				return false
			}
		}
	}

	return true
}

proc ::pki::x509::verify_cert {cert trustedcerts {intermediatecerts ""}} {
	# Validate cert
	if {![validate_cert $cert]} {
		return false;
	}

	# Load trusted certs
	foreach trustedcert_list $trustedcerts {
		if {![validate_cert $trustedcert_list -sign_cert ANY -1]} {
			continue
		}

		unset -nocomplain trustedcert
		array set trustedcert $trustedcert_list

		set subject $trustedcert(subject)

		set trustedcertinfo($subject) $trustedcert_list
	}

	# Load intermediate certs
	foreach intermediatecert_list $intermediatecerts {
		if {![validate_cert $intermediatecert_list -sign_cert ANY -1]} {
			continue
		}

		unset -nocomplain intermediatecert
		array set intermediatecert $intermediatecert_list

		set subject $intermediatecert(subject)

		set intermediatecertinfo($subject) $intermediatecert_list
	}

	# Load cert
	array set cert_arr $cert

	# Verify certificate
	## Encode certificate to hash later
	set message [binary format H* $cert_arr(cert)]

	## Find CA to verify against
	if {![info exists trustedcertinfo($cert_arr(issuer))]} {
		## XXX: Try to find an intermediate path

		## XXX: Verify each cert in the intermediate path, returning in
		##      failure if a link in the chain breaks

		## Otherwise, return in failure
		return false
	}

	set cacert $trustedcertinfo($cert_arr(issuer))
	array set cacert_arr $cacert

	## Set signature to binary form
	set signature [::pki::_dec_to_ascii 0x$cert_arr(signature) $cacert_arr(l)]

	## Verify
	set ret [::pki::verify $signature $message $cacert]

	return $ret
}

# Generate a PKCS#10 Certificate Signing Request
proc ::pki::pkcs::create_csr {keylist namelist {encodePem 0} {algo "sha1"}} {
	array set key $keylist

	set name [::pki::x509::_list_to_dn $namelist]

	set type $key(type)

	switch -- $type {
		"rsa" {
			set pubkey [::asn::asnSequence \
				[::asn::asnBigInteger [::math::bignum::fromstr $key(n)]] \
				[::asn::asnBigInteger [::math::bignum::fromstr $key(e)]] \
			]
			set pubkey_algo_params [::asn::asnNull]
		}
	}
	binary scan $pubkey B* pubkey_bitstring

	set cert_req_info [::asn::asnSequence \
		[::asn::asnInteger 0] \
		[::asn::asnSequence $name] \
		[::asn::asnSequence \
			[::asn::asnSequence \
				[::asn::asnObjectIdentifier [::pki::_oid_name_to_number ${type}Encryption]] \
				$pubkey_algo_params \
			] \
			[::asn::asnBitString $pubkey_bitstring] \
		] \
		[::asn::asnContextConstr 0 ""] \
	]

	set signature [::pki::sign $cert_req_info $keylist $algo]
	binary scan $signature B* signature_bitstring
	
	set cert_req [::asn::asnSequence \
		$cert_req_info \
		[::asn::asnSequence [::asn::asnObjectIdentifier [::pki::_oid_name_to_number "${algo}With${type}Encryption"]] [::asn::asnNull]] \
		[::asn::asnBitString $signature_bitstring] \
	]

	if {$encodePem} {
		set cert_req [::pki::_encode_pem $cert_req "-----BEGIN CERTIFICATE REQUEST-----" "-----END CERTIFICATE REQUEST-----"]
	}

	return $cert_req
}

# Parse a PKCS#10 CSR
proc ::pki::pkcs::parse_csr {csr} {
	array set ret [list]

	array set parsed_csr [::pki::_parse_pem $csr "-----BEGIN CERTIFICATE REQUEST-----" "-----END CERTIFICATE REQUEST-----"]
	set csr $parsed_csr(data)

	::asn::asnGetSequence csr cert_req_seq
		::asn::asnGetSequence cert_req_seq cert_req_info

	set cert_req_info_saved [::asn::asnSequence $cert_req_info]

			::asn::asnGetInteger cert_req_info version
			::asn::asnGetSequence cert_req_info name
			::asn::asnGetSequence cert_req_info pubkeyinfo
				::asn::asnGetSequence pubkeyinfo pubkey_algoid
					::asn::asnGetObjectIdentifier pubkey_algoid pubkey_type
					::asn::asnGetBitString pubkeyinfo pubkey
		::asn::asnGetSequence cert_req_seq signature_algo_seq
			::asn::asnGetObjectIdentifier signature_algo_seq signature_algo
		::asn::asnGetBitString cert_req_seq signature_bitstring

	# Convert parsed fields to native types
	set signature [binary format B* $signature_bitstring]
	set ret(subject) [::pki::x509::_dn_to_string $name]

	## Convert Pubkey type to string
	set pubkey_type [::pki::_oid_number_to_name $pubkey_type]

	# Parse public key, based on type
	switch -- $pubkey_type {
		"rsaEncryption" {
			set pubkey [binary format B* $pubkey]

			::asn::asnGetSequence pubkey pubkey_parts
			::asn::asnGetBigInteger pubkey_parts key(n)
			::asn::asnGetBigInteger pubkey_parts key(e)

			set key(n) [::math::bignum::tostr $key(n)]
			set key(e) [::math::bignum::tostr $key(e)]
			set key(l) [expr {2**int(ceil(log([::pki::_bits $key(n)])/log(2)))}]
			set key(type) rsa
		}
		default {
			return -code error "Unsupported key type: $pubkey_type"
		}
	}

	# Convert key to RSA parts
	set keylist [array get key]

	# Validate CSR requestor has access to the private key
	set csrValid [::pki::verify $signature $cert_req_info_saved $keylist]
	if {!$csrValid} {
		return -code error "CSR Signature check failed"
	}

	array set ret $keylist

	return [array get ret]
}

proc ::pki::x509::create_cert {signreqlist cakeylist serial_number notBefore notAfter isCA extensions {encodePem 0} {algo "sha1"}} {
	# Parse parameters
	array set cakey $cakeylist
	array set signreq $signreqlist

	set type $signreq(type)

	# Process extensions
	set extensions_list $extensions
	unset extensions
	array set extensions $extensions_list

	# If we are generating a CA cert, add a CA extension
	if {$isCA} {
		set extensions(id-ce-basicConstraints) [list true true -1]
	}

	# Determine what version we need to use (default to 1)
	if {[array get extensions] == ""} {
		set version 1
	} else {
		set version 3
	}

	set certlist [list]

	# Create certificate to be signed
	## Insert version number (if not version 1)
	if {$version != 1} {
		lappend certlist [::asn::asnContextConstr 0 [::asn::asnInteger [expr {$version - 1}]]]
	}

	## Insert serial number
	lappend certlist [::asn::asnBigInteger [math::bignum::fromstr $serial_number]]

	## Insert data algorithm
	lappend certlist [::asn::asnSequence \
		[::asn::asnObjectIdentifier [::pki::_oid_name_to_number "${algo}With${type}Encryption"]] \
		[::asn::asnNull] \
	]

	## Insert issuer
	lappend certlist [::asn::asnSequence [::pki::x509::_string_to_dn $cakey(subject)]]

	## Insert validity requirements
	lappend certlist [::asn::asnSequence \
		[::asn::asnUTCTime [::pki::x509::_native_to_utctime $notBefore]] \
		[::asn::asnUTCTime [::pki::x509::_native_to_utctime $notAfter]] \
	]

	## Insert subject
	lappend certlist [::asn::asnSequence [::pki::x509::_string_to_dn $signreq(subject)]]

	## Insert public key information
	switch -- $type {
		"rsa" {
			set pubkey [::asn::asnSequence \
				[::asn::asnBigInteger [::math::bignum::fromstr $signreq(n)]] \
				[::asn::asnBigInteger [::math::bignum::fromstr $signreq(e)]] \
			]

			set pubkey_algo_params [::asn::asnNull]
		}
	}
	binary scan $pubkey B* pubkey_bitstring

	lappend certlist [::asn::asnSequence \
		[::asn::asnSequence \
			[::asn::asnObjectIdentifier [::pki::_oid_name_to_number "${type}Encryption"]] \
			$pubkey_algo_params \
		] \
		[::asn::asnBitString $pubkey_bitstring] \
	]

	## Insert extensions
	if {[array get extensions] != ""} {
		set extensionslist [list]

		foreach {extension extvalue} [array get extensions] {
			set critical 0

			switch -- $extension {
				"id-ce-basicConstraints" {
					set critical [lindex $extvalue 0]
					set allowCA [lindex $extvalue 1]
					set caDepth [lindex $extvalue 2]

					if {$caDepth < 0} {
						set extvalue [::asn::asnSequence [::asn::asnBoolean $allowCA]]
					} else {
						set extvalue [::asn::asnSequence [::asn::asnBoolean $allowCA] [::asn::asnInteger $caDepth]]
					}
				}
				default {
					return -code error "Unknown extension: $extension"
				}
			}

			lappend extensionslist [::asn::asnSequence \
				[::asn::asnObjectIdentifier [::pki::_oid_name_to_number $extension]] \
				[::asn::asnBoolean $critical] \
				[::asn::asnOctetString $extvalue] \
			]
		}

		lappend certlist [::asn::asnContextConstr 3 [::asn::asnSequenceFromList $extensionslist]]
	}

	## Enclose certificate data in an ASN.1 sequence
	set cert [::asn::asnSequenceFromList $certlist]

	# Sign certificate request using CA
	set signature [::pki::sign $cert $cakeylist $algo]
	binary scan $signature B* signature_bitstring

	set cert [::asn::asnSequence \
		$cert \
		[::asn::asnSequence \
			[::asn::asnObjectIdentifier [::pki::_oid_name_to_number "${algo}With${type}Encryption"]] \
			[::asn::asnNull] \
		] \
		[::asn::asnBitString $signature_bitstring] \
	]

	if {$encodePem} {
		set cert [::pki::_encode_pem $cert "-----BEGIN CERTIFICATE-----" "-----END CERTIFICATE-----"]
	}

	return $cert
}

proc ::pki::_bits {num} {
	if {$num == 0} {
		return 0
	}

	set num [format %llx $num]

	set numlen [string length $num]

	set numprecise 2

	if {$numlen > $numprecise} {
		set basebits [expr {($numlen - $numprecise) * 4}]
	} else {
		set basebits 0
	}

	set highbits 0x[string range $num 0 [expr {$numprecise - 1}]]

	set ret [expr {$basebits + log($highbits) / 0.69314718055994530941723}]

	set ret [expr {floor($ret) + 1}]

	set ret [lindex [split $ret .] 0]

	return $ret
}

proc ::pki::_random args {
	if {[lindex $args 0] == "-binary"} {
		set outputmode binary
	} else {
		set outputmode numeric
	}

	if {![info exists ::pki::_random_dev]} {
		foreach trydev [list /dev/urandom /dev/random __RAND__] {
			if {$trydev != "__RAND__"} {
				if {[catch {
					set fd [open $trydev [list RDONLY BINARY]]
					close $fd
					unset fd
				}]} {
					continue
				}
			}

			set ::pki::_random_dev $trydev

			break
		}
	}

	set dev ${::pki::_random_dev}

	switch -- $dev {
		"__RAND__" {
			set ret [expr {int(rand() * 2147483647)}]
		}
		default {
			set fd [open $dev [list RDONLY BINARY]]
			set data [read $fd 8]
			close $fd

			binary scan $data H* ret
			set ret [expr 0x$ret]
		}
	}

	switch -- $outputmode {
		"numeric" {
			# Do nothing, results are already numeric
		}
		"binary" {
			set ret [binary format H* [format %02llx $ret]]
		}
	}

	return $ret
}

proc ::pki::_isprime {n} {
	set k 10

	if {$n <= 3} {
		return true
	}

	if {$n % 2 == 0} {
		return false
	}
	
	# write n - 1 as 2^sd with d odd by factoring powers of 2 from n \u2212 1
	set d [expr {$n - 1}]
	set s 0
	while {$d % 2 == 0} {
		set d [expr {$d / 2}]
		incr s
	}
	
	while {$k > 0} {
		incr k -1
		set rand_1 [expr {int(rand() * $::pki::INT_MAX)}]
		set rand_2 [expr {int(rand() * $::pki::INT_MAX)}]
		if {$rand_1 < $rand_2} {
			set rand_num $rand_1
			set rand_den $rand_2
		} else {
			set rand_num $rand_2
			set rand_den $rand_1
		}

		set a [expr {2 + (($n - 4) * $rand_num / $rand_den)}]

		set x [_powm $a $d $n]
		if {$x == 1 || $x == $n - 1} {
			continue
		}

		for {set r 1} {$r < $s} {incr r} {
			set x [_powm $x 2 $n]
			if {$x == 1} {
				return false
			}
			if {$x == $n - 1} {
				break
			}
		}

		if {$x != $n - 1} {
			return false
		}
	}

	return true
}

proc ::pki::rsa::_generate_private {p q e bitlength} {
	set totient [expr {($p - 1) * ($q - 1)}]

	for {set di 1} {$di < $e} {incr di} {
		set dchk [expr {($totient * $di + 1) / $e}]
		set chkval [expr {$dchk * $e - 1}]

		set rem [expr {$chkval % $totient}]
		if {$rem == 0} {
			break
		}
	}

	# puts "bd=[_bits $dchk], di = $di"
	for {} {1} {incr di $e} {
		set dchk [expr {($totient * $di + 1) / $e}]
		set chkval [expr {$dchk * $e - 1}]

		set rem [expr {$chkval % $totient}]
		if {$rem == 0} {
			if {[::pki::_bits $dchk] > $bitlength} {
				if {![info exists d]} {
					set d $dchk
				}

				break
			}

			set d $dchk
		}

	}

	return $d
}

proc ::pki::rsa::generate {bitlength {exponent 0x10001}} {
	set e $exponent

	# Step 1. Pick 2 numbers that when multiplied together will give a number with the appropriate length
	set componentbitlen [expr {$bitlength / 2}]
	set bitmask [expr {(1 << $componentbitlen) - 1}]

	set p 0
	set q 0
	while 1 {
		set plen [::pki::_bits $p]
		set qlen [::pki::_bits $q]

		if {$plen >= $componentbitlen} {
			set p [expr {$p & $bitmask}]

			set plen [::pki::_bits $p]
		}

		if {$qlen >= $componentbitlen} {
			set q [expr {$q & $bitmask}]

			set qlen [::pki::_bits $q]
		}

		if {$plen >= $componentbitlen && $qlen >= $componentbitlen} {
			break
		}

		set x [::pki::_random]
		set y [::pki::_random]

		set xlen [expr {[::pki::_bits $x] / 2}]
		set ylen [expr {[::pki::_bits $y] / 2}]

		set xmask [expr {(1 << $xlen) - 1}]
		set ymask [expr {(1 << $ylen) - 1}]

		set p [expr {($p << $xlen) + ($x & $xmask)}]
		set q [expr {($q << $ylen) + ($y & $ymask)}]
	}


	# Step 2. Verify that "p" and "q" are useful
	## Step 2.a. Verify that they are not too close
	### Where "too close" is defined as 2*n^(1/4)
	set quadroot_of_n [expr {isqrt(isqrt($p * $q))}]
	set min_distance [expr {2 * $quadroot_of_n}]
	set distance [expr {abs($p - $q)}]

	if {$distance < $min_distance} {
		#### Try again.

		return [::pki::rsa::generate $bitlength $exponent]
	}

	# Step 3. Convert the numbers into prime numbers
	if {$p % 2 == 0} {
		incr p -1
	}
	while {![::pki::_isprime $p]} {
		incr p -2
	}

	if {$q % 2 == 0} {
		incr q -1
	}
	while {![::pki::_isprime $q]} {
		incr q -2
	}

	# Step 4. Compute N by multiplying P and Q
	set n [expr {$p * $q}]
	set retkey(n) $n

	# Step 5. Compute D ...
	## Step 5.a. Generate D
	set d [::pki::rsa::_generate_private $p $q $e $bitlength]
	set retkey(d) $d

	## Step 5.b. Verify D is large enough
	### Verify that D is greater than (1/3)*n^(1/4) 
	set quadroot_of_n [expr {isqrt(isqrt($n))}]
	set min_d [expr {$quadroot_of_n / 3}]
	if {$d < $min_d} {
		#### Try again.

		return [::pki::rsa::generate $bitlength $exponent]
	}

	# Step 6. Encode key information
	set retkey(type) rsa
	set retkey(e) $e
	set retkey(l) $bitlength

	# Step 7. Record additional information that will be needed to write out a PKCS#1 compliant key
	set retkey(p) $p
	set retkey(q) $q

	return [array get retkey]
}

## Initialize parsing routines, which may load additional packages (base64)
::pki::_parse_init

# # ## ### ##### ######## #############
## Ready

package provide pki 0.6
